#include <stdio.h>
#include <window.h>
#include <comm.h>
#include <setjmp.h>

#define	TICK	peekw(0x40,0x6C)
#define	EVENTS	128
#define	EDITN	0x17		// Editor normal display
#define	EDITH	0x71		// Editor hilited display
#define	EDITE	0x07		// Editor editing display
#define	EDITD	0x67		// Edit display field
#define	STATUS	0x70		// Status line
#define	MFIELD	4			// Maximum field number
#define	RETRY	3			// # comm retries

struct EVENT {
	unsigned char	Mode;
	unsigned char	Days;
	unsigned char 	Hour;
	unsigned char	Minite;
	unsigned		Unit;
	unsigned char	House;
	unsigned char	Function;
	} Events[EVENTS], *E;

struct WINDOW
	*mwin,					// Main window
	*swin;					// Status window

jmp_buf
	gojmp;

FILE
	*fp;

unsigned
	Com = 1,				// COM port
	Timeout = 18,			// Serial timeout value
	Sync = 16,				// Sync bytes
	Ctop,					// Top of command line
	Etop;					// Top of event list

unsigned char
	*ptr,					// General pointer
	*Cmds[25],				// Command input
	Buffer[16],				// Temp buffer
	Base,					// Base housecode
	Cactive,				// COM port is active
	Wait = 1,				// Command wait mode
	Status = 1,				// Status update pending
	Sy,						// Select Y
	Sp,						// Stack pointer
	ShowName = 255,			// Display names
	SaveName,				// Save names
	Debug,					// Debug output
	Sx[MFIELD+1],			// Select X
	Graphic[64][8],			// Graphic data
	Stack[256][8];			// Cut/Paste stack

unsigned char Housecodes[] = { "MECKOGAINFDLPHBJ" };

unsigned char *Modes[] = {
	"Not SET",		// 0
	"?1?",			// 1
	"TOMORROW",		// 2
	"?3?",			// 3
	"TODAY",		// 4
	"?5?",			// 5
	"?6?",			// 6
	"?7?",			// 7
	"NORMAL",		// 8
	"SECURITY",		// 9
	"?A?",			// A
	"?B?",			// B
	"?C?",			// C
	"?D?",			// D
	"?E?",			// E
	"?F?" };		// F

unsigned Umask[] = {
	0x0080, 0x0040, 0x0020, 0x0010, 0x0008, 0x0004, 0x0002, 0x0001,
	0x8000, 0x4000, 0x2000, 0x1000, 0x0800, 0x0400, 0x0200, 0x0100 };

unsigned char *Hcodes[] = {
	"A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", 0 };

unsigned char *Units[] = {
	"1", "2", "3", "4", "5", "6", "7", "8", "9",
	"10", "11", "12", "13", "14", "15", "16", 0 };

unsigned char *Dow[] =
		{ "MO", "TU", "WE", "TH", "FR", "SA", "SU", "??" };

#include "R:\\Help.h"

/* ---- Communications ---- */

/*
 * Report an error & exit
 */
register error(unsigned args)
{
	unsigned char buf[128];
	_format_(nargs() * 2 + &args, buf);
	fputs(buf, stderr);
	putc('\n', stderr);
	if(Cactive)
		Cclose();
	exit(-1);
}

/*
 * Display diagnostic message
 */
register Xprint(unsigned args)
{
	unsigned n;
	unsigned char buf[128];
	n = nargs();
	if(Debug) {
		_format_(n * 2 + &args, buf);
		fputs(buf, stdout); }
}

/*
 * Get character with timeout
 */
int Cgett(void)
{
	int c;
	unsigned t;
	t = TICK;
	do {
		if((c = Ctestc()) != -1)
			return c; }
	while((TICK - t) <= Timeout);
	longjmp(gojmp, "Timeout");
}

/*
 * Being command output
 */
void command(unsigned c)
{
	unsigned i;

	if(!Cactive) {
		if(Copen(Com, _600, PAR_NO|DATA_8|STOP_1, SET_RTS|SET_DTR|OUTPUT_2))
			error("Cannot open COM port");
		Cflags |= TRANSPARENT;
		Cactive = 255; }

	for(i=0; i < Sync; ++i)
		Cputc(0xFF);
	while(Ctestc() != -1);
	Cputc(c);
}

/*
 * Begin reception of status from interface
 */
int receive(void)
{
	int c;
	unsigned i;
	for(;;) {
		i = 0;
		while((c = Cgett()) == 0xFF)
			++i;
		if(i > 2)
			return c; }
}

/* ---- Editor ---- */

/*
 * Display status message
 */
register status(unsigned args)
{
	unsigned char buf[128];
	_format_(nargs() * 2 + &args, buf);
	w_clwin(swin);
	w_puts(buf, swin);
	Status = 1;
}

/* 
 * Display single-field
 */
void show_field(unsigned f)
{
	unsigned i, j;
	unsigned char h, a;
	static unsigned char M, Days[] = { "mtwtfss" };

	if(!f) {
		wprintf("%-8s", Modes[M = E->Mode & 15]);
		return; }

	if(!M)
		return 1;
	switch(f) {
	case 1 :			// Function
		switch(E->Function & 15) {
		default: wprintf("?%02x? ", E->Function);	break;
		case 0x02: wprintf("ON   ");				break;
		case 0x03: wprintf("OFF  ");				break;
		case 0x05: wprintf("DIM%-2u", E->Function >> 4); }
		return;

	case 2 :			// Time
		a = 'A';
		if((h = E->Hour) >= 12) {
			a = 'P';
			h -= 12; }
		if(!h)
			h = 12;
		wprintf("%2u:%02u%c", h, E->Minite, a);
		return;

	case 3 :				// Days
		for(i=a=0; i < 7; ++i) {
			h = Days[i];
			if(E->Days & (1<<i))
				wputc(h - 0x20);
			else
				wputc(h); }
		return;

	case 4 :				// Units
		wputc(Housecodes[E->House>>4]);
		wcleol();
		for(i=0; i < 16; ++i) {
			if(E->Unit & Umask[i]) {
				wputc(' ');
				if(ShowName) {
					a = (E->House & 0xF0) | i;
					for(j=0; j < 64; ++j) {
						if((*Graphic[j] == a) && Graphic[j][1])
							break; }
					if(j < 64) {
						for(h=1; h < 8; ++h) {
							if(!(a = Graphic[j][h]))
								break;
							wputc(a); }
						continue; } }
				wprintf("%u", i+1); } }
	}
}

// Adjust mode
void adjust_mode(void)
{
	unsigned s, n, d;

	s = n = E->Mode & 15;
	status("MODE: \x18\x19=select  ENTER=save  ESC=cancel");
	*W_OPEN = EDITE;
	for(;;) {
		wgotoxy(Sx[0], Sy);
		show_field(0);
xx:		switch(wgetc()) {
		default: goto xx;
		case _KUA: d = -1;	break;
		case _KDA: d = 1;	break;
		case 0x1B:
			E->Mode = s;
		case '\n' :
			*W_OPEN = EDITN;
			return; }
		do {
			n = (n + d) & 15; }
		while(*Modes[n] == '?');
		E->Mode = n; }
}

// Adjust function
void adjust_function(void)
{
	unsigned s, n;
	static unsigned char Funcs[] = { 0x02, 0x03, 0x05, 0x15, 0x25, 0x35,
	0x45, 0x55, 0x65, 0x75, 0x85, 0x95, 0xA5, 0xB5, 0xC5, 0xD5, 0xE5, 0xF5 };

	s = E->Function;
	for(n=0; n < sizeof(Funcs); ++n) {
		if(Funcs[n] == s)
			break; }
	status("FUNCTION: \x18\x19=select  ENTER=save  ESC=cancel");
	*W_OPEN = EDITE;
	for(;;) {
		wgotoxy(Sx[1], Sy);
		show_field(1);
xx:		switch(wgetc()) {
		default: goto xx;
		case _KDA: n = (n ? n : sizeof(Funcs)) - 1;	break;
		case _KUA: if(++n >= sizeof(Funcs)) n = 0;	break;
		case 0x1B:
			E->Function = s;
		case '\n' :
			*W_OPEN = EDITN;
			return; }
		E->Function = Funcs[n]; }
}

// Adjust time
void adjust_time(void)
{
	unsigned h, sh, m, sm;
	status("TIME: \x18\x19=hour  \x1B\x1A=minite  ENTER=save  ESC=cancel");
	h = sh = E->Hour;
	m = sm = E->Minite;
	*W_OPEN = EDITE;
	for(;;) {
		wgotoxy(Sx[2], Sy);
		show_field(2);
xx:		switch(wgetc()) {
		default: goto xx;
		case _KDA: h = (h ? h : 24) - 1;	break;
		case _KUA: if(++h >= 24) h = 0;		break;
		case _KLA: m = (m ? m : 60) - 1;	break;
		case _KRA: if(++m >= 60) m = 0;		break;
		case 0x1B:
			E->Hour = sh;
			E->Minite = sm;
		case '\n' :
			*W_OPEN = EDITN;
			return; }
		E->Hour = h;
		E->Minite = m; }
}

void adjust_days(void)
{
	unsigned char s, b;
	status("DAYS: Mon Tue Wed tHr Fri Sat sUn   SPACE=all  ENTER=save  ESC=cancel");
	*W_OPEN = EDITE;
	s = E->Days;
	for(;;) {
		wgotoxy(Sx[3], Sy);
		show_field(3);
xx:		switch(toupper(wgetc())) {
		default: goto xx;
		case 'M' : b = 0x01;	break;
		case 'T' : b = 0x02;	break;
		case 'W' : b = 0x04;	break;
		case 'H' : b = 0x08;	break;
		case 'F' : b = 0x10;	break;
		case 'S' : b = 0x20;	break;
		case 'U' : b = 0x40;	break;
		case ' ' :
			E->Days = E->Days ? 0 : 0x7F;
			continue;
		case 0x1B:
			E->Days = s;
		case '\n':
			*W_OPEN = EDITN;
			return; }
		E->Days ^= b; }
}

void adjust_units(void)
{
	unsigned i, j, c, su;
	unsigned char f, sh, sn;
	static unsigned Us;

	sn = ShowName;
	su = E->Unit;
	sh = E->House;
	ShowName = 0;

	status("UNIT: F1=house  F2=unit  SPACE=all  ENTER=save  ESC=cancel");

x0:	*W_OPEN = EDITD;
	f = E->House & 0xF0;
	for(i=0; i < 16; ++i) {
		wgotoxy(4, i+3);
		wprintf("%c%-2u=%9s", Housecodes[f>>4], i+1, ""); }
	for(i=0; i < 64; ++i) {
		j = *Graphic[i];
		if(((j & 0xF0) == f) && Graphic[i][1]) {
			wgotoxy(9, (j & 0x0F)+3);
			for(j=1; j < 8; ++j) {
				if(!(c = Graphic[i][j]))
					break;
				wputc(c); } } }

	*W_OPEN = EDITE;
	for(;;) {
		wgotoxy(Sx[4], Sy);
		show_field(4);
		switch(toupper(wgetc())) {
		case ' ' :
			E->Unit = E->Unit ? 0 : 0xFFFF;
			continue;
		case _K1:
			i = Housecodes[E->House >> 4] - 'A';
			if(wmenu(30, 3, WSAVE|WCOPEN|WBOX1|EDITH, Hcodes, &i))
				continue;
			i += 'A';
			for(j=0; Housecodes[j] != i; ++j);
			E->House = j << 4;
			goto x0;
		case _K2:
			if(wmenu(30, 3, WSAVE|WCOPEN|WBOX1|EDITH, Units, &Us))
				continue;
			E->Unit ^= Umask[Us];
			continue;
		case 0x1B:
			E->Unit = su;
			E->House = sh;
		case '\n':
			ShowName = sn;
			*W_OPEN = EDITN;
			return; } }
}

sort_names()
{
	unsigned i, j, t;
	unsigned char v1, v2, *p1, *p2;
	for(i=t=0; i < 64; ++i) {
		if(Graphic[i][1])
			memcpy(Graphic[t++], Graphic[i], 8); }
	for(i=t; i < 64; ++i)
		memset(Graphic[i], 0, 8);

	for(i=0; i < t; ++i) {
		p1 = Graphic[i];
		for(j=i+1; j < t; ++j) {
			p2 = Graphic[j];
			v1 = ((Housecodes[*p1 >> 4]-'A')<<4) | (*p1 & 15);
			v2 = ((Housecodes[*p2 >> 4]-'A')<<4) | (*p2 & 15);
			if(v1 > v2) {
				memcpy(Buffer, p1, 8);
				memcpy(p1, p2, 8);
				memcpy(p2, Buffer, 8); } } }
}

void edit_names(void)
{
	unsigned i, j, x, y;
	unsigned char c, save[512];
	static unsigned Select;
	static unsigned char Stack[256][8], Sp;

	sort_names();
	wclwin();
	status("NAMES: F1=name  F2=house  F3=unit  DEL/INS=cut/paste  ENTER=save  ESC=cancel");
	memcpy(save, Graphic, sizeof(save));
redraw:
	Select &= 0x3F;
	wgotoxy(35, 23); wprintf("%3u of 64", Select+1);
	for(i=0; i < 64; ++i) {
		wgotoxy(x=((i>>4)*16)+8, y=(i&15)+3);
		if(i == Select)
			*W_OPEN = EDITH;
		if(!Graphic[i][1])
			wputf("", 13);
		else {
			c = *Graphic[i];
			wprintf("%c%-2u=%-9s", Housecodes[c>>4], (c&15)+1, "");
			wgotoxy(x+5, y);
			for(j=1; j < 8; ++j) {
				if(!(c = Graphic[i][j]))
					break;
				wputc(c); } }
		*W_OPEN = EDITN; }

	c = *Graphic[Select];
	for(;;) switch(i=wgetc()) {
	case _KUA: --Select;					goto redraw;
	case _KDA: ++Select;					goto redraw;
	case _KRA: Select += 16;				goto redraw;
	case _KLA: Select -= 16;				goto redraw;
	case _KHO: Select = (Select-1) & 0x30;	goto redraw;
	case _KEN: Select = (Select+1) | 0x0F;	goto redraw;
	case _KDL:
		memcpy(Stack[Sp++], Graphic[i=Select], 8);
		while(++i < 63)
			memcpy(Graphic[i-1], Graphic[i], 8);
		memset(Graphic[63], 0, 8);
		goto redraw;
	case _KIN:
		i = 64;
		while(--i > Select)
			memcpy(Graphic[i], Graphic[i-1], 8);
		memcpy(Graphic[Select], Stack[--Sp], 8);
		memset(Stack[Sp], 0, 8);
		goto redraw;
	case _K1:
		memcpy(Buffer, Graphic[Select]+1, 7);
		Buffer[7] = 0;
		for(;;) switch(wgets(((Select>>4)*16)+(8+5), (Select&15)+3, Buffer, 7)) {
		case '\n' :
			memcpy(Graphic[Select]+1, Buffer, 7);
		case 0x1B:
			wcursor_off();
			goto redraw; }
	case _K2:
		i = Housecodes[c >> 4] - 'A';
		if(wmenu(30, 3, WSAVE|WCOPEN|WBOX1|EDITH, Hcodes, &i))
			continue;
		i += 'A';
		for(j=0; Housecodes[j] != i; ++j);
		*Graphic[Select] = (j << 4) | (c & 0x0F);
		goto redraw;
	case _K3:
		i = c & 0x0F;
		if(wmenu(30, 3, WSAVE|WCOPEN|WBOX1|EDITH, Units, &i))
			continue;
		*Graphic[Select] = (c & 0xF0) | i;
		goto redraw;
	case 0x1B:
		memcpy(Graphic, save, sizeof(save));
	case '\n':
		return; }
}
void count_events()
{
	Etop = EVENTS;
	while(--Etop) {
		if(Events[Etop-1].Mode)
			break; }
	++Etop;
}

int edit_events(void)
{
	unsigned i, j;
	static unsigned Stop, Select, Field;
	swin = wopen(0,24, 80,  1, WSAVE|WCOPEN|STATUS);
	mwin = wopen(0, 0, 80, 24, WSAVE|WCOPEN|EDITN);
	wcursor_off();

	if((Ctop == 2) && !fp) {
		status("New file: %s", Cmds[1]);
		Status = 2; }

recount:
	count_events();
redraw:
	if(Select & 0x8000)
		Select = 0;
	if(Select >= Etop)
		Select = Etop-1;
	if(Select < Stop)
		Stop = Select;
	if(Select >= (Stop+23))
		Stop = (Select > 23) ? (Select - 23) : 0;
	for(i=0; i < 24; ++i) {
		wgotoxy(0, i);
		if((j = Stop + i) < Etop) {
			E = Events[Stop+i];
			wprintf("%-3u", Stop+i);
			if(!(E->Mode & 15))
				wcleol();
			if(j == Select)
				Sy = i;
			else {
				for(j=0; j <= MFIELD; ++j) {
					wputc(' ');
					show_field(j); } } }
		wcleol(); }
reline:
	if(Field & 0x8000)
		Field = MFIELD;
	if(Field > MFIELD)
		Field = 0;
	wgotoxy(0, Sy);
	wprintf("%-3u", Select);
	E = Events[Select];
	if(!(E->Mode & 15)) {
		Field = 0;
		wcleol(); }
	for(j=0; j <= MFIELD; ++j) {
		wputc(' ');
		Sx[j] = W_OPEN->WINcurx;
		if(j == Field)
			*W_OPEN = EDITH;
		show_field(j);
		*W_OPEN = EDITN; }
restat:
	if(Status) {
		if(!--Status)
			status("F1:%c  F2=name  F3=setname  DEL/INS=cut/paste  ENTER=modify  F10=save  ESC=cancel",
				Housecodes[Base >> 4]); }
	for(;;) switch(wgetc()) {
	case _KLA:	--Field;						goto reline;
	case _KRA:	++Field;						goto reline;
	case _KHO: 	Field = 0;						goto reline;
	case _KEN:	Field = MFIELD;					goto reline;
	case _KUA:	--Select;						goto redraw;
	case _KDA:	++Select;						goto redraw;
	case _KPU:	Select -= 23;					goto redraw;
	case _KPD:	Select += 23;					goto redraw;
	case _CHO:	Select = 0;						goto redraw;
	case _CEN:	Select = Etop-1;				goto redraw;
	case _K1:
		i = Housecodes[Base >> 4] - 'A';
		if(wmenu(30, 3, WSAVE|WCOPEN|WBOX1|EDITH, Hcodes, &i))
			continue;
		i += 'A';
		for(j=0; Housecodes[j] != i; ++j);
		Base = j << 4;
		Status = 1;
		goto restat;
	case _K2:	ShowName = ShowName ? 0 : 255;	goto redraw;
	case _K3:	edit_names();					goto redraw;
	case _KDL:
		memcpy(Stack[Sp++], Events[i=Select], sizeof(struct EVENT));
		while(++i < (EVENTS-1))
			memcpy(Events[i-1], Events[i], sizeof(struct EVENT));
		memset(Events[EVENTS-1], 0, sizeof(struct EVENT));
		goto recount;
	case _KIN:
		if(Etop >= EVENTS)
			continue;
		i = EVENTS;
		while(--i > Select)
			memcpy(Events[i], Events[i-1], sizeof(struct EVENT));
		memcpy(Events[Select], Stack[--Sp], sizeof(struct EVENT));
		memset(Stack[Sp], 0, sizeof(struct EVENT));
		goto recount;
	case '\n' :
		switch(Field) {
		case 0 : adjust_mode();					goto recount;
		case 1 : adjust_function();				goto reline;
		case 2 : adjust_time();					goto reline;
		case 3 : adjust_days();					goto reline;
		case 4 : adjust_units();				goto redraw;
		} continue;
	
	case 0x1B:
		wclose();
		wclose();
		return 0;
	case _K10:
		wclose();
		wclose();
		return 255;
	}
}

/* ---- Main program ---- */

/*
 * Get a value from the command line
 */
unsigned value(unsigned char t, unsigned l, unsigned h, unsigned char *name)
{
	unsigned c, v;
	unsigned char f;
	v = f = 0;
nd:	c = *ptr;
	if((c >= '0') && (c <= '9')) {
		v = (v * 10) + (c-'0');
		f = 255;
		++ptr;
		goto nd; }
	if((c == t) || !c) {
		if((v < l) || (v > h))
			error("%s value must be %u-%u", name, l, h);
		return v; }
	error("Bad %s value input", name);
}
		
/*
 * Get House/Module(s) from command argument 1
 */
unsigned get_unit(void)
{
	unsigned char c;
	unsigned u;

	ptr = Cmds[1];
	c = *ptr++;
	if((c < 'A') || (c > 'M'))
		error("Module must be A-M\n");
	for(u=0; Housecodes[u] != c; ++u);
	Base = u << 4;

	if(!strcmp(ptr, "$ALL"))
		return 0xFFFF;

	u = 0;
nu:	u |= Umask[value(',', 1, 16, "Module")-1];
	if(*ptr == ',') {
		++ptr;
		goto nu; }

	return u;
}
	
/*
 * Load events from CP290
 */
void load_events(void)
{
	int c;
	unsigned i, j, r;
	unsigned char *p, s, ck;

	Xprint("Loading events: ");
	r = RETRY;
	if(p = setjmp(gojmp)) {
		if(!--r)
			error(p);
		Xprint("%s-Retry ", p); }

	command(5);
	s = receive();
	for(i=ck=0; i < EVENTS; ++i) {
		p = Events[i];
		if((c = Cgett()) == 0xFF) {
			memset(p, 0, sizeof(struct EVENT));
			continue; }
		ck += (*p++ = c);
		for(j=0; j < 7; ++j)
			ck += (*p++ = Cgett()); }
	i = Cgett();
	if(i != ck)
		longjmp(gojmp, "Bad Checksum");
	Xprint("Done\n");
}

/*
 * Load graphic data from CP290
 */
void load_graphic(void)
{
	int c;
	unsigned i, r;
	unsigned char *p, s, ck;

	Xprint("Loading graphic data: ");
	r = RETRY;
	if(p = setjmp(gojmp)) {
		if(!--r)
			error(p);
		Xprint("%s-Retry ", p); }

	command(6);
	s = receive();
	p = Graphic;
	for(i=ck=0; i < 256; ++i) {
		if((c = Cgett()) == 0xFF) {
			*p++ = *p++ = 0;
			continue; }
		ck += (*p++ = c);
		ck += (*p++ = Cgett()); }
	if(Cgett() != ck)
		longjmp(gojmp, "Bad checksum");
	Xprint("Done\n");
}

/*
 * Perform a transaction with the CP290
 */
void transaction(unsigned char cmd, unsigned length, unsigned char *prompt)
{
	unsigned i, r, l, cs;
	unsigned char *p, ck, s;

	l = length & 0xFFF;
	Xprint(prompt);
	r = RETRY;
	if(p=setjmp(gojmp)) {
		if(!--r)
			error(p);
		Xprint("%s-Retry ", p); }

	command(cmd);
	cs = ck = 0;
	if(length & 0x4000)
		cs = 2;
	for(i=0; i < l; ++i) {
		if(cs)
			--cs;
		else
			ck += Buffer[i];
		Cputc(Buffer[i]); }
	if(!(length & 0x8000))
		Cputc(ck);

	s = receive();
	if(s > 1)
		longjmp(gojmp, "Bad ACK");
	Xprint("Done\n");
}

/*
 * Wait for an event to complete
 */
int wait_event(void)
{
	unsigned i, j;
	unsigned char *p, s, ck;
	static unsigned r = RETRY;

	if(Wait) {
		Xprint("Getting status:");
		if(p = setjmp(gojmp)) {
			if(!--r)
				error(p);
			return 255; }

		Timeout = 250;
		s = receive();
		Timeout = 18;
		for(i=ck=0; i < 4; ++i)
			ck += (Buffer[i] = Cgett());
		if(Cgett() != ck)
			longjmp(gojmp, "Bad checksum");
		if(Wait > 1) {
			printf("Base:%c %x %c", Housecodes[Buffer[3]>>4], Buffer[0] & 15, Housecodes[Buffer[0] >> 4]);
			j = (Buffer[1] << 8) | Buffer[2];
			for(i=0; i < 16; ++i) {
				if(Umask[i] & j)
					printf(" %u", i+1); }
			putc('\n', stdout); } }

	Xprint("Done\n");
	return 0;
}

/*
 * Get Base housecode & Clock
 */
get_info(char pf)
{
	unsigned i, r;
	unsigned char *p, s, ck;

	Xprint("Getting info: ");
	r = RETRY;
	if(p = setjmp(gojmp)) {
		if(!--r)
			error(p);
		Xprint("%s-Retry ", p); }

	command(4);		// Request clock & housecode
	s = receive();
	for(i = ck = 0; i < 4; ++i)
		ck += (Buffer[i] = Cgett());
	if(Cgett() != ck)
		longjmp(gojmp, "Bad checksum");
	Xprint("Done\n");

	Base = Buffer[3];
	if(pf) {
		s = 'A';
		i = Buffer[1];
		if(i >= 12) {
			s = 'P';
			i -= 12; }
		if(!i)
			i = 12;
		for(r=0; r < 7; ++r) {
			if((1<<r) & Buffer[2])
				break; }
		printf("%s%3u:%02u%c %c\n", Dow[r], i, Buffer[0], s, Housecodes[Base >> 4]); }
}

/*
 * Set CP290 clock
 */
void set_clock(void)
{
	unsigned i, d, h, m;

	d = get_date(&i, &i, &i);
	get_time(&h, &m, &i);
	d = d ? (d-1) : 6;
	switch(Ctop) {
	case 4 :
		for(d=0; d < 7; ++d) {
			if(!strcmp(Cmds[3], Dow[d]))
				goto dok; }
		error("Day must be: MO TU WE TH FR SA SU\n");
dok:
	case 3 : ptr = Cmds[2]; m = value(0, 0, 59, "Minite");
	case 2 : switch(*Cmds[1]) {
		case 'A' :
			ptr = Cmds[1]+1; h = value(0, 1, 12, "Hour");
			if(h == 12)
				h = 0;
			break;
		case 'P' :
			ptr = Cmds[1]+1; h = value(0, 1, 12, "Hour");
			if(h == 24)
				h = 12;
			break;
		default: ptr = Cmds[1]; h = value(0, 0, 23, "Hour"); } }

	Buffer[0] = m;
	Buffer[1] = h;
	Buffer[2] = 1<<d;
	transaction(2, 3, "Setting clock:");
}

/*
 * Set CP290 basecode
 */
void set_base(void)
{
	unsigned i;

	if(!Cmds[1][1]) {
		for(i=0; i < 16; ++i) {
			if(Housecodes[i] == *Cmds[1])
				goto bok; } }
	error("Base code must be A-M\n");
bok:
	Buffer[0] = i << 4;
	transaction(0, 0x8001, "Setting base:");
}

/*
 * Save events/graphic-data to interface
 */
void save_data(void)
{
	unsigned i, j, k, l, m;
	unsigned char *p;

	printf("Writing data to CP290. (ESC to abort)\n");

	Buffer[0] = Base;
	transaction(0, 0x8001, "Setting base:");
	delay(200);
	Sync = 11;

	printf("%-3u of %3u Events", 0, Etop-1);
	for(i=0; i < Etop; ++i) {
		if(Events[i].Mode) {
			if(kbtst() == 0x1B)
				goto quit;
			printf("\r%u", i+1);
			Buffer[0] = i << 3;
			Buffer[1] = i >> 5;
			memcpy(Buffer+2, Events[i], 8);
			transaction(3, 0x400A, "Uploading timer event:"); } }

	if(SaveName) {
		for(i=m=0; i < 64; ++i) {
			if(Graphic[i][1])
				++m; }
		printf("\n%-3u of %3u Names", 0, m);
		for(i=0; i < 64; ++i) {
			p = Graphic[i];
			if(p[1]) {
				printf("\r%u", i+1);
				l = i * 4;
				for(j=0; j < 4; ++j) {
					k = *p++ << 8;
					k |= *p++;
					if(k) {
						if(kbtst() == 0x1B)
							goto quit;
						Buffer[0] = l << 1;
						Buffer[1] = (l >> 7) | 4;
						Buffer[2] = k >> 8;
						Buffer[3] = k;
						transaction(3, 0x4004, "Uploading graphic data:");
						++l; } } } } }
quit:
	putc('\n', stdout);

	Sync = 16;
}

/*ChtTxt R:\Help.h
X10/Cp290 control		Dave Dunfield		https://dunfield.themindfactory.com

Use:	CP290 [options] [command]

opts:	C=1-4	set Comm port				[1]
		/D		display Diagnostic messages
		/N		save Names to CP290			[slow]
		/W		do not Wait for action to complete	[Default is to wait]
		/WS		Wait for action complete and Show	[but not display]

cmds:	INFO							Display unit time and base housecode
		BASE <A-P>						Set base housecode
		CLOCK [[[hour] [min] MO-SU]		Set clock	[use Ah Ph or 24hr]
		ON <module(s)>					Turn module ON
		OFF <module(s)>					Turn module OFF
		DIM <module(s)> <0-15>			Dim  module	[0=Bright..15=Dim]
		EDIT [file]						Edit  schedule
		GET <file>						Read  schedule from unit
		PUT <file>						Write schedule to unit

Modules are entered as HOUSECODE followed by module number, eg: A5
Multiple modules can be specified -or- $ALL, eg:  A1,2,3  -or-  A$ALL
*/
main(int argc, char *argv[])
{
	unsigned i, j;
	unsigned char c;

	static unsigned char *Commands[] = {
		"INFO",	"CLOCK",	"BASE",	"OFF",	"ON",	"DIM",	"EDIT",
		"GET", "PUT",
		0 };
	static unsigned char Minargs[] = {
		1,		1,			2,		2, 		2, 		3, 		1,
		2,		2 };

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch( (toupper(*ptr++) << 8) | toupper(*ptr++) ) {
		case 'C=' :	Com = value(0, 1, 4, "C=");	continue;
		case '-D' :
		case '/D' : Debug = 255;				continue;
		case '-N' :
		case '/N' : SaveName = 255;				continue;
		case '-W' :
		case '/W' : switch(*ptr) {
			case 's' :
			case 'S' :  Wait = 2;				continue;
			case 0 :	Wait = 0;				continue;
			} goto he; }
		strupr(Cmds[Ctop++] = ptr-2); }

	if(!Ctop) {
he:		ptr = Help;
		while(c = *ptr++) {
			if(c & 0x80) {
				while(c-- & 0x7F)
					putc(' ', stdout);
				continue; }
			putc(c, stdout); }
		return; }

	for(i=0; ptr = Commands[i]; ++i) {
		if(strbeg(ptr, Cmds[0]))
			goto cfound; }
	abort(Help);
cfound:
	for(j=i+1; ptr = Commands[j]; ++j) {
		if(strbeg(ptr, Cmds[0]))
			abort(Help); }
	if(Ctop < Minargs[i])
		abort(Help);
	switch(i) {
	case 0 :	get_info(255);		break;			// INFO
	case 1 :	set_clock();		break;			// CLOCK
	case 2 :	set_base();			break;			// BASE
	case 3 :	j = 0x03;			goto dof;		// OFF
	case 4 :	j = 0x02;			goto dof;		// ON
	case 5 :										// DIM
		ptr = Cmds[2];
		j = value(0, 0, 15, "DIM") | 5;
dof:	i = get_unit();
		do {
			Buffer[0] = j;
			Buffer[1] = Base;
			Buffer[2] = i >> 8;
			Buffer[3] = i;
			transaction(1, 4, "Sending command:"); }
		while(wait_event());
		break;
	case 6 :										// EDIT
		switch(Ctop) {
		default: abort(Help);
		case 1 :
			printf("Retrieving data from CP290.\n");
			get_info(0);
			load_events();
			load_graphic();
			break;
		case 2 :
			if(fp = fopen(Cmds[1], "rvb")) {
				i = fget(&Base, sizeof(Base), fp);
				i += fget(Events, sizeof(Events), fp);
				i += fget(Graphic, sizeof(Graphic), fp);
				i += fget(Buffer, sizeof(Buffer), fp);
				fclose(fp);
				if(i != (sizeof(Base)+sizeof(Events)+sizeof(Graphic)))
					abort("Corrupt image\n"); } }
		if(edit_events()) switch(Ctop) {
		case 1 : save_data();	break;
		case 2 :
			fp = fopen(Cmds[1], "wvqb");
			fput(&Base, sizeof(Base), fp);
			fput(Events, sizeof(Events), fp);
			fput(Graphic, sizeof(Graphic), fp);
			fclose(fp); }
		break;
	case 7 :										// GET
		fp = fopen(Cmds[1], "wvqb");
		printf("Retrieving data from CP290.\n");
		get_info(0);
		load_events();
		load_graphic();
		fput(&Base, sizeof(Base), fp);
		fput(Events, sizeof(Events), fp);
		fput(Graphic, sizeof(Graphic), fp);
		fclose(fp);
		break;
	case 8 :										// PUT
		fp = fopen(Cmds[1], "rvqb");
		i = fget(&Base, sizeof(Base), fp);
		i += fget(Events, sizeof(Events), fp);
		i += fget(Graphic, sizeof(Graphic), fp);
		i += fget(Buffer, sizeof(Buffer), fp);
		if(i != (sizeof(Base)+sizeof(Events)+sizeof(Graphic)))
			abort("Corrupt image\n");
		fclose(fp);
		count_events();
		save_data();
	}
	if(Cactive)
		Cclose();
}
