/*
 * TCL pre-processor
 *
 * ?COPY.TXT 2000-2006 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile with DDS Micro-C/PC:
 *	makedb example.tcp >R:\example
 *	cc TCLP -fop m=s
 */
// #define	DEBUG	1
#include <stdio.h>
#include <debug.h>

#define	POOL_SIZE	50000	// Memory storage pool size
#define	Xerror		error	// Replace error code

// Fixed parameters
#define	LINE_SIZE	150		// Maximum size of input line
#define	SYMBOL_SIZE	25		// Maximum size of a symbol name
#define	FILES		5		// Maximum number of nested files
#define	MAX_CHAR	120		// Maximum characters in a code block
#define	IOBUFF_SIZE	512		// Size of I/O buffers
#define	MAX_MEM		2000	// Maximum number of memory locations

// Memory map allocation flags
#define	MEM_NA		0		// Memory not available
#define	MEM_AVAIL	1		// Memory is available
#define	MEM_USED	2		// Memory used flag
#define	MEM_FREE	99		// Memory available flag

// Memory pool variable types (must have high bit set)
#define	MEMORY		128		// Symbol table type is memory location
#define	VALUE		129		// Symbol table type is simple value
#define	MACRO		130		// Macro definition
#define	STRING		131		// String definition
#define	LABEL		132		// Symbol table type is code label

FILE
	*fpi,					// Input file pointer
	*fpo,					// Output file pointer
	*file_stack[FILES];		// Stack of pending files

unsigned
	line_number,			// Current file line number
	total_lines,			// Total number of lines read
	cmd_number,				// Current command line number
	char_count,				// Count of characters in block
	char_total,				// Character total count
	pool_top,				// High water marker in data pool
	symbol_top,				// General symbol location
	block,					// Active block
	block_count,			// Count of allocated blocks
	file_top,				// Pending file stack
	line_stack[FILES],		// Stack of file line numbers
	gvalue = 8,				// Global value of symbol
	macro_seg,				// Macro storage segment
	macro_top=1,			// Macro storage limit
	macro_count,			// Macro count
	macro_ptr,				// Pointer to active macro
	day,					// Build day
	month,					// Build month
	year,					// Build year
	hour,					// Build hour
	minute,					// Build minute
	second;					// Build second

unsigned char
	Pass,					// Pass indicator
	*parse_ptr,				// General parse pointer
	buffer[LINE_SIZE+1],	// General (input) buffer
	symbol[SYMBOL_SIZE+1],	// General symbol buffer
	expand = -1,			// Output expand flag
	stats = -1,				// Statistics output
	Budl,					// Output big format UDL
	casef,					// Case sensitive flag
	repref,					// Report duplicate symbols
	process_flag = -1,		// Processing enabled flag
	gtype,					// Global type symbol
	udl,					// UDL output flag
	Upartial = '0',			// UDL partial download flag
	Uprefix,				// UDL prefix character
	macro_parms[80],		// Macro parameters
	pool[POOL_SIZE],		// Literal storage pool
	mem_map[MAX_MEM];		// memory map flags

extern unsigned IOB_size;

/*
 * Formatted print of error message
 */
register error(args)
	unsigned args;
{
	char buf[LINE_SIZE+1];
	unsigned i;

	_format_(nargs() * 2 + &args, buf);
	for(i=0; i < file_top; ++i)
		fprintf(stderr,"%u:", line_stack[i]);
	fprintf(stderr, "%u: %s\n", line_number, buf);
	fputs(buffer, stderr);
	putc('\n', stderr);
	exit(-1);
}

/*
 * Convert character to upper case if NOT case sensitive
 */
int chupper(int c)
{
	return casef ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Test that a character is a valid symbol character
 */
int issymbol(int c)
{
	return isalpha(c) || (c == '_');
}

/*
 * Skip blanks in the input
 */
int skip_blanks()
{
	while(isspace(*parse_ptr))
		++parse_ptr;
	return *parse_ptr;
}

int skip_end()
{
	int c;
	if((c = skip_blanks()) == ';')
		return 0;
	return c;
}

void test_end()
{
	if(skip_end())
		error("Syntax error");
}

/*
 * Strip a comment from a text buffer
 */
char *strip_comment(char s[])
{
	unsigned i;
	char c;

	i = 0;
	while(c = s[i]) {
		if(c == ';')
			break;
		++i; }
	while(i && isspace(s[i-1]))
		--i;
	s[i] = 0;
	return s;
}

/*
 * Parse a decimal number from the input
 */
unsigned parse_number()
{
	unsigned v;

	if(!isdigit(skip_blanks()))
		error("Bad number");
	v = 0;
	while(isdigit(*parse_ptr))
		v = (v * 10) + (*parse_ptr++ - '0');
	return v;
}

/*
 * Parse a symbol name from the input
 */
void parse_symbol()
{
	symbol_top = 0;
	while(issymbol(*parse_ptr) || isdigit(*parse_ptr))
		symbol[symbol_top++] = chupper(*parse_ptr++);
	symbol[symbol_top] = 0;
	if(symbol_top > SYMBOL_SIZE)
		error("Symbol '%s' exceeds "#SYMBOL_SIZE" characters (%u).\n",
			symbol, symbol_top);
}

/*
 * Mark a block of memory to a specific state
 */
void set_memory(unsigned low, unsigned high, unsigned char state)
{
	while(low <= high)
		mem_map[low++] = state;
}

/*
 * Lookup a symbol in the symbol table
 */
int lookup() asm
{
		XOR		BX,BX				; Zero index
; Test next entry in symbol pool
look1:	CMP		BX,DGRP:_pool_top	; Over top?
		JAE		look3				; Yes, exit
		MOV		SI,OFFSET DGRP:_symbol; Get symbol name
look2:	MOV		AL,[SI]				; Get char from symbol bame
		AND		AL,AL				; At end?
		JZ		look5				; Yes, exit
		CMP		AL,DGRP:_pool[BX]	; Does it match this one?
		JNZ		look4				; No, it does not
		INC		SI					; Advance to next in symbol
		INC		BX					; Advance to next in pool
		JMP	short look2				; Keep looking
; symbol was not found in the pool
look3:	XOR		AX,AX				; Symbol was not found
		JMP short look7				; exit
; This symbol is NOT the one, advance to next
look4:	MOV		AL,DGRP:_pool[BX]	; Get data from pool
		INC		BX					; Skip to next
		AND		AL,80h				; End of symbol?
		JZ		look4				; No, keep looking
look4a:	ADD		BX,2				; Advance to next
		JMP short look1				; And try again
; This could be the one
look5:	MOV		AL,DGRP:_pool[BX]	; Get data from pool
		INC		BX					; Skip to next
		TEST	AL,80h				; End of symbol?
		JZ		look4				; No, keep looking
		AND		AL,0BFh				; Clear reference bit
; This definately is the one
		MOV		DGRP:_gtype,AL		; Save type
		SUB		AL,LABEL			; Is this a label
		JB		look6				; No, normal symbol
		MOV		AH,AL				; Set high block
		MOV		AL,DGRP:_pool[BX]	; Get low  block
		CMP		AX,DGRP:_block		; Are we in this block
		JNZ		look4a				; No, keep looking
		MOV		AL,DGRP:_pool+1[BX]	; Read command value
		XOR		AH,AH				; Zero high
		MOV		DGRP:_gvalue,AX		; Save value
		MOV		AX,-2				; Indicate label found
		JMP short look7				; exit
; Standard symbol
look6:	MOV		AH,DGRP:_pool[BX]	; Get high value
		MOV		AL,DGRP:_pool+1[BX]	; Get low value
		MOV		DGRP:_gvalue,AX		; Save value
		MOV		AX,-1				; Indicate normal value
look7:
}

/*
 * Lookup a symbol, and mark it as referenced
 */
int lookup_mark() asm
{
		CALL	_lookup				; Lookup the symbol
		AND		AX,AX				; Was it successful?
		JZ		look8				; Not found
		MOV		CL,DGRP:_pool-1[BX]	; Get type back
		OR		CL,40h				; Set reference bit
		MOV		DGRP:_pool-1[BX],CL	; Resave it
look8:
}

/*
 * Add an entry to the symbol table
 */
void add_symbol(char *s, unsigned v, unsigned t)
{
	Debug(("Add [%02x]: %s = %u\n", t, s, v))

	do {
		pool[pool_top++] = *s; }
	while(*++s);
	pool[pool_top++] = t;
	pool[pool_top++] = v >> 8;
	pool[pool_top++] = v;
	if(pool_top > POOL_SIZE)
		Xerror("Out of memory");
}

/*
 * Check memory location to make sure available
 */
void checkmem(unsigned t)
{
	if(t >= MAX_MEM) {
		or:	error("Location out of range 0-%u", Budl ? 1999 : 999); }
	switch(mem_map[t]) {
	case MEM_NA		: goto or;
	case MEM_USED	: error("Location already assigned"); }
}

/*
 * Add a memory entry to the symbol table
 */
void add_memory(char *s, unsigned m)
{
	unsigned i;

	if(m == -1) {	/* Locate a free location */
		for(i=block; i < sizeof(mem_map); ++i)
			if(mem_map[i] == MEM_FREE)
				goto found;
		error("No memory locations available");
	found:
		m = i; }

	checkmem(m);

	mem_map[block = m] = MEM_USED;

	add_symbol(s, m, MEMORY);
}

/*
 * Generate "duplicate symbol" error
 */
void duplicate_error()
{
	error("Duplicate symbol: %s", symbol);
}

/*
 * Test a symbol to insure that if already defined, it is a MEMORY
 * type, and the location has not already been assigned.
 */
int test_symbol()
{
	int x;
	if(x = lookup()) {
		if(gtype != MEMORY)
			error("Incorrect symbol type: %s", symbol);

		checkmem(gvalue);
		mem_map[gvalue] = MEM_USED; }
	return x;
}
test_macro()
{
	switch(gtype) {
	case MACRO :
	case STRING:
		error("Incorrect symbol type: %s", symbol); }
}

/*
 * Process a conditional directive
 */
void conditional()
{
	unsigned value[10], vptr;
	char nflag;

	vptr = nflag = 0;

	switch(*++parse_ptr) {
	case '$' :		/* Toggle/else */
		process_flag = process_flag ? 0 : -1;
		return;
	case '!' :		/* Invert condition */
		nflag = -1;
		++parse_ptr; }

	while(skip_end()) {
		if(isdigit(skip_blanks()))
			gvalue = parse_number();
		else {
			parse_symbol();
			if((!lookup_mark()) || (gtype >= LABEL))
				error("Unknown symbol: %s\n", symbol);
			test_macro(); }
		value[vptr++] = gvalue; }

	if(!vptr)
		goto always;

	if(vptr == 1) {
		if(value[0])
			goto always; }
	else {
		while(vptr > 1) {
			if(value[0] == value[--vptr]) {
		always:
				process_flag = nflag ? 0 : -1;
				return; } } }

	process_flag = nflag ? -1 : 0;
}

/*
 * Skip command in the input
 */
void skip_command()
{
	unsigned i;
	unsigned char c, *p;

	p = parse_ptr;

	switch(*parse_ptr) {
	case '+' :
	case '*' :
	case '!' :
		++parse_ptr; }
	if(!isalpha(*parse_ptr))
		error("Bad command format: %s", p);
	++parse_ptr;
loop:
	switch(*parse_ptr) {
	case ' ' :
	case '\t':
	case '.' :
	case ',' :
	case '#' :
	case '-' :
	case '@' :
		++parse_ptr;
		goto loop; }
	if(isdigit(*parse_ptr)) {
		do {
			++parse_ptr; }
		while(isdigit(*parse_ptr));
		goto loop; }
	if(*parse_ptr == '\'') {
		do {
			if(!*++parse_ptr)
				error("Unterminated string"); }
		while(*parse_ptr != '\'');
		++parse_ptr;
		goto loop; }
	i = 0;
	while(issymbol(parse_ptr[i]) || isdigit(parse_ptr[i]))
		++i;
	if(i) {
		parse_ptr += i;
		goto loop; }

	c = *parse_ptr;
	*parse_ptr = 0;
	Debug(("Cmd: '%s'\n", p))
	*parse_ptr = c;
}

/*
 * Get a byte from the code block text
 */
unsigned help_ptr;
char get_text() asm
{
	MOV	BX,DGRP:_help_ptr			; Get offset
	INC WORD PTR DGRP:_help_ptr		; Increment
	MOV	AL,CS:byte ptr HELLO[BX]	; Get byte
}

/*
 * Display a help topic
 */
void help(unsigned topic)
{
	unsigned char c;
	static unsigned output, tab = 8;

	if(topic != 99) {
		while(topic--) {
			while(get_text());
			if(!(tab = get_text()))
				error("Help block not available"); } }
show_next:
	while(c = get_text()) {
		if(c == '\t') {
			do
				putc(' ', stdout);
			while(++output % tab); }
		else {
			output = (c != '\n') && output+1;
			putc(c, stdout); } }
	if((topic == 99) && (tab = get_text())) {
		putc('\n', stdout);
		goto show_next; }
}

/*
 * Read a symnbol table entry (pointed to by: symbol_top)
 * into symbol, gtype and gvalue entries
 * returns unmodified symbol type entry (reference bit intact)
 * or 0 if end of table
 */
unsigned read_symbol_entry(unsigned w)
{
	unsigned l;
	unsigned char b;

	while(symbol_top < pool_top) {
		l = 0;
		while(!(pool[symbol_top] & 0x80))
			symbol[l++] = pool[symbol_top++];
		while(l < w)
			symbol[l++] = ' ';
		symbol[l] = 0;
		gtype = (b = pool[symbol_top++]) & 0xBF;
		if(gtype >= LABEL) {
			block = ((gtype - LABEL) << 8) | pool[symbol_top++];
			gvalue = pool[symbol_top++];
			return b; }
		gvalue = pool[symbol_top++] << 8;
		gvalue |= pool[symbol_top++];
		return b; }
	return 0;
}

/*
 * Write a string to the output file with symbol substitutions
 */
void write_string()
{
	unsigned p;
	unsigned char c, buffer[256];

	p = 0;
	while(c = *parse_ptr++) {
		if(c == '%') {
			if(issymbol(c = *parse_ptr)) {
				parse_symbol();
				if((!lookup_mark()) || (gtype >= LABEL))
					error("Unknown symbol: %s\n", symbol);
				if(gtype == STRING) {
					while(c = peek(macro_seg, gvalue++))
						buffer[p++] = c; }
				else {
					test_macro();
					sprintf(buffer+p, "%u", gvalue);
			xz:		while(buffer[p]) ++p; }
				continue; }
			if(isdigit(c)) {
				++parse_ptr;
				switch(c) {
				case '1' : c = second;	break;
				case '2' : c = minute;	break;
				case '3' : c = hour;	break;
				case '4' : c = day;		break;
				case '5' : c = month;	break;
				case '6' : c = year%100;break;
				case '7' : sprintf(buffer+p, "%04u", year); goto xz;
				default: error("Bad time index '%c'\n", c); }
				sprintf(buffer+p, "%02u", c);
				goto xz; }
			if(!c)
				error("No '%%' data at end of line.");
			buffer[p++] = *parse_ptr++;
			continue; }
		if(c == ';') {
			if(expand) {
				strcpy(buffer+p, parse_ptr-1);
				while(buffer[p]) ++p; }
			while(p && isspace(buffer[p-1]))
				--p;
			break; }
		if((c == '\t') && udl)
			continue;
		buffer[p++] = c; }
	buffer[p] = 0;
	fputs(buffer, fpo);
}

unsigned eval();
unsigned get_value()
{
	unsigned b, c, v;
	skip_blanks();
	if(isdigit(*parse_ptr)) {
		b = 10;
donum:	v = 0;
		for(;;) {
			if(isdigit(c = toupper(*parse_ptr)))
				c -= '0';
			else if((c >= 'A') && (c <= 'F'))
				c -= ('A'-10);
			else
				return v;
			if(c >= b)
				return v;
			++parse_ptr;
			v = (v * b) + c; } }
	else if(issymbol(*parse_ptr)) {
		parse_symbol();
		if(!lookup()) {
			if(!(Pass & 0xF0)) {
				Pass |= 0x01;
				return 0; }
			error("Unknown symbol: %s\n", symbol); }
		if(gtype > VALUE)
			error("Incorrect symbol type: %s", symbol);
		return gvalue; }
	switch(*parse_ptr++) {
		case '{' : return eval();
		case '-' : return -get_value();
		case '~' : return ~get_value();
		case '$' : b = 16;	goto donum;
		case '@' : b = 8;	goto donum;
		case '!' : b = 2;	goto donum;
		case '\'':
			++parse_ptr;
			if(*parse_ptr++ == '\'')
				return *(parse_ptr-2); }
	error("Syntax error");
}
unsigned eval()
{
	unsigned v;

	v = get_value();
top:
	skip_blanks();
	switch(*parse_ptr++) {
	case '}' : return v;
	case '+' : v += get_value();			goto top;
	case '-' : v -= get_value();			goto top;
	case '*' : v *= get_value();			goto top;
	case '/' : v /= get_value();			goto top;
	case '%' : v %= get_value();			goto top;
	case '&' : v &= get_value();			goto top;
	case '|' : v |= get_value();			goto top;
	case '^' : v ^= get_value();			goto top;
	case '=' : v = v == get_value();		goto top;
	case '<' : switch(*parse_ptr++) {
		case '>' : v = v != get_value();	goto top;
		case '<' : v <<= get_value();		goto top;
		case '=' : v = v <= get_value();	goto top; }
		--parse_ptr; v = v < get_value();	goto top;
	case '>' : switch(*parse_ptr++) {
		case '>' : v >>= get_value();		goto top;
		case '=' : v =  v >= get_value();	goto top; }
		--parse_ptr; v = v > get_value();	goto top; }

	error("Syntax error");
}
int get_line()
{
	unsigned char c, *p, *p1;
	unsigned char lb[LINE_SIZE+1], fmt[10];

	if(macro_ptr) {		// Macro substitution in effect
		parse_ptr = lb;
		do {
			while(c = peek(macro_seg, macro_ptr++)) {
				if(c == 0xFF) {
					macro_ptr = 0;
					break; }
				if(c == '~') {	// Parameter value
					if(!isdigit(c = peek(macro_seg, macro_ptr++)))
						error("Bad parameter in macro.");
					c -= ('0'-1);
					p = macro_parms-1;
					do {		// Find end of parameter
						while(isspace(*++p));
						p1 = p;
						while((*p != ',') && (*p != ';') && *p)
							++p;
						if(!--c) {	// Parameter found
							while((p > p1) && isspace(*(p-1)))
								--p;
							while(p1 < p)
								*parse_ptr++ = *p1++;
							break; } }
					while(*p == ',');
					continue; }
				*parse_ptr++ = c; }
			*parse_ptr = 0;
			if(*(parse_ptr = lb))
				goto doin; }
		while(macro_ptr); }
	if(!fgets(parse_ptr=lb, LINE_SIZE, fpi))
		return 0;
	++line_number;
	++total_lines;
doin:
	p = buffer;
	while(c = *parse_ptr++) {
		if(c == '{') {
			strcpy(p1 = fmt, "%u");
			if(*parse_ptr == '%') {
				for(;;) switch(c = toupper(*parse_ptr++)) {
				case 'C' :
				case 'X' : Pass |= 0x02;
				case 'D' :
				case 'U' :
				case 'B' :
				case 'O' : *p1++ = c + ('a'-'A'); *p1 = 0; goto dofmt;
				default:
					if(!isdigit(c))
						error("Bad format");
				case '%' :
				case '-' : *p1++ = c; } }
	dofmt:	sprintf(p, fmt, eval());
			while(*p) ++p;
			continue; }
		*p++ = c; }
	*p = 0;
	parse_ptr = buffer;
	return -1;
}

/*
 * Close a UDL output file
 */
void close_udl(unsigned char nl)
{
	if(udl == 255) {
		udl = 15;
		putc('"', fpo); }
	else if(nl)
		return;
	putc('\n', fpo);
}

/*
 * Test for UDL output, and generate prefix if so
 */
int test_udl()
{
	if(udl) {
		close_udl(255);
		return udl = 255; }
	return 0;
}

unsigned char *getf(unsigned char *file, unsigned char *dext)
{
	unsigned char *p, *ext;

	p = buffer+70;
top:
	ext = 0;
loop:
	switch(*p++ = toupper(*file++)) {
		case ':' :
		case '\\': goto top;
		case '.' : ext = p;
		default: goto loop;
		case 0 : }
	if(!ext) {
		*(p-1) = '.';
		strcpy(ext = p, dext); }
	return ext;
}

void getmac()
{
	if(!macro_seg) {
		if(!(macro_seg = alloc_seg(4096)))
			error("Out of memory"); }
}

main(int argc, char *argv[])
{
	unsigned i, t, pp;
	char *p, flag, pf;

	IOB_size = IOBUFF_SIZE;
	t = 0;		/* Assume help block 0 */

	get_date(&day, &month, &year);
	get_time(&hour, &minute, &second);

	/*
	 * Parse command line options and filenames
	 */
	for(i=1; i < argc; ++i) {
		parse_ptr = argv[i];
		switch((toupper(*parse_ptr++) << 8) | toupper(*parse_ptr++)) {
		case ('/'<<8)|'A' :
		case ('-'<<8)|'A' : Upartial = '1';	continue;
		case ('/'<<8)|'C' :
		case ('-'<<8)|'C' : casef = -1;		continue;
		case ('/'<<8)|'H' :
		case ('-'<<8)|'H' : help(*parse_ptr ? parse_number() : 1); return;
		case ('/'<<8)|'R' :
		case ('-'<<8)|'R' : repref = -1;	continue;
		case ('/'<<8)|'S' :
		case ('-'<<8)|'S' : stats = 0;		continue;
		case ('/'<<8)|'U' :
		case ('-'<<8)|'U' : Budl = 255;
			set_memory(1000, 1049, MEM_AVAIL);
			set_memory(1050, 1999, MEM_FREE);
			continue;
		case ('/'<<8)|'W' :
		case ('-'<<8)|'W' :	expand = 0;		continue;
		case ('?'<<8):
		case ('/'<<8)|'?' :
		case ('-'<<8)|'?' : goto do_help; }

		flag = issymbol(*(parse_ptr = argv[i]));
		while(*parse_ptr = chupper(*parse_ptr)) {
			if(*parse_ptr == '=')
				goto set_symbol;
			if(!(issymbol(*parse_ptr) || isdigit(*parse_ptr)))
				flag = 0;
			++parse_ptr; }

		if(!fpi) {
			getf(argv[i], "TCP");
			fpi = fopen(buffer+70, "rvq");
			continue; }
		if(!fpo) {
			if(!strcmp(p = getf(argv[i], "TCL"), "UDL")) {
				udl = 15;
				expand = 0; }
			else if(strcmp(p, "TCL"))
				error("Output must be 'TCL' or 'UDL'");
			fpo = fopen(buffer+70, "wvq");
			continue; }
		error("Too many filenames!");

	set_symbol:
		if(!flag)
			error("Bad symbol name");
		*parse_ptr++ = 0;
		if(*parse_ptr == '"') {
			getmac();
			add_symbol(argv[i], macro_top, STRING);
			do {
				poke(macro_seg, macro_top++, *++parse_ptr); }
			while(*parse_ptr);
			continue; }
		add_symbol(argv[i], parse_number(), VALUE);
		test_end(); }

	if(!fpi) {	/* No input file - issue help text */
	do_help:
		help(0);
		return; }

	if(!fpo) goto do_help;

	/* Issue "welcome" message */
	if(stats) {
		while((flag = get_text()) >= ' ')
			putc(flag, stdout);
		printf(" - ?COPY.TXT 2000-2006 Dave Dunfield -  -- see COPY.TXT --.\n"); }

	/*
	 * Mark general memory locations
	 */
	set_memory(0, 999, MEM_AVAIL);
	set_memory(40, 99, MEM_FREE);
	for(i=100; i <= 800; i += 100)
		set_memory(i+13, i+99, MEM_FREE);
	set_memory(913, 949, MEM_FREE);

	/*
	 * 1st pass - read source file and define all symbols
	 */
pass1:
	while(Pass = 0, get_line()) {
		Debug(("%u.%u: %s\n", line_number, cmd_number, buffer))
		if(*buffer == '?') {	/* Conditional block */
			if(Pass)
				error("Illegal forward reference");
			conditional();
			continue; }
		if(!process_flag)
			continue;
		if((Pass & 3) == 3)
			error("Illegal forward reference");
		/* Handle special case input lines */
		switch(*buffer) {
		case '@' :		/* Include file */
			if(file_top >= (FILES-1))
				error("Nested files exceed "#FILES" deep.");
			line_stack[file_top] = line_number;
			file_stack[file_top++] = fpi;
			getf(strip_comment(buffer+1), "TCP");
			fpi = fopen(buffer+70, "rvq");
			line_number = 0;
			continue;
		case '!' :		/* "raw" output line */
			++parse_ptr;
			cmd_number += parse_number();
		case '%' :		/* EPROM version */
			continue;
		case '~' :		/* MACRO definition */
		remac1:
			if(!issymbol(*++parse_ptr))
				error("Bad symbol name");
			parse_symbol();
			if(lookup())
				duplicate_error();
			getmac();
			add_symbol(symbol, macro_top, MACRO);
			++macro_count;
			do {
				do
					poke(macro_seg, macro_top++, *parse_ptr);
				while(*parse_ptr++);
				if(!fgets(parse_ptr = buffer, LINE_SIZE, fpi))
					error("End of file in macro");
				++line_number; }
			while(*buffer != '~');
			poke(macro_seg, macro_top++, 0xFF);
			if(issymbol(buffer[1]))
				goto remac1;
			continue;
		case '&' : if(*++parse_ptr != '=') break;
		case '=' : doeq1:
			if(mem_map[++block] != MEM_FREE)
				error("Cannot allocate sequential location");
			mem_map[block] = MEM_USED;
			continue; }
		/* Handle numeric block definitions */
		if(isdigit(*parse_ptr)) {
			t = parse_number();
			checkmem(t);
			mem_map[block = t] = MEM_USED;
			while(isdigit(*parse_ptr))
				++parse_ptr;
			if(skip_blanks() == '$') {
				++block_count;
				cmd_number = 0; }
			continue; }
		/* Handle symbolic definitions */
		if(issymbol(*parse_ptr)) {
			p = parse_ptr;
			parse_symbol();
			t = -1;
			switch(skip_blanks()) {
			case '@' :		/* Memory block at specified address */
				++parse_ptr;
				t = parse_number();
				if(lookup())
					duplicate_error();
				switch(skip_end()) {
				default: error("Bad memory location");
				case 0 :
					add_symbol(symbol, block = t, MEMORY);
					continue;
				case '=' : goto do_equal;
				case '$' : goto do_dollar; }
			case '$' :		/* Code block */
				if(!test_symbol()) {
			do_dollar:
					add_memory(symbol, t); }
				++block_count;
				cmd_number = 0;
				++parse_ptr;
				break;
			case '=' :		/* Data block */
				if(!test_symbol()) {
			do_equal:
					add_memory(symbol, t); }
				continue;
			case '#' :		/* Constant definition */
				if((flag = *++parse_ptr) == '?')
					++parse_ptr;
				if(lookup()) {
					if(flag == '?')
						continue;
					duplicate_error(); }
				add_symbol(symbol, parse_number(), VALUE);
				test_end();
				continue;
			case '"' :		/* String definition */
				if((flag = *++parse_ptr) == '?')
					++parse_ptr;
				if(lookup()) {
					if(flag == '?')
						continue;
					duplicate_error(); }
				getmac();
				add_symbol(symbol, macro_top, STRING);
				strip_comment(parse_ptr);
				for(;;) {
					poke(macro_seg, macro_top++, *parse_ptr);
					if(!*parse_ptr)
						break;
					++parse_ptr; }
				continue;
			case ':' :		/* Code label */
				++parse_ptr;
				if(lookup())
					duplicate_error();
				add_symbol(symbol, (block << 8) | cmd_number, (block >> 8) + LABEL);
				break;
			default:
				parse_ptr = p; } }

		/*
		 * Process remainder of line
		 */
		if(skip_end()) {
			if(*parse_ptr == '=') goto doeq1;
			if(*parse_ptr == '~') {
				++parse_ptr;
				parse_symbol();
				if((!lookup_mark()) || (gtype != MACRO))
					error("Unknown Macro: %s", symbol);
				skip_blanks();
				strcpy(macro_parms, parse_ptr);
				macro_ptr = gvalue;
				continue; }
			skip_command();
			++cmd_number;
			if(skip_end()) {
					error("Unknown command format"); } } }

	if(macro_ptr)
		error("Unterminated macro");

	/* End of file - if stacked input, restore previous and keep going */
	if(file_top) {
		fclose(fpi);
		fpi = file_stack[--file_top];
		line_number = line_stack[file_top];
		goto pass1; }

	Pass = 255;
	/*
	 * 2nd pass - read again and generate output code
	 */
	rewind(fpi);
	total_lines = line_number = cmd_number = block = 0;
	process_flag = -1;
pass2:
	while(get_line()) {
		Debug(("%u.%u: %s\n", line_number, cmd_number, buffer))
		Uprefix = Upartial;
		if(*buffer == '?') {	/* Conditional block */
			conditional();
			continue; }
		if(!process_flag)
			continue;
		pf = 0;
		/* Handle special case input lines */
		switch(*buffer) {
		case '%' :		/* EPROM version */
			if(!udl) {
				fputs(buffer, fpo);
				putc('\n', fpo); }
			continue;
		case '!' :		/* "raw" output line */
			test_udl();
			++parse_ptr;
			cmd_number += parse_number();
			fputs(parse_ptr, fpo);
			close_udl(0);
			continue;
		case '@' :		/* Include file */
			line_stack[file_top] = line_number;
			file_stack[file_top++] = fpi;
			fpi = fopen(strip_comment(buffer+1), "rvq");
			line_number = 0;
			continue;
		case '~' :		/* MACRO definition */
		remac2:
			do {
				if(!fgets(parse_ptr = buffer, LINE_SIZE, fpi))
					error("End of file in macro");
				++line_number; }
			while(*buffer != '~');
			if(issymbol(buffer[1]))
				goto remac2;
			continue;
		case '&' : pf = Uprefix = '1';
			if(*++parse_ptr != '=') break;
		case '=' : doeq2:
			gvalue = ++block;
			goto do_eq2; }
		/* Handle numeric block definitions */
		if(isdigit(*parse_ptr)) {
			gvalue = parse_number();
			switch(skip_blanks()) {
			case '=' : goto do_eq2;
			case '$' : goto do_co1; }
			error("Absolute block must be '$' or '='"); }
		/* Handle symbolic definitions */
		if(issymbol(*parse_ptr)) {
			p = parse_ptr;
			parse_symbol();
			switch(skip_blanks()) {
			case '@' :		/* Memory block at specified address */
				++parse_ptr;
				parse_number();
				switch(skip_end()) {
				case 0 : if(pf) goto badcmd;
					continue;
				case '=' : goto do_eq1;
				case '$' : }
			case '$' :		/* code block */
				if(!lookup())
					error("Unknown symbol: %s\n", symbol);
			do_co1:
				if(test_udl())
					fprintf(fpo, Budl ? "\"%c,%04u," : "\"%c%03u", Uprefix, block = gvalue);
				else {
					if(Uprefix == '1') putc('&', fpo);
					fprintf(fpo, "%u", block = gvalue);
					putc(*parse_ptr, fpo); }
				++parse_ptr;
				char_total += char_count;
				cmd_number = char_count = pf = 0;
				break;
			case '=' :		/* Data block */
			do_eq1:
				if(!lookup())
					error("Unknown symbol: %s\n", symbol);
			do_eq2:
				if(test_udl()) {
					fprintf(fpo, Budl ? "\"%c,%04u," : "\"%c%03u", Uprefix, block = gvalue);
					++parse_ptr; }
				else {
					if(Uprefix == '1') putc('&', fpo);
					fprintf(fpo, "%u", block = gvalue); }
				write_string(fpo);
				close_udl(pf=0);
				continue;
			case ':' :		/* Code label */
				++parse_ptr;
				break;
			case '#' :		/* Constant definition */
			case '"' :		/* String definition */
				if(pf) goto badcmd;
				continue;
			default:
				parse_ptr = p; } }

	/*
	 * Process remainder of command line
	 * Advance command count if command present.
	 * Perform symbol substitution in command options.
	 * Add up characters in block and check againt maximum.
	 */
			flag = pf;
	skip_cmd:
			p = parse_ptr;
		/* Ignore whitespace preceeding command */
//			while(isspace(*parse_ptr))
//				++parse_ptr;
//			switch(*parse_ptr) {
			switch(skip_blanks()) {
			case '=' : goto doeq2;
			case ';' :	/* Comment - no command */
				if(expand)
					fputs(p, fpo);
			case 0 :	/* End of line - no command */
				if(pf) goto badcmd;
				if(!udl) {
					if(expand || (p != buffer))
						putc('\n', fpo); }
				continue; }
			if(flag) {
				badcmd: error("Unknown command format"); }
			if(*parse_ptr == '~') {		// Macro expansion
				++parse_ptr;
				parse_symbol();
				if((!lookup_mark()) || (gtype != MACRO))
					error("Unknown Macro: %s", symbol);
				skip_blanks();
				strcpy(macro_parms, parse_ptr);
				macro_ptr = gvalue;
				continue; }
			if(expand) {				// Expand whitespace
				while(p < parse_ptr)
					putc(*p++, fpo); }
			switch(*parse_ptr) {
			case '+' :	/* Allowed command prefix - ignore */
			case '*' :	/* Allowed command prefix - ignore */
			case '!' :	/* Allowed command prefix - ignore */
				putc(*parse_ptr++, fpo);
				++char_count; }
			/* Expect alphabetic command */
			if(!isalpha(*parse_ptr))
				error("Bad command format: %s", p);
			putc(*parse_ptr++, fpo);
			++char_count;
	skip_op:
		/* Ignore whitespace in command */
		if(isspace(*parse_ptr)) {
			if(expand)
				putc(*parse_ptr, fpo);
			++parse_ptr;
			goto skip_op; };
		/* Allow special characters in command options */
		switch(*parse_ptr) {
		case '.' :
		case ',' :
		case '#' :
		case '-' :
		case '@' :
			putc(*parse_ptr++, fpo);
			++char_count;
			goto skip_op; }
		/* Allow constant numbers in command options */
		if(isdigit(*parse_ptr)) {
			do {
				putc(*parse_ptr++, fpo);
				++char_count; }
			while(isdigit(*parse_ptr));
			goto skip_op; }
		/* Allow strings in command options */
		if(*parse_ptr == '\'') {
			putc('\'', fpo);
			++char_count;
			do {
				if(!*++parse_ptr)
					error("Unterminated string");
				putc(*parse_ptr, fpo);
				++char_count; }
			while(*parse_ptr != '\'');
			++parse_ptr;
			goto skip_op; }
		/* Process symbols in command and substitute definition */
/*		symbol_top = 0;
		while(issymbol(parse_ptr[symbol_top]) || isdigit(parse_ptr[symbol_top])) {
			symbol[symbol_top] = chupper(parse_ptr[symbol_top]);
			++symbol_top; } */
		parse_symbol();
		if(symbol_top) {	/* Symbol was found */
/*			symbol[symbol_top] = 0;
			parse_ptr += symbol_top; */
			if(!lookup_mark())
				error("Unknown symbol: %s", symbol);
			test_macro();
			if(gtype >= LABEL) {	/* Code label - compute distance from here */
				if(gvalue == cmd_number)
					error("Cannot skip to current line");
				if(gvalue > cmd_number)
					--gvalue;
				char_count += fprintf(fpo, "%d", gvalue - cmd_number); }
			else					/* Memory address or constant */
				char_count += fprintf(fpo, "%u", gvalue);
			goto skip_op; }
		if(char_count > MAX_CHAR)
			error("Block exceeds %u characters", MAX_CHAR);
		++cmd_number;
		flag = -1;
		goto skip_cmd; }

	/* End of file - if stacked input, restore previous and keep going */
	if(file_top) {
		fclose(fpi);
		fpi = file_stack[--file_top];
		line_number = line_stack[file_top];
		goto pass2; }

	/* If status output enabled - generate status records */
	if(stats) {
		for(i=t=0; i < sizeof(mem_map); ++i) {
			if(mem_map[i] == MEM_USED)
				++t; }
		printf("%u lines, %u blocks, %u code blocks, containing %u command bytes\n",
			total_lines, t, block_count, char_total + char_count);
		printf("%u of %u bytes of symbol table memory used.\n", pool_top, POOL_SIZE);
		if(macro_top & 0xFFFE)
			printf("%u macros, %u bytes of macro/string storage used.\n", macro_count, macro_top);
		if(repref)
			printf("\n"); }

	if(repref) {
		symbol_top = 0;
		while(t = read_symbol_entry(SYMBOL_SIZE)) {
			if(!(t & 0x40)) {
				switch(gtype) {
				case MACRO  : p = "~"; break;
				case MEMORY : p = "@"; break;
				case VALUE  : p = "#"; break;
				default: p = ":"; ++gvalue; }
				if(repref) {
					printf("Unreferenced symbols:\n");
					repref = 0; }
				printf("   %s%s", symbol, p);
				if(gtype >= LABEL) {
					file_top = gvalue;
					fputs(" ", stdout);
					pp = block;
					t = symbol_top;
					symbol_top = flag = 0;
					while(read_symbol_entry(0)) {
						if((gtype == MEMORY) && (gvalue == pp)) {
							if(flag)
								putc(',', stdout);
							fputs(symbol, stdout);
							flag = -1; } }
					symbol_top = t;
					printf("@%u -Line", pp);
					gvalue = file_top; }
				printf(" %u\n", gvalue); } } }
	close_udl(255);
	fclose(fpo);
	fclose(fpi);
}

/*
 * Help text - stored in code segment to avoid using precious data space
 */
asm {
HELLO:
DB'TCL Preprocessor 1.4'
DB 10,10
DB'use:	TCLP <input_file[.TCP]> <output_file[.TCL/UDL] [options]',10,10
DB'opts:	-A		= mark All .UDL locations for parameter download',10
DB'	-C		= Case sensitive symbols',10
DB'	-H[n]		= display TCLP Help topics',10
DB'	-R		= Report unreferenced symbols',10
DB'	-S		= do not output Statistics',10
DB'	-U		= enable Upper memory (1000-1999)',10
DB'	-W		= remove Whitespace and comments',10
DB'	symbol=["]value	= Preset symbol value',10,10
DB'?COPY.TXT 2000-2006 Dave Dunfield',10
DB' -- see COPY.TXT --.',10,0,8
DB'TCLP HELP Topics:',10,10
DB'	-H0	- TCLP command syntax',10
DB'	-H1	- TCLP help topics',10
DB'	-H2	- TCLP overview',10
DB'	-H3	- TCLP extended source format',10
DB'	-H4	- TCLP conditional processing',10
DB'	-H5	- TCLP macro processing',10
DB'	-H6	- TCLP data timestamp encoding',10
DB'	-H7	- TCLP arithmetic functions',10
DB'	-H8	- TCLP example source code',10
DB'	-H9	- Contact information',10,10
DB'	-H99	- Display all topics',10,10
DB'To obtain help output in a file, use the DOS redirection operator:',10,10
DB'	eg: TCLP -H8 >EXAMPLE.TCP',10,0
; TCLP overview
DB	4
DB'TCLP Overview:',10,10
DB'TCLP adds symbolic capability to the VeriFone Terminal Control Language.',10,10
DB'Symbols may be defined for any memory location, any constant value, or any',10
DB'line within a code block. These memory addresses, constants and code lines',10
DB'may then be referenced within command operands by symbolic name instead of',10
DB'by absolute (memory and constants) or relative (code lines) values.',10,10
DB'TCLP operates in two passes...',10
DB'On the first pass, symbols are defined and code commands are counted in order',10
DB'to be able to calculate the relative offsets for references to code labels.',10
DB'On the second pass, the output code is generated with symbol substitutions.',10,10
DB'TCLP does some parsing of the command lines in order to be able to count the',10
DB'commands and keep track of the size of each code block, however it DOES NOT',10
DB'perform comprehensive error checking on the command operands. In particular,',10
DB'TCLP does not know the expected type for the operand positions of each command.',10
DB'It will happily put a memory address or constant value in a "skip" position',10
DB'if that type of symbol is placed in that operand position. It will likewise',10
DB'put a command line "skip" value in a location that is expecting a memory',10
DB'address if a code label is referenced in that position.',10,0
; Extended source format help
DB	8
DB'TCLP Extended Source Format:',10,10
DB'symbol@xxx		<- Define symbol for memory location xxx	*1',10
DB'[&]symbol@xxx=init	<- Define symbol and initialize location     *1 *2',10
DB'[&]symbol@xxx$		<- Define symbol and begin code block		*1',10
DB'[&]symbol=init		<- Initialize symbolic memory location	  *1 *2 *3',10
DB'[&]=init		<- Initialize next (no symbol) location		*2',10
DB'[&]symbol$		<- Begin code block in symbolic location     *1 *3',10
DB'[&]symbol:		<- Define local label (within code block)	*1',10
DB'symbol#n		<- Define symbol with constant value (0-65535)	*1',10
DB'symbol"string		<- Define symbol with string value	     *1 *4',10
DB'@filename		<- Include named file',10
DB'?[!][v ...]		<- Conditional processing (-H4)',10
DB'~[symbol ...]		<- Macro (-H5)',10
DB'!n text			<- Send text to output file with no processing	*5',10
DB'   C ...		<- Command with operands (one per line)		*1',10,10
DB'*1 Symbol names must begin in column #1',10
DB"*2 '%symbol' puts symbol value in data. '%n'(1-7) encodes time (-H6)",10
DB"   If you want %symbols substituted in data comments, use '%;'",10
DB'*3 If symbol does not exist, The next unused general location (40-99,',10
DB'   x13-x99, 913-949) following the last location assigned will be used.',10
DB'*4 String symbols can ONLY be used in ',39,'=',39,' data blocks',10
DB'*5 n=#commands in line (necessary to track command labels)',10,0
DB	8
DB'TCLP Conditional Processing:',10,10
DB'TCLP supports conditional processing of the source file, allowing you to',10
DB'use symbols to modify the logic of the code which gets loaded into the unit.',10
DB'This is useful to enable/disable or change code blocks for debugging and',10
DB'shipping software releases.',10,10
DB'?v			<- Process this section only if v is non-zero',10
DB'?!v			<- Process this section only if v is zero',10
DB'?v n ...		<- Process section only if v =  at least one n',10
DB"?!v n ...		<- Process section only if v != all n's",10
DB'?			<- Always process this section (end conditional)',10
DB'?!			<- Never process this section (disable section)',10
DB'?$			<- Toggle processing ON/OFF   (else)',10
DB'symbol#?n		<- Define constant symbol only if not already defined',10
DB'symbol"?string		<- Define string   symbol only if not already defined',10,10
DB'See -H3 for overall details of the source file format.',10,0
DB	8
DB'TCLP macro processing:',10,10
DB'TCLP supports a MACRO capability, which allows you define a sequence of TCL',10
DB'instructions, and invoke the entire sequence with a single symbolic name.',10
DB'The macro is processed as if the entire sequence appeared in the source file',10
DB'at the point where the symbolic macro name appears. This is useful to avoid',10
DB're-entering commonly used instruction sequences again and again',10,10
DB'~name			<- Begin macro definition',10
DB'	C ~0...~9	<- Macro definition text',10
DB'~			<- End macro definition',10,10
DB'	...',10
DB'	~name p1,...	<- Macro invocation (within control string)',10,10
DB"Macro names can be any valid symbol name.",10
DB"Macros may have up to 10 parameters, which are referenced within the macro",10
DB"body by the escape sequences ~0-~9. Parameters passed to the macro MUST be",10
DB"separated by ','. (Use '.' to pass multiple TCL operands as one MACRO parm.)",10,10
DB'See -H3 for overall details of the source file format.',10,0
DB	8
DB"TCLP Timestamp Encoding:",10,10
DB"TCLP supports the encoding of the time of processing into a data string",10
DB"with the following '%n' directives in the data string:",10,10
DB"	%1 = Second",10
DB"	%2 = Minute",10
DB"	%3 = Hour",10
DB"	%4 = Day",10
DB"	%5 = Month",10
DB"	%6 = Year as 2 digits",10
DB"	%7 = Year as 4 digits",10,0
DB	4
DB"TCLP Arithmetic functions:",10,10
DB"At any point in your TCLP source file, you may use the following special",10
DB"arithmetic functions:",10
DB"	{expression}	= Insert expression value as unsigned decimal number",10
DB" {%[w]f expression}	= Insert specially formatted expression value",10
DB"		w:	Field width 	- = Left justify	Leading0 = Zero fill",10
DB"		f:	(U)nsigned signed-(D)ecimal (B)inary he(X) (O)ctal (C)haracter",10
DB"		*	You cannot use forward symbol references with %x, %c or any",10
DB"			{...} expression occuring in ?[!]conditional control statements",10
DB"expression consists of one or more values:",10
DB"	0..9	= Decimal number		-value	= Any other value negated",10
DB"	'c'		= Character constant	~value	= Any other value complimented",10
DB"	$0..F	= Hex number	  {expression}	= Subexpression (force precedence)",10
DB"	@0..7	= Octal number			!0..1	= Binary number",10
DB"and optional operators:",10
DB"	+	= Addition				<<	= Shift left",10
DB"	-	= Subtraction			>>	= Shift right",10
DB"	*	= Multiplication		=	= 1 if equal, 0 otherwise",10
DB"	/	= Division				<>	= 1 if not-equal, 0 otherwise",10
DB"	%	= Remainder				>	= 1 if greater, 0 otherwise",10
DB"	&	= Bitwise AND			>=	= 1 if greater or equal, 0 otherwise",10
DB"	|	= Bitwise OR			<	= 1 if less, 0 otherwise",10
DB"	^	= Bitwise XOR			<=	= 1 if less or equal, 0 otherwise",10,0
DB	8
#include "R:\example.h"
;DB	0,4
;#include "D:TCL"
DB 0,8,'TCLP was created by:',10,10
DB'	Dunfield Development Services',10
DB'	115 Manion Heights Cres.',10
DB'	RR#2 Carp, Ontario Canada',10
DB'	K0A 1L0',10,10
DB' Specializing in low cost embedded software development tools',10
DB' and software/firmware development services!',10,10
DB' http://www.dunfield.com',10,10
DB' FAX:	613-256-5821',10,10
DB'This program was compiled with DDS Micro-C/PC.',10
DB	0,0
}
