#include <stdio.h>
#include <file.h>
#include <comm.h>

#define	TICK	peekw(0x40,0x6C)

#define	MAX_SEG	4			// Maximum number of segments
#define	MAX_MEM	2000		// Maximum number of memory locations
#define	MAX_DTZ	1000		// Maximum number of DTZ location
#define	RETRY	5			// Number of retries

// fixed DTZ file offsets
#define	DTZ_CSUM	0x0080				// Checksum goes here
#define	DTZ_FREE	0x0082				// Free address goes here
#define	DTZ_FIXED	0x0084				// Fixed constants
#define	DTZ_PW		0x0087				// Password

#define	SOH			0x01
#define	STX			0x02
#define	ETX			0x03
#define	EOT			0x04
#define	ENQ			0x05
#define	ACK			0x06
#define	NAK			0x15
#define	FS			0x1C
#define	GS			0x1D

unsigned
	Cport,					// Comm port
	Timeout = 18,			// Comm timeout
	seg,					// General segment pointer
	image_size,				// Size of DTZ image
	MemAddr[MAX_MEM],		// Address of memory offsets
	SegList[8],				// List of segments
	SegTop[8],				// Size of segments
	Nptr,					// DTZ pointer to active nibble
	crc = 0x5555;			// Global crc

unsigned char
	Mem,					// Memory image is loaded
	Dtz,					// DTZ image is loaded
	Udl,					// Force UDL format terminal
	Budl,					// Big format UDL output
	Pass,					// Display DTZ password
	*pass,					// Set DTZ password
	Yes,					// Command YES option
	prc,					// Pending RX character
	function,				// Operating function
	verbose = 255,			// Verbose flag
	rmask = 0x7F,			// Receive data mask
	Pflags,					// Permanent flags
	*id,					// ID string
	*ptr,					// General pointer
	*pptr,					// Path pointer
	path[65],				// Home path
	wfile[65],				// File to write
	type,					// Type of file to read
	wtype,					// Type of file to write
	buffer[256],			// General buffer
	MemSeg[MAX_MEM],		// Memory segment indexes
	MemSize[MAX_MEM],		// Memory size
	ptable[256],			// Parity table
	Nflag;					// DTZ nibble pending flag

FILE
	*fp;

/*
 * Constant memory from the terminal database
 */
unsigned char
	DTZ_id[10] = { "IF@@OOOO" },		// Terminal ID string
	DTZ_init[] = {						// Fixed memory constants
		0x00,0x00,0x07,0x07,
		0x5A,0x36,0x36,0x38,0x33,0x31,0x00,0x00,
		0x00,0x00,0x00,0x96,0xFF,0xFF,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };
unsigned
	DTZ_ptr = 0x00FA,					// Pointer table address
	DTZ_adjust = 0x9E4A - 0x08CA,		// Address adjust value
	DTZ_l2 = 0x8900,					// Location 2 init. pointer
	DTZ_l3 = 0x8A00,					// Location 3 init. pointer
	DTZ_total = 27263;					// Total free memory

#define	DSIZE	(sizeof(DTZ_id)+sizeof(DTZ_init)+sizeof(DTZ_ptr)\
	+sizeof(DTZ_adjust)+sizeof(DTZ_l2)+sizeof(DTZ_l3)+sizeof(DTZ_total))

unsigned crctab[256] = {
	0x0000, 0x8005, 0x800F, 0x000A, 0x801B, 0x001E, 0x0014, 0x8011,
	0x8033, 0x0036, 0x003C, 0x8039, 0x0028, 0x802D, 0x8027, 0x0022,
	0x8063, 0x0066, 0x006C, 0x8069, 0x0078, 0x807D, 0x8077, 0x0072,
	0x0050, 0x8055, 0x805F, 0x005A, 0x804B, 0x004E, 0x0044, 0x8041,
	0x80C3, 0x00C6, 0x00CC, 0x80C9, 0x00D8, 0x80DD, 0x80D7, 0x00D2,
	0x00F0, 0x80F5, 0x80FF, 0x00FA, 0x80EB, 0x00EE, 0x00E4, 0x80E1,
	0x00A0, 0x80A5, 0x80AF, 0x00AA, 0x80BB, 0x00BE, 0x00B4, 0x80B1,
	0x8093, 0x0096, 0x009C, 0x8099, 0x0088, 0x808D, 0x8087, 0x0082,
	0x8183, 0x0186, 0x018C, 0x8189, 0x0198, 0x819D, 0x8197, 0x0192,
	0x01B0, 0x81B5, 0x81BF, 0x01BA, 0x81AB, 0x01AE, 0x01A4, 0x81A1,
	0x01E0, 0x81E5, 0x81EF, 0x01EA, 0x81FB, 0x01FE, 0x01F4, 0x81F1,
	0x81D3, 0x01D6, 0x01DC, 0x81D9, 0x01C8, 0x81CD, 0x81C7, 0x01C2,
	0x0140, 0x8145, 0x814F, 0x014A, 0x815B, 0x015E, 0x0154, 0x8151,
	0x8173, 0x0176, 0x017C, 0x8179, 0x0168, 0x816D, 0x8167, 0x0162,
	0x8123, 0x0126, 0x012C, 0x8129, 0x0138, 0x813D, 0x8137, 0x0132,
	0x0110, 0x8115, 0x811F, 0x011A, 0x810B, 0x010E, 0x0104, 0x8101,
	0x8303, 0x0306, 0x030C, 0x8309, 0x0318, 0x831D, 0x8317, 0x0312,
	0x0330, 0x8335, 0x833F, 0x033A, 0x832B, 0x032E, 0x0324, 0x8321,
	0x0360, 0x8365, 0x836F, 0x036A, 0x837B, 0x037E, 0x0374, 0x8371,
	0x8353, 0x0356, 0x035C, 0x8359, 0x0348, 0x834D, 0x8347, 0x0342,
	0x03C0, 0x83C5, 0x83CF, 0x03CA, 0x83DB, 0x03DE, 0x03D4, 0x83D1,
	0x83F3, 0x03F6, 0x03FC, 0x83F9, 0x03E8, 0x83ED, 0x83E7, 0x03E2,
	0x83A3, 0x03A6, 0x03AC, 0x83A9, 0x03B8, 0x83BD, 0x83B7, 0x03B2,
	0x0390, 0x8395, 0x839F, 0x039A, 0x838B, 0x038E, 0x0384, 0x8381,
	0x0280, 0x8285, 0x828F, 0x028A, 0x829B, 0x029E, 0x0294, 0x8291,
	0x82B3, 0x02B6, 0x02BC, 0x82B9, 0x02A8, 0x82AD, 0x82A7, 0x02A2,
	0x82E3, 0x02E6, 0x02EC, 0x82E9, 0x02F8, 0x82FD, 0x82F7, 0x02F2,
	0x02D0, 0x82D5, 0x82DF, 0x02DA, 0x82CB, 0x02CE, 0x02C4, 0x82C1,
	0x8243, 0x0246, 0x024C, 0x8249, 0x0258, 0x825D, 0x8257, 0x0252,
	0x0270, 0x8275, 0x827F, 0x027A, 0x826B, 0x026E, 0x0264, 0x8261,
	0x0220, 0x8225, 0x822F, 0x022A, 0x823B, 0x023E, 0x0234, 0x8231,
	0x8213, 0x0216, 0x021C, 0x8219, 0x0208, 0x820D, 0x8207, 0x0202 };

/*
 * Print formatted error message & exit
 */
register error(unsigned args)
{
	unsigned char buffer[150];
	_format_(nargs() * 2 + &args, buffer);
	fputs(buffer, stderr);
	if(fp)
		fclose(fp);
	Cclose();
	exit(-1);
}

/*
 * Prompt with a yes/no question and wait for response
 */
int yesno(char *prompt)
{
	unsigned char c;

	switch(Yes) {
	case 15 : return 0;
	case 255 : return 255; }

	printf("%s (Y/N)?", prompt);
	for(;;) switch(c = kbget()) {
		case 'Y' :
		case 'y' :
			printf("%c\n", c);
			return 255;
		case 'N' :
		case 'n' :
			printf("%c\n", c);
			return 0;
		case 0x03:
		case 0x1B: error("Exiting..."); }
}

/*
 * Display string with ASCII->hex transaction for non-printables
 */
void estring(unsigned char *msg, unsigned char *d)
{
	unsigned l;
	unsigned char c;
	printf("%s ", msg);
	for(l=0; (c = *d++) && (l < 10); ++l) {
		if((c < ' ') || (c > 0x7F)) {
			printf("{%02x}", c);
			continue; }
		putc(c, stdout); }
	putc('\n', stdout);
}

/*
 * Add character to running CRC value
 */
void addcrc(unsigned char c)
{
	crc = crctab[((crc >> 8) ^ c) & 255] ^ (crc << 8);
}

/*
 * Build table to transmit character with ODD partty over 8-bit channel
 */
void build_parity_table(void)
{
	unsigned i, j, k, c;
	for(i=0; i < 256; ++i) {
		ptable[i] = k = i;
		for(j=c=0; j < 8; ++j) {
			if(k & 1)
				++c;
			k >>= 1; }
		if(c & 1)
			ptable[i] |= 0x80; }
}

/*
 * Write a value to memory
 */
void write_memory(unsigned m, unsigned char *p, unsigned l, unsigned char flags)
{
	unsigned s, seg, off;

	if(m >= MAX_MEM)
		error("Bad memory location %u", m);
	if(MemSeg[m])
		error("Duplicate write to location");

	if(!l)
		return;

	if(m > MAX_DTZ)
		Budl = 255;

	// Find a segment with room to accept the value
	off = 65535-l;
	for(s=0; s < MAX_SEG; ++s) {
		if(!(seg = SegList[s])) {	// No memory allocated
			if(!(SegList[s] = seg = alloc_seg(4096)))
				break; }
		if(SegTop[s] < off)
			goto found_seg; }
	error("Out of memory");
found_seg:
	MemSeg[m] = (s+1) | flags | Pflags;
	MemAddr[m] = off = SegTop[s];
	MemSize[m] = l;
//printf("%u %04x:%04x %04x %u '", s, seg, off, p, l);
//fputs(p, stdout);
//printf("'\n");
	do {
		poke(seg, off++, *p++); }
	while(--l);
	SegTop[s] = off;
}

/*
 * Read a value from memory
 */
unsigned read_memory(unsigned m, unsigned char *p)
{
	unsigned seg, off, l;
	if(!(seg = MemSeg[m] & 15)) {
		return MemSize[m] = *p = 0; }
	seg = SegList[seg-1];
	off = MemAddr[m];
	l = MemSize[m];
//printf("%04x:%04x %u\n", seg, off, l);
	while(l) {
		*p++ = toupper(peek(seg, off++));
		--l; }
	*p = 0;
	return MemSize[m];
}


/*
 * ---- DTZ image building functions ----
 */

/*
 * Write a nibble to a DTZ image memory location
 */
void write_nibble(unsigned char n)
{
	if(Nflag) {
		poke(seg, image_size, peek(seg, image_size) | n);
		++image_size;
		Nflag = 0;
		return; }
	poke(seg, image_size, n << 4);
	Nflag = 255;
}

/*
 * Write to a DTZ image memory location
 */
void write_dtz_memory(unsigned location, unsigned char *data)
{
	unsigned p;
	unsigned char c;

	if(!*data)		// This location is empty - do not allocate
		return;

	if(location >= MAX_DTZ)
		error("Cannot create DTZ over location 999");

	pokew(seg, (location*2) + DTZ_ptr, image_size + DTZ_adjust);
	p = image_size++;		// Hold location
	Nptr = image_size;
	Nflag = 0;
	while(c = *data++) {
		if((c >= '0') && (c <= '9')) {
			write_nibble(c - '0');
			continue; }
		if(c > 0x5F)
			c -= 0x20;
		c += 0xA0;
		write_nibble(c >> 4);
		write_nibble(c & 15); }
	if(Nflag)
		write_nibble(15);
	Nflag = 0;
	poke(seg, p, image_size - p);
}

/*
 * Patch new DTZ password
 */
void copy_dtz_password(void)
{
	unsigned i;
	unsigned char *p;

	poke(seg, DTZ_PW, strlen(p=pass)+1);
	for(i=0; i < 10; ++i) {
		poke(seg, (DTZ_PW+1)+i, *p);
		if(*p) ++p; }
}

/*
 * Preset the DTZ image with fixed values
 */
void preset_dtz(void)
{
	unsigned i;
	unsigned char *p;

	if(!seg) {
		if(!(seg = alloc_seg(4096)))
			error("Out of memory"); }

	i = image_size = Dtz = 0;
	do {
		pokew(seg, i-=2, 0); }
	while(i);

	// Write terminal ID
	p = DTZ_id;
	do {
		poke(seg, i++, *p); }
	while(*p++);

	// Write fixed constants
	for(i=0; i < sizeof(DTZ_init); ++i)
		poke(seg, i+DTZ_FIXED, DTZ_init[i]);

	// Write location 2 & 3
	pokew(seg, DTZ_ptr+4, DTZ_l2);	// Location 2
	pokew(seg, DTZ_ptr+6, DTZ_l3);	// Location 3

	// Write fixed table
	for(i = 0x00A0; i < 0x00FA; i += 2)
		pokew(seg, i, 0x0001);

	if(pass)
		copy_dtz_password();

	image_size = DTZ_ptr + 2000;
}

/*
 * Compute and store DTZ data area checksum
 */
void checksum_dtz(void)
{
	unsigned i, c;


	// Compute and store data checksum
	pokew(seg, 0x0082, image_size + DTZ_adjust);	// Size
	c = 0;
	for(i = 0x0086; i < image_size; ++i)
		c += peek(seg, i);
	pokew(seg, 0x0080, c);					// Checksum
	Dtz = 255;
}

/*
 * Test the integrety of a freshly loaded DTZ file
 */
void verify_dtz(void)
{
	unsigned c, i, l;
	unsigned char f;
	f = 0;

	// Test for image signature
	for(i=0; i < 8; ++i)
		buffer[i] = peek(seg, i);
	buffer[8] = 0;
	if(strcmp(buffer, DTZ_id)) {
		estring("DTZ terminal ID does not match selected one", buffer);
		f = 255; }

	// Test image size
	i = peekw(seg, 0x0082) - DTZ_adjust;
	if(i != image_size) {
		printf("DTZ Image size incorrect: %04x %04x\n", i, image_size);
		f = 255; }
	if(i > DTZ_total) {
		printf("Data exceeds DTZ memory area\n");
		f = 255; }

	// Compute and store data checksum
	c = 0;
	for(i = 0x0086; i < image_size; ++i)
		c += peek(seg, i);
	i = peekw(seg, 0x0080);
	if(i != c) {
		printf("DTZ Checksum incorrect: %04x %04x\n", i, c);
		f = 255; }

	l = DTZ_ptr + 2000;
	for(i=0; i < MAX_DTZ; ++i) {
		if((i == 2) || (i == 3))
			continue;
		c = peekw(seg, (i*2)+DTZ_ptr) - DTZ_adjust;
		if((c < l) || (c & 0x8000)) {
			printf("DTZ Memory index table corrupt.\n");
			f = 255;
			break; } }

	if(f) {
		if(!yesno("Continue"))
			error("Exiting..."); }
	Dtz = 255;
}

/*
 * Analyze the DTZ image extracted from the terminal and
 * attempt to determine it's parameters.
 */
void analyze_dtz(unsigned blocks)
{
	unsigned i, j, c, DTZ_data;

//	DTZ_total = (blocks * 128)-1;
	DTZ_total = (blocks * 128) + 127;

	for(DTZ_ptr = 0xA0; (peekw(seg, DTZ_ptr) == 1) || (peekw(seg, DTZ_ptr) == 0x101); DTZ_ptr += 2);
	DTZ_data = DTZ_ptr + 2000;

	// Find lowest data address & compute actual image size
	DTZ_adjust = -1;
	for(i=0; i < MAX_DTZ; ++i) {
		if((i == 2) || (i == 3))
			continue;
		if(j = peekw(seg, (i*2)+DTZ_ptr)) {
			if(j < DTZ_adjust)
				DTZ_adjust = j; } }
	if(DTZ_adjust == -1)
		error("Image has no written locations - cannot analyze.");
	DTZ_adjust -= DTZ_data;
	image_size = peekw(seg, 0x82) - DTZ_adjust;

	c = 0;
	for(i=0x86; i < image_size; ++i)
		c += peek(seg, i);
	i = peekw(seg, 0x80);
	if(c == i)
		printf("DTZ description validated.\n");
	else {
		printf("Unable to validate DTZ description\n");
		printf("Check value: Computed=%04x, DTZ=%04x\n", c, i);
		if(!yesno("Continue"))
			error("Exiting..."); }

	DTZ_l2 = peekw(seg, DTZ_ptr+4);
	DTZ_l3 = peekw(seg, DTZ_ptr+6);

	for(i=0; i < sizeof(DTZ_id); ++i)
		DTZ_id[i] = peek(seg, i);

	for(i=0; i < sizeof(DTZ_init); ++i)
		DTZ_init[i] = peek(seg, i + 0x0084);

	printf("ID '%s'\n", DTZ_id);
	printf("DTZ_ptr = %04x\n", DTZ_ptr);
	printf("DTZ_adjust = %04x\n", DTZ_adjust);
	printf("DTZ_l2 = %04x\n", DTZ_l2);
	printf("DTZ_l3 = %04x\n", DTZ_l3);
	printf("DTZ_total = %u\n", DTZ_total);
}

/*
 * Read a nibble from compressed memory
 */
unsigned read_nibble(void)
{
	if(!Nflag) {
		Nflag = 255;
		return peek(seg, Nptr) >> 4; }
	Nflag = 0;
	return peek(seg, Nptr++) & 15;
}

/*
 * Decode compressed memory
 */
unsigned decode(unsigned char *buf)
{
	unsigned char n, c, l, *b;
	unsigned Nend;
	Nflag = 0;
	b = buf;
	if(l = peek(seg, Nptr++)) {
		Nend = (Nptr + l) - 1;
		while(Nptr < Nend) {
			if((n = read_nibble()) < 10) {
				*b++ = n + '0';
				continue; }
			if(Nptr == Nend)
				break;
			c = (n << 4) | read_nibble();
			*b++ = c - 0xA0; } }
	*b = 0;
	return b - buf;
}

/*
 * Encode memory image into DTZ
 */
void mem_to_dtz()
{
	unsigned i;
	if(!Dtz) {
		if(!Mem)
			error("No input");
		preset_dtz();
		for(i=0; i < MAX_DTZ; ++i) {
			if(read_memory(i, buffer))
				write_dtz_memory(i, buffer); }
		while(i < MAX_MEM) {
			if(MemSeg[i++]) {
				printf("Image contains locations above 999 - will be lost in DTZ\n");
				if(yesno("Continue"))
					break;
				error("Exiting..."); } }
		// Predefine default locations (if not already set)
		if(!peekw(seg, (4+4)+DTZ_ptr))
			write_dtz_memory(4, "000000");
		if(!peekw(seg, (5+5)+DTZ_ptr))
			write_dtz_memory(5, "0000");
		if(!peek(seg, (18+18)+DTZ_ptr))
			write_dtz_memory(18, "0000000000000000");
		checksum_dtz(); }
	verify_dtz();
}

/*
 * Decode memory image from DTZ
 */
void dtz_to_mem()
{
	unsigned i,l;
	if(!Mem) {
		if(!Dtz)
			error("Input required");
		for(i=0; i < MAX_DTZ; ++i) {
			if(!(Nptr = peekw(seg, (i+i)+DTZ_ptr)))
				continue;
			Nptr -= DTZ_adjust;
			if(Nptr & 0x8000) continue;
			l = decode(buffer);
			write_memory(i, buffer, l, 0); }
		verify_dtz(); }
}

/*
 * Load a UDL file into memory
 */
void load_udl(unsigned char *file)
{
	unsigned char *p, *p1, pf;
	unsigned line, m;

	line = 0;
	fp = fopen(file, "rvq");
	while(fgets(p = buffer, sizeof(buffer)-1, fp)) {
		++line;
		while(isspace(*p)) ++p;
		if(*p++ != '"')
			continue;
		m = 0;
		switch(pf = *p++) {	// Skip Full/Partial indicator
		default:
		badudl: error("%u: Bad UDL format\n", line);
		case '1' : pf = 0x80;
		case '0' : }
		if(*p == ',') { // Large format
			while(isdigit(*++p))
				buffer[m++] = *p;
			if(*p++ != ',') goto badudl; }
		else {
			do {
				if(!isdigit(*p))
					goto badudl;
				buffer[m++] = *p++; }
			while(m < 3); }
		buffer[m] = 0;
		m = atoi(buffer);
		if(m >= MAX_MEM)
			error("%u: location %u out of range (0000-1999)\n", line, m);
		p1 = p;
		while(*p1) ++p1;
		while(isspace(*--p1));
		if(*p1 != '"') {
			fclose(fp);
			error("%u: Bad UDL format\n", line); }
		*p1 = 0;
		write_memory(m, p, p1-p, pf);
	} fclose(fp);
	Mem = 255;
}

/*
 * Load a TCL file into memory
 */
void load_tcl(unsigned char *file)
{
	unsigned locn, count, lnum;
	char line[150], *ptr, *ptr1, *lptr, c, pf1, pf2;
	char sflag, qflag;
	FILE *fp;

	fp = fopen(file, "rvq");
	locn = -1;
	lnum = pf2 = 0;
read_next:
	while(fgets(ptr = buffer, sizeof(buffer)-1, fp)) {
		++lnum;
		pf1 = 0;
		while(isspace(*ptr)) ++ptr;
		switch(*ptr) {
		case ';':
		case '%':	continue;
		case '&':	pf1 = 0x80; ++ptr; }
		if(isdigit(*(ptr1 = ptr))) {
			while(isdigit(*++ptr));
			switch(c = *ptr) {
				case '$' : sflag = 0; goto doit;
				case '=' : sflag = -1;
				doit:
					if(locn != -1) {
						if(count > 120)
							error("%u: location %u >120 characters.\n", lnum, locn);
						while((lptr > line) && isspace(*(lptr-1)))
							--lptr;
						*lptr = 0;
						write_memory(locn, line, lptr-line, pf2); }
					if(!c) {
						fclose(fp);
						Mem = 255;
						return; }
					qflag = count = 0;
					locn = atoi(ptr1);
					pf2 = pf1;
					if(locn >= MAX_MEM)
						printf("%u: location %u out of range (000-1999)\n", lnum, locn);
					lptr = line;
					++ptr; } }
		while(c = *ptr++) {
			switch(c) {
			case ' ' :
			case '\t':
				if(sflag || qflag)
					break;
				continue;
			case '\'':
				qflag = qflag ? 0 : -1;
				break;
			case ';' :
				if(!qflag)
					goto read_next; }
			if(++count <= 120)
				*lptr++ = c; } }

	c = 0;
	goto doit;
}

/*
 * Write memory content as UDL data
 */
void write_udl(unsigned char *file)
{
	unsigned i;
	fp = fopen(file, "wvq");
	for(i=0; i < MAX_MEM; ++i) {
		if(!read_memory(i, buffer))
			continue;
		fprintf(fp, Budl ? "\"%c,%04u," : "\"%c%03u", (MemSeg[i] & 0x80) ? '1' : '0', i);
		fputs(buffer, fp);
		fputs("\"\n", fp); }
	fclose(fp);
}

/*
 * Write memory content as TCL data
 */
void write_tcl(unsigned char *file)
{
	unsigned i;

	fp = fopen(file, "wvq");
	for(i=0; i < MAX_MEM; ++i) {
		if(!read_memory(i, buffer))
			continue;
		if(MemSeg[i] & 0x80) putc('&', fp);
		fprintf(fp, "%u=", i);
		fputs(buffer, fp);
		putc('\n', fp); }
	fclose(fp);
}

/*
 * ---- Communication functions ----
 */

/*
 * Get a character from the serial stream with timeout
 *
 * Input: Time to wait in 1/18 second intervals.
 * Return: Character read (0-127) or -1 for timeout.
 */
int Cgett(unsigned timeout)
{
	unsigned t;
	int c;

	if(prc) {
		c = prc;
		prc = 0;
		return c; }
	/* If a char is pending, report it */
	t = TICK;
	do {
		if((c = Ctestc()) != -1)
			return c & rmask;
		switch(kbtst()) {
		case 0x1B :
		case 0x03 : error("Exiting..."); } }
	while((TICK - t) < timeout);
	return -1;
}

/*
 * Flush the input stream ... wait for 1/2 second of no data
 */
void Cflush(void)
{
	while(Cgett(Timeout/2) != -1);
}

/*
 * Write a character with parity
 */
void Pputc(unsigned char c)
{
	Cputc(ptable[c]);
}

/*
 * Lookup ID in database
 */
scan_database(unsigned char *id, unsigned char f)
{
	unsigned s;
	unsigned char buf[DSIZE];
	if(strbeg("UDL", id)) {
		Udl = Budl = 255;
		return 255; }
	strcpy(pptr, "DAT");
	if(fp = fopen(path, "rvb")) {
		while((s = fget(buf, DSIZE, fp)) == DSIZE) {
			if(!strcmp(buf, id)) {
				fclose(fp);
				memcpy(DTZ_id, buf, DSIZE);
				return 255; } }
		fclose(fp);
		if(s)
			error("Database is corrupt"); }

	printf("Terminal ID '%s' not found in database.\n", id);
	if(f) {
		if(!yesno("Continue"))
			error("Exiting..."); }

	return 0;
}

/*
 * Display the content of the terminal ID database
 */
void list_database(void)
{
	unsigned c, s;
	unsigned char dtz_id[DSIZE];
	strcpy(pptr, "INI");
	if(fp = fopen(path, "r")) {
		fgets(buffer, sizeof(buffer)-1, fp);
		fclose(fp); }
	if(verbose) printf("%-5u", 1);
	printf("%-12s%s\n", "UDL", strbeg("UDL", buffer) ? "  (Selected)" : "");
	strcpy(pptr, "DAT");
	fp = fopen(path, "rvqb");
	c = 1;
	while((s = fget(dtz_id, DSIZE, fp)) == DSIZE) {
		if(verbose)	printf("%-5u", ++c);
		printf("%-12s%s\n", dtz_id, strcmp(dtz_id, buffer) ? "" : "  (Selected)"); }
	fclose(fp);
	if(s)
		printf("Database file is corrupt!");
}

/*
 * Delete an entry from the terminal ID database
 */
void delete_database(unsigned char *id)
{
	unsigned s, o, c;
	unsigned char f, dtz_id[DSIZE];
	strcpy(pptr, "DAT");
	f = o = 0;
	fp = fopen(path, "rvqb");
	while((s = fget(dtz_id, DSIZE, fp)) == DSIZE) {
		if(!strcmp(dtz_id, id)) {
			f = 255;
			continue; }
		for(c=0; c < DSIZE; ++c)
			poke(seg, o++, dtz_id[c]); }
	fclose(fp);
	if(!f) {
		printf("ID '%s' does not exist in database.\n", id);
		return; }
	fp = fopen(path, "wvqb");
	for(c=0; c < o; ++c)
		putc(peek(seg, c), fp);
	fclose(fp);
	if(verbose)
		printf("ID '%s' has been deleted.\n", id);
}

/*
 * Write a new database entry
 */
void write_database(void)
{
	strcpy(pptr, "DAT");
	fp = fopen(path, "wvqba");
	fput(DTZ_id, DSIZE, fp);
	fclose(fp);
	if(verbose)
		printf("Database updated.\n");
}

/*
 * Receive packet from host
 * 	t =	1xxxxxxx	= Send NAK instead of ENQ
 *		x1xxxxxx	= Accept ETX
 *		xx1xxxxx	= Accept ETX+80
 *		xxx1xxxx	= Perform LRC test
 *		xxxx1xxx	= Perform CRC test
 */
#define	RP_ENQ		0x80		// Send NAK on timeout
#define	RP_ETX1		0x40		// Accept ETX 0x03
#define	RP_ETX2		0x20		// Accept ETX 0x83
#define	RP_LRC		0x10		// Check with LRC
#define	RP_CRC		0x08		// Check with CRC
#define	RP_BIN		0x04		// Accept binary
receive_packet(unsigned char t)
{
	unsigned c, l, r;
	unsigned char lrc;

	rmask = (t & RP_BIN) ? 0xFF : 0x7F;

	// Wait for STX
	r = RETRY+1;
rx1: switch(Cgett(Timeout)) {
	default: switch(r) {
		case RETRY+1: if(t & RP_ENQ) printf("Waiting for terminal.\n");
		default:		if(!--r) goto fail;
		case RETRY:		Cputc((t & RP_ENQ) ? ENQ : NAK);	goto rx1; }
	case EOT:		return 0;
	case STX: }

	// Collect data until ETX
	crc = lrc = l = 0;
rx2: if((c = Cgett(Timeout)) == -1) {
		if(--r) {
nak:		Cputc(NAK);
			goto rx1; }
	fail:	error("Terminal request failed"); }
	lrc ^= c;
	addcrc(c);
	if(l < sizeof(buffer))
		buffer[l++] = c;
	switch(c) {
	case ETX: if(t & RP_ETX1) break;
		goto rx2;
	case ETX|0x80: if(t & RP_ETX2) break;
	default: goto rx2; }

	// Validate check sequence
	buffer[--l] = 0;
	if(t & RP_LRC) {
		if(Cgett(Timeout) != lrc)
			goto nak;
		return l; }
	if(t & RP_CRC) {
		rmask = 0xFF;
		c = Cgett(Timeout) << 8;
		c = Cgett(Timeout) | c;
		if(c != crc)
			goto nak;
		return l; }

	// Unknown packet type - identify
	if(strbeg(buffer, "VFI")) {
		rmask = 0xFF;
		c = Cgett(Timeout) << 8;
		c = Cgett(Timeout) | c;
		if(c != crc)
			goto nak;
		return 254; }

	if(Cgett(Timeout) != lrc)
		goto nak;
	return 253;
}

static char fmsg[] = { "Terminal did not accept request.\n\
Possible DTZ<>UDL mismatch or ID mismatch (use -S to select).\n"};

send_packet(unsigned char packet[], unsigned char u, unsigned char etx)
{
	unsigned l, c, lrc, r;
	unsigned char f;
	r = RETRY;
	f = 0;
rtx: lrc = crc = l = 0;
	if(u&15) Cputc(STX); else Pputc(STX);
	while(c = packet[l++]) {
		if(u&15) Cputc(c); else Pputc(c);
		lrc ^= c;
		addcrc(c); }
	if(u&15) Cputc(etx); else Pputc(etx);
	lrc ^= etx;
	addcrc(etx);
	if(u&15) {
		Cputc(crc >> 8);
		Cputc(crc); }
	else
		Pputc(lrc);

	for(;;)	switch(c = Cgett(Timeout)) {
		case NAK : f = 255;
		case -1 :
			if(--r) goto rtx;
			error((u & 0x80) ? fmsg : "communications error");
		case STX: prc = STX;
		case ACK: return; }
}

send_dtz_record(unsigned offset)
{
	unsigned c, l, r;
	r = RETRY;
tx:	Cputc(STX);
	crc = 0;
	for(l=0; l < 128; ++l) {
		Cputc(c = peek(seg, offset+l));
		addcrc(c); }
	Cputc(crc >> 8);
	Cputc(crc);
	for(;;) switch(Cgett(Timeout)) {
	case NAK :
	case -1 : if(--r) goto tx;
		error("Send packet failed");
	case ACK: return; }
}

/*
 * Receive record (STX ... CRC1 CRC2)
 * - returns 0 if EOT received, indicating end of data
 */
int receive_dtz_record(unsigned offset)
{
	unsigned c, l, r;
	r = RETRY;

rx1: rmask = 0x7F;
	c = Cgett(Timeout);
	if(c == EOT)
		return 0;
	if(c != STX) {
fail:	if(--r) {
			goto rx1; }
		error("Failed to receive packet"); }
	crc = 0;
	rmask = 0xFF;		// Binary input
	for(l=0; l < 128; ++l) {
		c = Cgett(Timeout);
		if(c == -1) goto fail;
		poke(seg, offset+l, c);
		addcrc(c); }
	c = Cgett(Timeout) << 8;
	c = Cgett(Timeout) | c;
	rmask = 0x7F;		// ASCII input
	if(c != crc) goto fail;
	Cputc(ACK);
	return -1;
}

open_com()
{
	if(!Cport)
		error("You must specify a COM port");
	if(Copen(Cport, _19200, DATA_8|PAR_NO|STOP_1, SET_RTS|SET_DTR|OUTPUT_2))
		error("Cannot open COM port");
	Cflags |= TRANSPARENT;
}

/*
 * Display terminal type
 */
show_term(unsigned char *uid, unsigned char *did)
{
	printf("Terminal is TRANZ-");
	if(Udl) {
		printf("UDL");
		if(uid)
			printf(" : %s", uid); }
	else {
		printf("DTZ");
		if(did)
			printf(" : %s", did); }
	putc('\n', stdout);
}

/*
 * Perform an download to the terminal
 */
download()
{
	unsigned i, l;

	open_com();
	build_parity_table();
	printf("Please execute FUNC-%c on terminal or press ESC to exit.\n", '#');
	l = 0;
	switch(receive_packet(RP_ENQ|RP_ETX1|RP_ETX2)) {
	case 253 :
		Udl = 0; show_term(buffer, buffer);
		scan_database(buffer, 255);
		if(id && strcmp(id, buffer)) error("ID mismatch");
		Dtz = 0;
		mem_to_dtz();
		printf("0     blocks sent.");
		for(i=0x80; i < image_size; i += 128) {
			printf("\r%u", ++l);
			send_dtz_record(i); }
		break;
	case 254 :
		Udl = 255; show_term(buffer, buffer);
		send_packet("MDOWNLOADING", 15, ETX|0x80);
		printf("0     blocks sent.");
		for(i=l=0; i < MAX_MEM; ++i) {
			sprintf(buffer,"L%04u\x1C", i);
			if(!read_memory(i, buffer+6))
				continue;
			printf("\r%u", ++l);
			send_packet(buffer, 15, ETX|0x80); }
		if(pass) {
			sprintf(buffer, "P%s", pass);
			send_packet(buffer, 15, ETX|0x80); }
		send_packet("MDOWNLOAD DONE", 15, ETX|0x80);
		send_packet("S", 255, ETX|0x80); }
	Cputc(EOT);
	printf("\n");
	Cflush();
	Cclose();
}

/*
 * Perform an upload from the terminal
 */
upload(unsigned char *id, unsigned char finish)
{
	unsigned c, o;
	unsigned char *p;
	build_parity_table();
	if(finish) show_term(0, id);
	open_com();
	printf("Please execute FUNC-%c on terminal or press ESC to exit.\n", '*');
	p = -1;
	while((c=Cgett(Timeout)) != ENQ) {
		if(c == o) {
			printf("Waiting for terminal.\n");
			o = -2; } }
	o = c = 0;
	if(Udl) {
		send_packet("VFI,,XDL,F,,,,", 255, ETX);
		printf("\r0     blocks received.");
		while(receive_packet(RP_ETX2|RP_CRC|RP_BIN)) {
			switch(buffer[0]) {
			case 'P' :
				if(Pass)
					printf("\nUDL password        : %s", buffer+1);
				break;
			case 'L' :
				printf("\r%u", ++c);
				p = buffer+1;
				while(isdigit(*p)) ++p;
				if(*p++ != FS) error("Bad upload format");
				switch(o = atoi(buffer+1)) {
				default: write_memory(o, p, strlen(p), 0);
				case 2 :
				case 3 : } }
			Cputc(ACK); }
		printf("\n");
		Cclose();
		Mem = 255;
		return c; }

	preset_dtz();
	send_packet(id, 0x80, ETX);
	printf("\r0     blocks received.");
	while(receive_dtz_record(o += 0x80))
		printf("\r%u", ++c);
	printf("\n");
	Cclose();
	for(o=0; o < 128; ++o)
		poke(seg, o, *id ? *id++ : 0);
	if(finish) {
		image_size = peekw(seg, 0x0082) - DTZ_adjust;
		verify_dtz();
		Dtz = 255;
		dtz_to_mem(); }
	return c;
}


/*
 * Parse a filename and return the TCL file type
 */
unsigned getfile(unsigned char *f, unsigned char *buffer)
{
	unsigned i;
	unsigned char c, *p, *p1;
	static unsigned char *file_types[] = { "TCL", "UDL", "DTZ", 0 };

	p1 = buffer;
	p = "";
	while(*p1++ = c = toupper(*f++)) {
		if(c == '.')
			p = p1; }
	*p1 = 0;
	for(i=0; p1 = file_types[i]; ++i) {
		if(!strcmp(p, p1))
			return i; }
	abort("File type must be 'DTZ', 'UDL' or 'TCL'\n");
}

/*
 * Select active terminal id
 */
select_id(unsigned char *id)
{
	strcpy(pptr, "INI");
	fp = fopen(path, "wvq");
	fputs(strbeg("UDL", id) ? "UDL" : id, fp);
	putc('\n', fp);
	fclose(fp);
	if(verbose)
		printf("ID '%s' selected.\n", id);
}

/*
 * Select a terminal database entry
 */
void select(void)
{
	unsigned char f;
	static char select_text[] = { "\n\
TCLL will now analyze an upload from the terminal to determine the\n\
structure of it's DTZ memory image.\n\n\
This works best if the terminal is BLANK, with no memory locations\n\
loaded - see help topics for information on how to clear the terminal.\n\n\
NOTE: If you see an 'invalid message size' message on the terminal,\n\
this is normal - a result of terminating a download transaction.\n\n" };

	build_parity_table();

	if(!(f = *buffer)) {
		if(verbose)
			printf("No ID specified.\n");
xfail:
		if(!Cport)
			error("Specify COM port (-n) to interrogate terminal.");
		open_com();
		// Get terminal ID from download request
		printf("Please execute FUNC-%c on terminal or press ESC to exit.\n", '#');
		receive_packet(RP_ENQ|RP_ETX1|RP_LRC);
		Cputc(EOT);
		Cflush();
		Cclose(); }

	if(scan_database(buffer, 0)) {
		if(verbose)
			printf("ID '%s' exists in database.\n", buffer);
		if(yesno("Select this ID"))
			select_id(buffer);
		return; }
	else {
		if(f) {
			f = 0;
			goto xfail; } }

	if(verbose)
		fputs(select_text, stdout);

	analyze_dtz(upload(buffer, 0));

	if(yesno("Add to database")) {
		write_database();
		if(yesno("Select this ID"))
			select_id(DTZ_id); }
}

/*
 * Get a byte from the code block text
 */
unsigned help_ptr;
char get_text() asm
{
	MOV	BX,DGRP:_help_ptr			; Get offset
	INC WORD PTR DGRP:_help_ptr		; Increment
	MOV	AL,CS:byte ptr HELLO[BX]	; Get byte
}

/*
 * Display a help topic
 */
void help(unsigned topic)
{
	unsigned char c;
	static unsigned output, tab = 8;

	if(topic != 99) {
		while(topic--) {
			while(get_text());
			if(!(tab = get_text()))
				abort("Help block not available\n"); } }
show_next:
	while(c = get_text()) {
		if(c == '\t') {
			do
				putc(' ', stdout);
			while(++output % tab); }
		else {
			output = (c != '\n') && output+1;
			putc(c, stdout); } }
	if((topic == 99) && (tab = get_text())) {
		putc('\n', stdout);
		goto show_next; }
}

#define	F_LIST		0x01
#define	F_SELECT	0x02
#define	F_DELETE	0x04
#define	F_FILE		0x08
#define	F_EXTRACT	0x10


main(int argc, char *argv[])
{
	unsigned i, l, s[2], s1[2];
	unsigned char *p;

	// Process command line options
	l = 0;
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++)<<8) | toupper(*ptr++)) {
		case '/1' :
		case '-1' : Cport = 1;						continue;
		case '/2' :
		case '-2' : Cport = 2;						continue;
		case '/3' :
		case '-3' : Cport = 3;						continue;
		case '/4' :
		case '-4' : Cport = 4;						continue;
		case '/A' :
		case '-A' : Pflags = 0x80;					continue;
		case '/D' :		// Delete database entry
		case '-D' :	function |= F_DELETE;			continue;
		case '/E' :		// Extract image from terminal
		case '-E' : function |= F_EXTRACT;			continue;
		case '/H' :		// Help request
		case '-H' : help(*ptr ? atoi(ptr) : 1);		return;
		case '/L' :		// List database
		case '-L' : function |= F_LIST;				continue;
		case '/N' :		// Auto-NO to prompts
		case '-N' : Yes = 15;						continue;
		case '/P' :		// Show DTZ password
		case '-P' : Pass = 255;						continue;
		case '/Q' :		// Quiet
		case '-Q' : verbose = 0;					continue;
		case '/S' :		// Select terminal
		case '-S' : function |= F_SELECT;			continue;
		case '/Y' :
		case '-Y' : Yes = 255;						continue;
		case 'I=' : strupr(id = ptr);				continue;
		case 'P=' : strupr(pass = ptr);				continue;
		case 'T=' : Timeout = atoi(ptr)/55;			continue;
		case 'W=' : wtype = getfile(ptr, wfile);	continue; }
		if(function & F_FILE)
			error("Unknown option: %s\n", ptr-2);
		strcpy(buffer, ptr-2);
		strupr(buffer);
		l = i;
		function |= F_FILE; }

	if(!function) {
		help(0);
		return; }

	while((i = get_text()) >= ' ')
		putc(i, stdout);
	printf(" - ?COPY.TXT 2000-2006 Dave Dunfield -  -- see COPY.TXT --.\n");

	IOB_size = 2048;
	ptr = argv[0];
	p = pptr = path;
	while(*p++ = i = *ptr++) {
		if(i == '.')
			pptr = p; }

	switch(function) {
	default: error("Incompatible options");
	case F_LIST:			list_database();			return;
	case F_DELETE:			error("You must specify a terminal ID code.");
	case F_DELETE|F_FILE:	delete_database(buffer);	return;
	case F_SELECT:
	case F_SELECT|F_FILE:	select();					return;
	case F_EXTRACT:			type = 3;					break;
	case F_FILE:
		type = getfile(argv[l], buffer); }

	if(id)
		scan_database(id, 255);
	else {
		strcpy(pptr, "INI");
		if(fp = fopen(path, "r")) {
			fgets(buffer+240, sizeof(buffer)-1, fp);
			fclose(fp);
			scan_database(buffer+240, 255); } }

	switch(type) {
	case 0 :	// TCL
		load_tcl(buffer);
		break;
	case 1 :	// UDL
		load_udl(buffer);
		break;
	case 2 :	// DTZ
		preset_dtz();
		image_size = 0;
		fp = fopen(buffer, "rvqb");
		while((i = getc(fp)) != EOF)
			poke(seg, image_size++, i);
		fclose(fp);
		verify_dtz();
		if(pass) {
			copy_dtz_password();
			checksum_dtz(); }
		dtz_to_mem();
		break;
	case 3 :	// Terminal upload
		upload(DTZ_id, 255);
		Cport = 0; }

	if(wtype >= 2) mem_to_dtz();

	if(verbose) {
		if(Dtz)	{
			estring("DTZ terminal ID     :", DTZ_id);
			if(Pass) {
				l = 0;
				if(i = peek(seg, DTZ_PW)) {
					while(--i)
						buffer[l] = peek(seg, DTZ_PW + ++l); }
				buffer[l] = 0;
				estring("DTZ password        :", buffer); }
			printf ("DTZ image size      : %-5u (%u blocks)\n", image_size-0x80, (image_size-1)/128);
			printf ("DTZ checksum        : %04x\n", peekw(seg, DTZ_CSUM));
			l = peekw(seg, DTZ_FREE) - DTZ_adjust;
			printf ("DTZ memory used/free: %u / %u\n", l, DTZ_total-l); }
		longset(s, 0);
		longset(s1, 0);
		for(i=l=0; i < MAX_MEM; ++i) {
			*s1 = MemSize[i];
			if(MemSeg[i] && *s1) {
				++l;
				longadd(s, s1); } }
		ltoa(s, buffer, 10);
		printf("Locations loaded    : %u\n", l);
		printf("Total image size    : %s\n", buffer); }

	if(Cport)
		download();

	if(*wfile) switch(wtype) {
		case 0 : write_tcl(wfile);	break;
		case 1 : write_udl(wfile);	break;
		case 2 :
			fp = fopen(wfile, "wvqb");
			for(i=0; i < image_size; ++i)
				putc(peek(seg, i), fp);
			fclose(fp); }
}

/*
 * Help text - stored in code segment to avoid using precious data space
 */
asm {
HELLO:
DB'TCL Loader 2.2'
DB 10,10
DB'use:	TCLL <input_file.TCL/UDL/DTZ> [options]',10
DB'			- or -',10
DB'	TCLL [database ID|UDL] [options]',10,10
DB'opts:	-n		= load terminal via COMn (1-4)',10
DB'	-A		= mark All .UDL locations for parameter download',10
DB'	-D		= Delete id code from terminal database',10
DB'	-E		= Extract image from terminal',10
DB'	-H[n]		= display TCLL Help topics',10
DB'	I=id|UDL	= override currently selected terminal ID code',10
DB'	-L		= List id codes in terminal database',10
DB'	-N		= auto-No to prompts',10
DB'	-P		= display Password in DTZ summary',10
DB'	P=string	= set upload/.DTZ Password to "string"',10
DB'	-Q		= Quiet - reduce console output',10
DB'	-S		= Select id code from terminal database',10
DB'	T=n		= set serial Timeout (milliseconds)	[1000]',10
DB'	-Y		= auto-Yes to prompts',10
DB'	W=file		= Write image to file (.TCL/UDL/DTZ)',10
DB 10,'?COPY.TXT 2001-2006 Dave Dunfield',10
DB' -- see COPY.TXT --.',10,0
; 1 - Help topics
DB	8
DB'TCLL HELP Topics:',10,10
DB'	-H0	- TCLL command syntax',10
DB'	-H1	- TCLL help topics',10
DB'	-H2	- TCLL overview',10
DB'	-H3	- File formats',10
DB'	-H4	- Direct load information',10
DB'	-H5	- TRANZ <> PC download cable pinout',10
DB'	-H6	- TCLL terminal information database',10
DB'	-H7	- Clearing a TRANZ terminal',10
DB'	-H8	- Contact information',10,10
DB'	-H99	- Display all topics',10,10
DB'To obtain help output in a file, use the DOS redirection operator:',10,10
DB'	eg: TCLL -H2 >OVERVIEW.TXT',10,0
; 2 - Overview
DB	8
DB'TCLL Overview:',10,10
DB'TCLL is a utility which loads any TRANZ application file (.TCL, .UDL',10
DB'or .DTZ) into the terminal using a direct PC<>Terminal connection. This',10
DB'is the fastest way to load an application into the terminal.',10,10
DB'TCLL can also be used to convert any of the TRANZ file types to a file',10
DB'of a different type, and to extract the files from the terminal via direct',10
DB'PC<>Terminal upload.',10,10
DB'TCLL uses a Terminal Information Database when building the memory image',10
DB'for direct download or .DTZ file (see -H6).',10,10
DB'TCLL is designed for use only with Verifone TRANZ terminals.',10,0
; 3 - file formats
DB	8
DB'TCLL File Formats:',10,10
DB'There are three application file formats commonly used with the TRANZ,',10
DB'each processed by different Verifone utility programs:',10,10
DB'  Type	Description				Verifone Utility',10
DB'  ----	-----------				----------------',10
DB'  TCL	Terminal Control Language source file	TCLOAD',10
DB'  UDL	ZonTalk format download file		ZONTALK',10
DB'  DTZ	TRANZ memory image			VLOAD',10,10
DB'TCLL can read and write ALL of the above format types. This means that',10
DB'you can load a .TCL, .DTZ or .UDL file directly into the terminal with',10
DB'this single program. You can also use TCLL to convert files from any of',10
DB'these types to any other type.',10,10
DB'NOTE: when writing .TCL files, TCLL will generate a file of simple data (=)',10
DB'statements only - TCLL does not try to regenerate ($) code records for code',10
DB'blocks as this information is lost in .UDL or .DTZ files.  If you wish to',10
DB'generate complete TCL source code, convert the file to a .UDL file, and then',10
DB'use the TCLD (TCL Decompiler) to build commented TCL source code from that.',10,0
; 4 - Direct load information
DB	8
DB'TCLL Direct Load:',10,10
DB'To perform a direct load, you must connect the TRANZ terminal to the',10
DB'PC using a direct PC<>terminal download cable. Initiate the download on',10
DB'the terminal by pressing: FUNCTION then #',10,10
DB'On the PC, run TCLL with the name of the file you wish to load, and a',10
DB'-n option to select the COM port.',10,10
DB'		eg: TCLOAD myfile.tcl -2',10,10
DB'	(Load MYFILE.TCL into terminal connected to COM2)',10,0
; 5 - Serial cable
DB	8
DB'TRANZ Serial cable:',10,10
DB'   Terminal			PC: 9-pin serial',10
DB' -------------------------------------------------',10
DB'  2-o     o',10
DB' 5-o   o   o-1,6,8		5   4   3   2   1',10
DB'  4-o     o-3			  9   8   7   6',10
DB'     7-o',10
DB' -------------------------------------------------',10
DB' 	1-DCD	3-TXD	5-GND	7-RTS	9-RI',10
DB' 	2-RXD	4-DTR	6-DSR	8-CTS',10,10
DB'     (Connectors viewed looking at end of cable)',10,0
; 6 - Terminal database
DB	8
DB'TCLL Terminal Information Database:',10,10
DB'Direct loading a TRANZ requires a complete terminal memory image to be built',10
DB'on the PC. The image layout differs with TRANS firmware versions. An ID code',10
DB'which identifies the image layout is stored in the DTZ (memory image) file',10
DB'and sent by the terminal when requesting a download. Verifone TCLOAD and VLOAD',10
DB'check that the two IDs match before allowing the download.',10,10
DB'TCLOAD uses a database to relate the firmware version number on the powerup',10
DB'screen to the ID code and exact layout required for the memory image. This is',10
DB'why you must use the firmware version (%) command in TCL files processed by',10
DB'TCLOAD and also why you cannot direct load a .UDL file (no version info.)',10,10
DB'TCLL however does not have reliable access to this proprietary information',10
DB'for all TRANZ versions. TCLL uses the ID code transmitted by the terminal and',10
DB'stored in .DTZ files to identify the memory image layout. ("%" in TCL files is',10
DB'ignored). The ID can be selected with the -S command option: TCLL -S IF@@OOOO',10,10
DB'TCLL maintains a small database of ID codes and image layout information in',10
DB'the file TCLL.DAT. When a new ID code is encountered that is not already in',10
DB'the database, TCLL will analyze an upload image from the terminal to determine',10
DB'the memory layout and add an appropriate record to the terminal database.',10
DB'The currently selected ID code is saved in the file TCLL.INI.',10,0
; 7 - Clearing the terminal
DB	8
DB'Clearing the terminal:',10,10
DB'When building a new terminal database entry, TCLL is most reliably able',10
DB'to analyze the terminal memory image when the terminal is completely empty',10
DB'(no memory loaded).',10,10
DB'To clear memory press *+CLEAR (If the terminal is running an application you',10
DB'may have to press CLEAR first, and then *+CLEAR within one second). At the',10
DB'ENTER PASSWORD prompt press 8 ALPHA 0 ALPHA 8 ALPHA 5 3 6 1 0 4 1 ALPHA ENTER',10
DB'The terminal should display SUCCESSFUL and memory is clear.',10,10
DB'An alternate way is with the command: TCLL NUL.TCL -n (n=com-port)',10
DB'Answer "Y" if you are prompted to "Continue (Y/N)?"',10,10
DB'If the terminal structure happens to match the ID you currently have',10
DB'selected the download may be successful and your terminal will be empty.',10,10
DB'If the terminal displays "PROGRAMMING ERROR", press FUNC+1 to clear the',10
DB'terminal memory - the terminal will now be empty!',10,0
; 8 - Contact information
DB	8
DB 'TCLL was created by:',10,10
DB'	Dunfield Development Services',10
DB'	115 Manion Heights Cres.',10
DB'	RR#2 Carp, Ontario Canada',10
DB'	K0A 1L0',10,10
DB' Specializing in low cost embedded software development tools',10
DB' and software/firmware development services!',10,10
DB' http://www.dunfield.com',10,10
DB' FAX:	613-256-5821',10,10
DB'This program was compiled with DDS Micro-C/PC.',10
DB	0,0
}
