/*
 * TCL De-compiler
 *
 * ?COPY.TXT 2001-2006 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile with DDS Micro-C/PC:
 *	makedb TCLDEF.DAT >R:\TCLDEF.H
 *	makedb XMAS.UDL >R:\XMAS.HU
 *	makedb XMAS.SYM >R:\XMAS.HS
 *	cc TCLD -fop
 */
// #define	DEBUG	1
#include <stdio.h>
#include <file.h>
#include <debug.h>

// General parameters
#define	N_CMD	78		// Maximum number of TCL commands
#define	N_MEM	2000	// Maximum number of memory locations

// Memory block markers
#define	M_DATA	0x01	// Memory is a data element
#define	M_USED	0x02	// Memory location was used
#define	M_CMT	0x04	// Memory location has comment

// Parameter types from source file
#define	P_NONE	0		// No parameter given
#define	P_NUM	1		// Numeric parameter
#define	P_VAR	2		// Variable parameter
#define	P_STR	13		// String parameter

// Parameter types from TCL definition file
#define	T_SKIP	0x80	// Skip count
#define	T_VAR	0x81	// Variable
#define	T_CONST	0x82	// Numeric constant
#define	T_BUF	0x83	// Buffer
#define	T_MEM	0x84	// Memory location
#define	T_STRC	0x85	// String/Constant
#define	T_STR	0x86	// String only
#define	T_REQD	0x87	// Offset for required constants

unsigned
	line,				// Current line number
	location = 100,		// Current memory location
	mem_name[N_MEM],	// Memory name pointers
	mem_lbls[N_MEM],	// Memory label name pointers
	mem_cmts[N_MEM],	// Memory comment blocks
	mem_lcms[N_MEM],	// Memory line comments
	data_seg,			// General data segment
	cmt_seg,			// Comment storage segment
	data_top,			// Point to free data memory
	cmt_top,			// Point to free comment memory
	tcl_cmds[N_CMD],	// TCL command definitions
	parm_value[10],		// Parsed parameter values
	tcl_count,			// TCL defined count
	parm_count,			// Parsed count
	out_cmds[128],		// Output commands
	out_cmts[128],		// Output comments
	out_count,			// Output counter
	comment;			// TCL comment

unsigned char
	mem_flags[N_MEM],	// Memory control flags
	mem_lcc[N_MEM],		// Memory line comment count
	out_lbls[128],		// Output labels
	*out_parm[128],		// Output parameter pointers
	out_buf[256],		// Output buffer
	*out_ptr,			// Output pointer
	tcl_parms[10],		// TCL defined parameters
	parm_type[10],		// Parameter type
	statpre[2],			// Statistics prefix
	buffer[256],		// General buffer
	temp[256];			// Temp buffer

FILE
	*fp,				// General file pointer
	*out_fp;			// Output file pointer

unsigned
	d_label,			// Count of labels defined
	d_code,				// Count of code blocks defined
	d_data,				// Count of data blocks defined
	p_label,			// Count of labels processed
	p_code,				// Count of code blocks processed
	p_data;				// Count of data blocks processed

// Command line options
static char
	Xstats = 255,		// Statistics
	Xwarn = 255,		// Warning messages
	Xdebug,				// Debug mode
	xloc = 255,			// Output location with name
	xskip = 255,		// Define skip symbols
	xnum,				// Output line numbers
	*in_file,			// Input file
	*tcl_file,			// TCL definition file
	*sym_file;			// Symbol definition file

static char				// Command range->text translation
	*cmd_txt[] = { "", "*", "+" };

/*
 * Formatted error output function
 */
register error(args)
	unsigned args;
{
	char buffer[1024+1];

	_format_(nargs() * 2 + &args, buffer);

	if(fp)
		fclose(fp);
	if(out_fp)
		fclose(out_fp);
	if(line)
		fprintf(stderr, "[%u] ", line);

	fputs(buffer, stderr);
	putc('\n', stderr);
	exit(-1);
}

/*
 * Formatted warning output function
 */
register warn(args)
	unsigned args;
{
	char buffer[1024+1];

	_format_(nargs() * 2 + &args, buffer);

	if(Xwarn) {
		fputs(statpre, stdout);
		if(line)
			printf("[%u] ", line);
		fputs(buffer, stdout);
		putc('\n', stdout); }
}

/*
 * Determine if a character is a valid symbol
 */
int issymbol(char c)
{
	return	((c >= 'A') && (c <= 'Z'))
		||	((c >= 'a') && (c <= 'z'))
		||	(c == '_');
}

/*
 * Get file name with optional extension
 */
unsigned char *getf(unsigned char *file, unsigned char *dext)
{
	unsigned char *p, *ext;

	p = buffer;
top:
	ext = 0;
loop:
	switch(*p++ = toupper(*file++)) {
		case ':' :
		case '\\': goto top;
		case '.' : ext = p;
		default: goto loop;
		case 0 : }
	if(!ext) {
		*(p-1) = '.';
		strcpy(ext = p, dext); }
	return ext;
}

/*
 * Load the TCL definition file into memory
 *
 * Layout of extended memory
 * <# of records>
 * .. 00-7F = Comment indicator
 * .. 8x    = Special code
 */
void load_tcl_definition(char *filename)
{
	unsigned
		i,				// General temp vars
		coff,			// Command offset
		ecount;			// Count of outputs
	unsigned char
		cmd,			// Command code
		pcode,			// Parameter code
		*ptr;			// General input pointer

	getf(filename, "DAT");
	fp = fopen(buffer, "rvq");

	cmd = coff = line = ecount = 0;

	while(fgets(ptr = buffer, sizeof(buffer)-1, fp)) {
		++line;
		if((*buffer == ';') || !*buffer)
			continue;
		if(!isspace(*ptr)) {		// New command definition
			coff = 0;
			switch(*ptr) {
			case '+' :
				coff += 26;
			case '*' :
				coff += 26;
				++ptr; }
			if(!isalpha(cmd = toupper(*ptr++)))
				error("Invalid TCL command: %s", buffer);
			poke(data_seg, data_top++, 0xFF);
			cmd -= 'A';
			tcl_cmds[coff += cmd] = data_top;
			ecount = 0; }
		while(*ptr) {				// Parse parameters
			switch(*ptr) {
			case '\t' :
			case ' ' :				// Ignore whitespace
				++ptr;
				continue;
			case ';' :				// Comment
				while(isspace(*++ptr));
				while(*ptr)
					poke(data_seg, data_top++, *ptr++);
				continue;
			case 's' :	pcode = T_SKIP;		goto savep;
			case '#' :	pcode = T_VAR;		goto savep;
			case 'c' :	pcode = T_CONST;	goto savep;
			case 'b' :	pcode = T_BUF;		goto savep;
			case 'm' :	pcode = T_MEM;		goto savep;
			case 't' :	pcode = T_STRC;		goto savep;
			case 'u' :	pcode = T_STR;
			savep:
				poke(data_seg, data_top++, pcode);
				if(*++ptr == '.')
					++ptr;
				continue; }
			if(isdigit(*ptr)) {
				i = 0;
				while(isdigit(*ptr))
					i = (i * 10) + (*ptr++ - '0');
				pcode = T_REQD + i;
				goto savep; }
			error("Bad TCL parameter: %s", ptr); } }
	poke(data_seg, data_top++, 0xFF);
	fclose(fp);
	fp = line = 0;
	Debug(("TCL def: %u\n", data_top))
}

/*
 * Load memory definition file
 */
void load_mem_definition(char *filename)
{
	unsigned m, ml, l, cmt_mark;
	unsigned char *ptr, c, mt, f, mf;

	getf(filename, "SYM");
	fp = fopen(buffer, "rvq");

	line = f = mt = 0;
	cmt_mark = -1;
	while(fgets(ptr = buffer, sizeof(buffer)-1, fp)) {
		++line;
		while(isspace(*ptr))
			++ptr;
		if(!*ptr)				// Null line
			continue;
		if(*ptr == ';') {		// Comment line
			if(*++ptr == ';')	// Special comment
				continue;
			if(isdigit(*ptr)) {
				if(mt != '$')
					error("Must be in CODE block");
				l = 0;
				while(isdigit(*ptr))
					l = (l * 10) + (*ptr++ - '0');
				if(!mem_lcc[ml]++)
					mem_lcms[ml] = cmt_top;
				poke(cmt_seg, cmt_top++, l-1); }
			else {
				mt =  0;
				if(cmt_mark == -1)
					cmt_mark = cmt_top; }
			while(isspace(*ptr))
				++ptr;
			c = 0;
			while(*ptr)
				poke(cmt_seg, cmt_top++, c = *ptr++);
			poke(cmt_seg, (c) ? cmt_top-1 : cmt_top++, c|0x80);
			continue; }
		if(isdigit(*ptr)) {
			if(f)
				poke(data_seg, data_top++, 0xFF);
			m = mt = f = mf = 0;
			while(isdigit(*ptr))
				m = (m * 10) + (*ptr++ - '0');
			ml = m;
			if(*ptr == '-') {
				mf = 255;
				++ptr;
				ml = 0;
				while(isdigit(*ptr))
					ml = (ml * 10) + (*ptr++ - '0');
				if(m < ml) {
					l = m;
					m = ml;
					ml = l; } }
			if(m >= N_MEM)
				error("Memory location out of range: %s", buffer);
			while(isspace(*ptr)) ++ptr;
			if(*ptr)
				mt = *ptr++;
			if(mf) {
				if(mt != '=')
					error("Range must be data: %s", buffer);
				while(isspace(*ptr)) ++ptr;
				l = ml;
				while(l <= m)
					mem_flags[l++] |= M_DATA; }
			while(isspace(*ptr)) ++ptr;
			if(mem_flags[ml] & M_USED)
				error("Duplicate location: %s", buffer);
			if(cmt_mark != -1) {
				mf = 0;
				poke(cmt_seg, cmt_top++, 0xFF);
				mem_cmts[ml] = cmt_mark;
				mem_flags[ml] |= M_CMT;
				cmt_mark = -1; }
			if(*ptr) {
				mf = 0;
				mem_name[ml] = data_top;
				if(!issymbol(*ptr)) {
				badsym:
					error("Bad symbol: %s", buffer); }
				while(c = *ptr++) {
					if(c == '@') {
						poke(data_seg, data_top++, c);
						while(isspace(*ptr)) ++ptr;
						if(*ptr) goto badsym;
						break; }
					if(issymbol(c) || isdigit(c) || (c == '@'))
						poke(data_seg, data_top++, c);
					else if(!isspace(c))
						goto badsym; }
				poke(data_seg, data_top++, 0); }
			if(mf) continue;
			switch(mt) {
			case '=' :
				mem_flags[ml] |= M_DATA;
			case '$' :
				mem_flags[ml] |= M_USED;
				continue; }
			error("Bad memory type: %s", buffer); }

		// Label definition
		out_ptr = out_buf;
		if(!issymbol(*ptr)) goto badsym;
		if(mt != '$')
			error("Must be in CODE block");
		if(cmt_mark != -1)
			error("Cannot comment individual lines: %s", buffer);
		while((c = *ptr) && !isspace(c)) {
			if(issymbol(c) || isdigit(c))
				*out_ptr++ = c;
			else
				goto badsym;
			++ptr; }
		while(isspace(*ptr)) ++ptr;
		*out_ptr = 0;
		if(!isdigit(*ptr))
			error("Bad line number: %s", buffer);
		l = 0;
		while(isdigit(*ptr))
			l = (l * 10) + (*ptr++ - '0');
		if(!f) {
			f = 255;
			mem_lbls[m] = data_top; }
		poke(data_seg, data_top++, l-1);
		out_ptr = out_buf;
		while(*out_ptr)
			poke(data_seg, data_top++, c = *out_ptr++);
		poke(data_seg, data_top-1, c | 0x80);
		++d_label; }

	if(f)
		poke(data_seg, data_top++, 0xFF);

	fclose(fp);
	fp = line = 0;
	Debug(("MEM def: %u\n", data_top))
	for(m=0; m < N_MEM; ++m) {
		if((mt = mem_flags[m]) & M_DATA)
			++d_data;
		else if(mt & M_USED)
			++d_code; }
}

/*
 * Dump the TCL definition table (Debugging aid)
 */
void dump_tcl_defs()
{
	unsigned i, o;
	unsigned char c, f;
	for(i=0; i < N_CMD; ++i) {
		o = tcl_cmds[i];
		printf("Cmd: %2u '%s%c' [%04x]\n\t", i, cmd_txt[i/26], (i%26)+'A', o);
		if(!o) {
			printf(" !!! NO DEFINITION !!!\n");
			continue; }
more_parms:
		f = 0;
		for(;;) {
			c = peek(data_seg, o++);
			if(c & 0x80) {
				if(f) break;
				printf(" %02x", c); }
			else {
				if(!f) {
					f = 255;
					printf(" ;"); }
				putc(c, stdout); } }
		if(c != 0xFF) {
			--o;
			printf("\n\t");
			goto more_parms; }
		putc('\n', stdout); }
}

/*
 * Write out a label with symbolic substitution
 */
unsigned write_label(unsigned char l, int skip, char flag)
{
	unsigned m, len;
	unsigned char l1;

	if(!(m = mem_lbls[location])) {
	no_symbol:
		if(xskip)
			return sprintf(out_ptr, "L%u", out_lbls[l]);
		if(flag)
			return sprintf(out_ptr,"%d", skip);
		return *out_ptr = 0; }

	for(;;) {
		l1 = peek(data_seg, m++);
		if(l1 == l)
			break;
		if(l1 == 0xFF)
			goto no_symbol;
		while(!(peek(data_seg, m++) & 0x80)); }
	len = 1;
	do {
		l1 = peek(data_seg, m++);
		*out_ptr++ = l1 & 0x7F;
		++len; }
	while(!(l1 & 0x80));
	*out_ptr = 0;
	return len;
}

/*
 * Decompile a TCL command and issue an output message
 */
int decompile_tcl_cmd(unsigned cmd)
{
	int ps;
	unsigned eptr, i, j, pv;
	unsigned char e, pt;
	if(Xdebug)
		printf("Cmd %u: %s%c (%u parms)\n", cmd, cmd_txt[cmd/26], (cmd%26)+'A', parm_count);

	if(!(eptr = tcl_cmds[cmd]))	{	// command not found
		if(Xdebug) printf("No cmd match!");
		return 0; }

load_tcl_parms:
	tcl_count = 0;
	while((e = peek(data_seg, eptr++)) & 0x80)
		tcl_parms[tcl_count++] = e;
	if(Xdebug)
		printf("%u TCL parms\n", tcl_count);
	if(parm_count > tcl_count) {		// Too many parameters
		if((parm_count != 1) || (parm_type[0] != P_NONE))
			goto try_next;
		parm_count = 0; }
	for(i=0; i < tcl_count; ++i) {
		pt = parm_type[i];
		pv = parm_value[i];
		if(Xdebug)
			printf("test: %u/%d against %02x\n", pt, pv, tcl_parms[i]);
		switch(j = tcl_parms[i]) {
		case T_SKIP: switch(pt) {
			case P_NUM:
			case P_NONE: continue; }
			goto try_next;
		case T_VAR:
			if((pt >= P_VAR) && (pt < P_STR))
				continue;
			goto try_next;
		case T_CONST: switch(pt) {
			case P_NUM:
			case P_NONE: continue; }
		var_ok:
			if((pt >= P_VAR) && (pt < P_STR))
				continue;
			goto try_next;
		case T_BUF : switch(pt) {
			case P_NUM:
				if((pv > 5) && (pv != 9))
					goto try_next;
			case P_NONE: continue; }
			goto try_next;
		case T_MEM : switch(pt) {
			case P_NUM:
			case P_NONE: continue; }
			goto var_ok;
		case T_STRC: switch(pt) {
			case P_NUM :
			case P_STR :
			case P_NONE: continue; }
			goto try_next;
		case T_STR: switch(pt) {
			case P_STR :
			case P_NONE: continue; }
			goto try_next; }
		// Forced constant
		switch(pt) {
		case P_NUM :
		case P_NONE:
			if((j - T_REQD) == pv)
				continue; }
	try_next:
		while(!((e = peek(data_seg, eptr++)) & 0x80));
		if(e == 0xFF) {
			if(Xdebug)
				printf("No parm match!\n");
			return 0; }
		--eptr;
		goto load_tcl_parms; }

	/*
	 * This command checks out... add to our output buffer
	 */
	out_cmds[out_count] = cmd;
	out_parm[out_count] = out_ptr;
	out_cmts[out_count] = eptr - 1;
	for(i=0; i < parm_count; ++i) {
		if(i)
			*out_ptr++ = '.';
		pt = parm_type[i];
		ps = pv = parm_value[i];
		switch(tcl_parms[i]) {
		case T_SKIP :	// Substitute a skip target label
			j = out_count + ps;
			if(ps >= 0)
				j += 1;
			if(j < 128) {
					*out_ptr = 0;
					if(ps) {
						if(!(pv = out_lbls[j]))
							out_lbls[j] = pv = j+1;
						write_label(j, ps, 255); }
				goto end_buf; }
			printf("Skip value out of range!\n");
			goto do_def;
		case T_MEM :	// Substitute a memory location name
			if(pv < N_MEM) {
				if(j = mem_name[pv]) {
					while((e = peek(data_seg, j++)) && (e != '@'))
						*out_ptr++ = e;
					continue; }
				goto do_def; }
			printf("Memory location out of range!\n");
		default:
		do_def:
			if(pt == P_NUM) {
				sprintf(out_ptr, "%d", pv);
			end_buf:
				while(*out_ptr)
					++out_ptr;
				continue; }
			else if((pt >= P_VAR) && (pt < P_STR)) {
				if(pt == P_VAR)
					sprintf(out_ptr, "#");
				else
					sprintf(out_ptr, "#%u", pt-P_VAR);
				goto end_buf; }
			else if(pt == P_STR) {
				sprintf(out_ptr, "'%s'", buffer);
				goto end_buf; } } }
	*out_ptr++ = 0;
	++out_count;
	return -1;
}

/*
 * Decode a single TCL string into a series of commands and parameters
 */
int decode_tcl_string(char *cmd_ptr)
{
	unsigned cmd, value, p;
	unsigned char *ptr, mflag, c;

	if(Xdebug) {
		printf("{%u}", location);
		fputs(cmd_ptr, stdout);
		printf("\n"); }

	out_count = 0;
	out_ptr = out_buf;
	memset(out_lbls, 0, sizeof(out_lbls));

	if(!*cmd_ptr)	// Null definition is data
		return 0;

	while(*cmd_ptr) {
		cmd = 0;
		switch(*cmd_ptr) {
			case '+' :
				cmd += 26;
			case '*' :
				cmd += 26;
				++cmd_ptr; }
		if(!isalpha(c = toupper(*cmd_ptr))) {
			warn("Non-alphabetic command: '%c' in location %u", c, location);
			return 0; }
		cmd += (c-'A');
		memset(parm_value, 0, sizeof(parm_value));
		memset(parm_type, 0, sizeof(parm_type));
		parm_count = 0;
	next_parm:
		value = p = mflag = 0;
	next_char:
		switch(c = *++cmd_ptr) {
		case '0' :		// Simple number
		case '1' :
		case '2' :
		case '3' :
		case '4' :
		case '5' :
		case '6' :
		case '7' :
		case '8' :
		case '9' :
			value = (value * 10) + (c - '0');
			p = P_NUM;
			goto next_char;
		case '#' :		// Variable
			p = 0;
			if(isdigit(*(cmd_ptr+1)))
				p = *++cmd_ptr - '0';
			value = p;
			p += P_VAR;
			goto next_char;
		case '-' :		// Negative number
			mflag = 255;
			goto next_char;
		case '\'' :		// Quoted string
			value = 0;
			ptr = buffer;
			while((p = *++cmd_ptr) && (p != '\''))
				*ptr++ = p;
			*ptr = 0;
			p = P_STR;
			goto next_char;
		default:
			parm_value[parm_count] = mflag ? -value : value;
			parm_type[parm_count++] = p;
			if((c == '.') || (c == ','))
				goto next_parm; }
		if(!decompile_tcl_cmd(cmd)) {
			warn("Did not decompile '%s%c' in location %u",
				cmd_txt[cmd/26], (cmd%26)+'A', location);
			return 0; } }

	if(p = mem_lbls[location]) {
		while((cmd = peek(data_seg, p++)) < 128) {
			if(!out_lbls[cmd]) {
				ptr = temp;
				do {
					*ptr++ = (c = peek(data_seg, p++)) & 0x7F; }
				while(!(c & 0x80));
				*ptr = 0;
				warn("Unreferenced label '%s' in line %u", temp, cmd+1);
				out_lbls[cmd] = cmd+1;
				continue; }
			while(!(peek(data_seg, p++) & 0x80)); } }
	return 255;
}

wc(unsigned char c)
{
	if(c &= 0x7F)
		putc(c, out_fp);
}

/*
 * Dump TCL command string output
 */
void dump_tcl_output()
{
	unsigned i, c, cp, x;
	unsigned char c1, *ptr;

	for(i=0; i < out_count; ++i) {
		c = out_cmds[i];
		if(out_lbls[i]) {
			++p_label;
			out_ptr = buffer;
			if(x = write_label(i, 0, 0)) {
				fprintf(out_fp, "%s:", buffer);
				if(x > 7)
					putc('\n', out_fp); } }
		fprintf(out_fp, "\t%s%c\t", cmd_txt[c / 26], (c % 26)+'A');
		ptr = out_parm[i];
		x = 0;
		while(*ptr) {
			putc(*ptr++, out_fp);
			++x; }
		if(x >= 24)
			putc(' ', out_fp);
		else while(x < 24) {
			putc('\t', out_fp);
			while(++x % 8); }
		if(xnum)
			fprintf(out_fp, ";%u ", i+1);
		else
			fputs("; ", out_fp);
		if(x = mem_lcc[location]) {
			cp = mem_lcms[location];
			do {
				if(peek(cmt_seg, cp++) == i) {
					do {
						wc(c1 = peek(cmt_seg, cp++)); }
					while(!(c1 & 0x80));
					goto found_cmt; }
				while(!(peek(cmt_seg, cp++) & 0x80)); }
			while(--x); }
		cp = out_cmts[i];
		while(!((c1 = peek(data_seg, cp++)) & 0x80))
			putc(c1, out_fp);
found_cmt:
		putc('\n', out_fp); }
	while(i < 128) {
		if(out_lbls[i]) {
			++p_label;
			out_ptr = buffer;
			if(write_label(i, 0, 0))
				fprintf(out_fp, "%s:\n", buffer); }
		++i; }
}

/*
 * Get a byte from the code block text
 */
unsigned help_ptr;
char get_text() asm
{
	MOV	BX,DGRP:_help_ptr			; Get offset
	INC WORD PTR DGRP:_help_ptr		; Increment
	MOV	AL,CS:byte ptr HELLO[BX]	; Get byte
}

/*
 * Display a help topic
 */
void help(unsigned topic)
{
	unsigned char c;
	static unsigned output, tab = 8;

	if(topic != 99) {
		while(topic--) {
			while(get_text());
			if(!(tab = get_text()))
				error("Help block not available"); } }
show_next:
	while(c = get_text()) {
		if(c == '\t') {
			do
				putc(' ', stdout);
			while(++output % tab); }
		else {
			output = (c != '\n') && output+1;
			putc(c, stdout); } }
	if((topic == 99) && (tab = get_text())) {
		putc('\n', stdout);
		goto show_next; }
}

main(int argc, char *argv[])
{
	unsigned i, j;
	unsigned char buf[256], *ptr, *ptr1, c, f;

	out_ptr = ptr1 = tcl_file = buf;
	ptr = argv[0];
	while(*ptr) {
		switch(*out_ptr++ = *ptr++) {
		case ':' :
		case '\\' : ptr1 = out_ptr; } }
	sprintf(ptr1, "TCLDEF");
	
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
		case '-D' :
		case '/D' :	Xdebug = 255;				continue;
		case '-H' :
		case '/H' : help(*ptr ? atoi(ptr) : 1);	return;
		case '-L' :
		case '/L' : xskip = 0;					continue;
		case '-N' :
		case '/N' : xnum = 255;					continue;
		case '-S' :
		case '/S' : Xstats = 0;					continue;
		case '-W' :
		case '/W' : Xwarn = 0;					continue;
		case '-@' :
		case '/@' : xloc = 0;					continue;
		case 'T=' : tcl_file = ptr; continue;
		case 'S=' : sym_file = ptr; continue; }
		if(!in_file) {	// Specify input file
			in_file = argv[i];
			continue; }
		if(!out_fp) {	// Specify output file
			getf(argv[i], "TCP");
			out_fp = fopen(buffer, "wvq");
			continue; }
		error("Unknown option: %s", argv[i]);
		exit(-1); }

	IOB_size = 1024;

	if(!in_file) {	/* Help request */
		help(0);
		return; }

	if(!out_fp) {
		out_fp = stdout = setbuf(stdout, 1024);
		*statpre = ';'; }

	if(!(data_seg = alloc_seg(8192)))
		error("Cannot allocate memory");
	cmt_seg = data_seg + 4096;
	Debug(("data=%04x cmt=%04x\n", data_seg, cmt_seg))

	load_tcl_definition(tcl_file);
	if(sym_file)
		load_mem_definition(sym_file);

	/* Issue "welcome" message */
	if(Xstats) {
		fputs(statpre, stdout);
		while((c = get_text()) >= ' ')
			putc(c, stdout);
		printf(" - ?COPY.TXT 2001-2006 Dave Dunfield -  -- see COPY.TXT --.\n"); }

	if(Xdebug)
		dump_tcl_defs();

	line = 0;
	getf(in_file, "UDL");
	fp = fopen(buffer, "rvq");
	while(fgets(ptr = buf, sizeof(buf), fp)) {
		++line;
		if(*ptr++ != '"') {
badudl:		error("Bad UDL format: %s", buf); }
		switch(*ptr++) {
		default: goto badudl;
		case '0' :
		case '1' : }
		location = 0;
		if(*ptr == ',') {		// Large format
			while(isdigit(*++ptr))
				location = (location * 10) + (*ptr - '0');
			if(*ptr++ != ',') goto badudl; }
		else {				// Small format
			for(i=0; i < 3; ++i) {
				if(!isdigit(*ptr)) goto badudl;
					location = (location * 10) + (*ptr++ - '0'); } }
		if(location >= N_MEM)
			error("Memory location out of range: %s", buf);

		ptr1 = ptr;
		while(*ptr1) ++ptr1;
		while(isspace(*--ptr1));
		if(*ptr1 != '"') goto badudl;
		*ptr1 = 0;

		i = mem_flags[location];
		if(i & M_DATA)
			j = 0;
		else
			j = decode_tcl_string(ptr);

		// Display comments for block
		if(mem_flags[location] & M_CMT) {
			i = mem_cmts[location];
			while(peek(cmt_seg, i) != 0xFF) {
				fputs("; ", out_fp);
				do {
					wc(c = peek(cmt_seg, i++)); }
				while(!(c & 0x80));
				putc('\n', out_fp); } }

		// Display location name
		if(i = mem_name[location]) {
			f = xloc;
			while(c = peek(data_seg, i++)) {
				if(c == '@') {
					f = 255;
					break; }
				putc(c, out_fp); }
			if(f)
				fprintf(out_fp, "@%u", location); }
		else
			fprintf(out_fp, (location > 999) ? "%04u" : "%03u", location);

		// Display location content
		if(j) {		// Code block
			++p_code;
			fputs("$\n", out_fp);
			dump_tcl_output();
			putc('\n', out_fp); }
		else {		// Data block
			++p_data;
			putc('=', out_fp);
			fputs(ptr, out_fp);
			putc('\n', out_fp); }
	}

	if(Xstats) {
		printf("%s%5u/65535 bytes of define  memory used/total.\n", statpre, data_top);
		printf("%s%5u/65535 bytes of comment memory used/total.\n", statpre, cmt_top);
		printf("%s%5u/%-5u data blocks defined/processed.\n", statpre, d_data, p_data);
		printf("%s%5u/%-5u code blocks defined/processed.\n", statpre, d_code, p_code);
		printf("%s%5u/%-5u code labels defined/processed.\n", statpre, d_label, p_label); }
	fclose(fp);
	fclose(out_fp);
}

/*
 * Help text - stored in code segment to avoid using precious data space
 */
asm {
HELLO:
DB'TCL Decompiler 1.3'
DB 10,10
DB'use:	TCLD <input_file[.UDL]> [output_file[.TCP]] [options]',10,10
DB'opts:	-D		= display Debug output',10
DB'	-H[n]		= display TCLD Help topics',10
DB'	-L		= do not generate Line (skip) labels',10
DB'	-N		= include line Numbers in output comments',10
DB'	-S		= do not output Statistics',10
DB'	-W		= do not display Warning messages',10
DB'	-@		= do not include "@nnn" with named locations',10
DB'	T=file[.DEF]	= specify TCL definition file [TCLDEF.DAT]',10
DB'	S=file[.SYM]	= specify Symbolic definition file',10
DB 10,'?COPY.TXT 2001-2006 Dave Dunfield',10
DB' -- see COPY.TXT --.',10,0
; Help topics
DB	8
DB'TCLD HELP Topics:',10,10
DB'	-H0	- TCLD command syntax',10
DB'	-H1	- TCLD help topics',10
DB'	-H2	- TCLD overview',10
DB'	-H3	- TCLD TCLDEF file format',10
DB'	-H4	- TCLD SYMBOL file format',10
DB'	-H5	- TCLD Sample UDL file',10
DB'	-H6	- TCLD Sample SYMBOL file',10
DB'	-H7	- TCLD Default TCLDEF.DAT file',10
DB'	-H8	- Contact information',10,10
DB'	-H99	- Display all topics',10,10
DB'To obtain help output in a file, use the DOS redirection operator:',10,10
DB'	eg: TCLD -H7 >TCLDEF.DAT',10,0
; Overview
DB	8
DB'TCLD Overview:',10,10
DB'TCLD "decompiles" a .UDL file into a TCLP (symbolic TCL) source file.',10,10
DB'The following operations are performed:',10
DB' - Control strings are separated into the individual TCL commands and',10
DB'   output one per line, with a comment describing the TCL operation.',10
DB' - Skip offsets are converted to TCLP line labels.',10
DB' - Memory locations which do not conform to defined TCL command syntax',10
DB'   are output as data strings.',10
DB' - An optional "SYMBOL" file can be provided to pre-define memory locations',10
DB'   name, type (code/data), line labels and block comments. (see -H4)',10,0
; TCLDEF file format
DB	8
DB'TCLDEF file format:',10,10
DB'TCLD uses a TCLDEF file to define the TCL commands, parameters and comments',10
DB'which are interpreted in the input file. The format of this file is:',10
DB' - Lines beginning with ";" are ignored (comments)',10
DB' - Each TCL command definition begins with the command name in column 1,',10
DB'   followed by a tab or spaces, the command operands, tabs or spaces and',10
DB'   the comment applicable to this command (Comment is REQUIRED)',10
DB' - Multiple operand/comment lines may follow the original command definition',10
DB'   to handle allowed variations of the command.',10,10
DB'Within the operand field, the following characters are allowed:',10
DB'  # =A variable (only)    b =Buffer (0-5,9)   c =Constant/Variable',10
DB'  m =Memory loc.(0-1999)  t =Constant/String  u =String (only)',10
DB'  s =Skip offset   0-9 =Fixed constant (must match)   . =Separator',10,10
DB'When decompiling, TCLD will match the first operand pattern which fits',10
DB'with the operands to the command from the source file.',10,10
DB'See supplied TCLDEF.DAT file (-H7) for an example.',10,0
; SYMBOL file format
DB	8
DB'SYMBOL file format:',10,10
DB'TCLD uses a "SYMBOL" file to define user supplied memory location types,',10
DB'names, comments and line labels. The format of this file is:',10
DB' - Lines beginning with ";" are: comments',10
DB'	; text		<- Block comment for next location defined',10
DB'	;n text		<- Line comment for line n in code block',10
DB'	;;		<- Symbol file comment - ignored',10
DB' - Lines beginning with number (nnn) are: memory location definition',10
DB'	nnn=		<- Identify location as DATA',10
DB'	nnn-xxx=	<- Identify range of locations as DATA',10
DB'	nnn=name[@]	<- Identify location as DATA and assign name',10
DB'	nnn$		<- Identify location as CODE',10
DB'	nnn$name[@]	<- Identify location as CODE and assign name',10
DB' - Lines beginning with symbol are: line (skip) label in code block',10
DB'	name nn		<- Assign label (name) to line (nn)',10,10
DB'NOTES:	- Locations identified as DATA are ALWAYS output as data strings.',10
DB'	- Unidentified and CODE locations will be output as CODE only if they',10
DB'	  conform to TCL command syntax. Otherwise they are output as DATA.',10
DB'	- @ causes named location to ALWAYS include "@nnn" with name',10,0
; Sample UDL file
DB	8
#include "R:\XMAS.hu"
; Sample SYMBOL file
DB	0,8
#include "R:\XMAS.hs"
DB	0,8
; TCLDEF file
#include "R:\TCLDEF.h"
DB	0
; Contact information
DB	8
DB 'TCLD was created by:',10,10
DB'	Dunfield Development Services',10
DB'	115 Manion Heights Cres.',10
DB'	RR#2 Carp, Ontario Canada',10
DB'	K0A 1L0',10,10
DB' Specializing in low cost embedded software development tools',10
DB' and software/firmware development services!',10,10
DB' http://www.dunfield.com',10,10
DB' FAX:	613-256-5821',10,10
DB'This program was compiled with DDS Micro-C/PC.',10
DB	0,0
}
