/*
 * TRANZ330 pinpad simulator
 */
#include <stdio.h>
#include <file.h>
#include <window.h>

#define	X			8			// Main window X position
#define	Y			13			// Main window Y position
#define BIOS_TICK	peekw(0x40,0x6C)
#define	HOST_FLAG	0x1234		// Host flag
#define	HOST_DATA	200			// Offset for host data

// Common control characters
#define	SOH			0x01		/* Start of header */
#define	STX			0x02		/* Start of text */
#define	ETX			0x03		/* End of text */
#define	EOT			0x04		/* End of transmission */
#define	ENQ			0x05		/* Enquiry */
#define	ACK			0x06		/* Positive acknowlegement */
#define	SO			0x0E		/* Shift-out */
#define	SI			0x0F		/* Shift-in */
#define	NAK			0x15		/* Negative acknowlegement */
#define	SUB			0x1A		/* */
#define	FS			0x1C		/* Field separator */
#define	GS			0x1D		/* Group separator */
#define	CR			0x0D		/* Carriage return */
#define	vACK		0x07		/* Visa ACK1 */
#define	vWACK		0x11		/* Visa Wait ACK */
#define	vWAIT		0x12		/* Visa WAIT */

unsigned
	itimer,			// Input timeout
	dsize,			// Display size
	seg,			// Segment of host SIM330 program
	off,			// Offset of comm buffer
	offr,			// Read offset into buffer
	offw;			// Write offset into buffer

unsigned char
	Nak = NAK,		// NAK character
	Fill = ' ',		// Fill character
	buffer[100],	// Temp buffer
	account[50],	// Customer account
	eworkkey[17],	// Encrypted working key
	pinblock[17],	// Pin-Block
	*ptr,			// General global pointer
	Lrc;			// Computed LRC
unsigned char
	dukpt[21] = { 0 };

/*
 * Data stored in host
 */
struct {
	unsigned	host_flag;
	unsigned 	key_flag;
	char		masterkey[17];
	char		display[75];
	} host_data = {
	int		HOST_FLAG, 0,
	char	'F','F','F','F','F','F','F','F',
			'F','F','F','F','F','F','F','F', 0
	};

struct WINDOW *ppwin, *pdwin;

#include "des.c"

static char help[] = { "\n\
Use:	PINPAD	KEY	<exactly 16 HEX digits>	- to change MASTER key\n\
		DUKPT	<10-20 HEX digits>	- to set DUKPT number\n\
		DUKPT	OFF			- to revert to MKEY\n\
\n?COPY.TXT 2000-2003 Dave Dunfield\n -- see COPY.TXT --.\n" };

extern unsigned char PSP[];

register error(unsigned args)
{
	unsigned i, c, l;
	unsigned char buffer[81], *ptr, x;

	_format_(nargs() * 2 + &args, buffer);
	wopen(0, 1, 79, 5, WSAVE|WCOPEN|WBOX2|REVERSE);
	wputs("PINPAD: ");
	if(!(x = *(ptr = buffer) - '^'))
		++ptr;
	wputs(ptr);
	wputc('\n');
	i = off;
	l = 0;
	while(c = peek(seg, i++)) {
		if(++l >= 78)
			break;
		wputc(c | 0x100); }
	wputs("\nPress ENTER to Abort/NAK");
	if(!x)
		wputs(", SPACE to proceed.");
	for(;;) switch(wgetc()) {
		case ' ' : if(!x) { wclose(); return; };
		case '\n' :
			wclose();
			poke(seg, off, NAK);
			poke(seg, off+1, 0);
			if(ppwin) { wclose(); wclose(); }
			exit(0); }
}

/*
 * Draw a key on the pinpad window in specified color
 */
void draw_p_key(unsigned k, unsigned char attr)
{
	unsigned x, y;
	unsigned char a;

	static char *keys1[] = {
	"qz ", "abc", "def",
	"ghi", "jkl", "mno",
	"prs", "tuv", "wxy",
	"CLR", "opr", "ENT" };

	static char *keys2[] = {
	" 1 ", " 2 ", " 3 ",
	" 4 ", " 5 ", " 6 ",
	" 7 ", " 8 ", " 9 ",
	" * ", " 0 ", " # " };

	x = ((k % 3) * 6) + 0;
	y = ((k / 3) * 2) + 1;

	a = *ppwin;
	*ppwin = attr;
	w_gotoxy(x, y, ppwin); w_printf(ppwin, "\xDA%s\xBF", keys1[k]);
	w_gotoxy(x,++y,ppwin); w_printf(ppwin, "\xC0%s\xD9", keys2[k]);
	*ppwin = a;
}

int testc()
{
	unsigned c, d;
	if(c = wtstc()) {
		switch(c) {
		case '1' :
		case '2' :
		case '3' :
		case '4' :
		case '5' :
		case '6' :
		case '7' :
		case '8' :
		case '9' : d = c - '1'; goto gotkey;
		case 'c' :
		case 'C' : c = '*';
		case '*' : d = 9;	goto gotkey;
		case '0' : d = 10;	goto gotkey;
		case '.' :
		case '\n': c = '#';
		case '#' : d = 11;
		gotkey:
			draw_p_key(d, REVERSE);
			beep(1000, 50);
			delay(50);
			draw_p_key(d, NORMAL);
			return c;
		case ' ' : if(itimer) return -1;
		default: beep(500, 200); break;
		case _K10 :
			error("^User aborted!"); } }
	return 0;
}

append_char(char c)
{
	poke(seg, offw++, c);
	poke(seg, offw, 0);
	Lrc ^= c;
}
append_string(char *p)
{
	while(*p) {
		poke(seg, offw++, *p);
		Lrc ^= *p++; }
	poke(seg, offw, 0);
}
extract_string(char *dest, unsigned l)
{
	unsigned char c;
	while(l--) {
		c = peek(seg, offr);
		if(c < ' ')
			break;
		*dest++ = c;
		++offr; }
	*dest = 0;
}
expect_char(char *c)
{
	if(peek(seg, offr++) != c)
		error("Expected:%02x, Got:%02x", c, peek(seg, offr-1));
}

/*
 * Display buffer in a width field
 */
display_width(char *p, unsigned w, struct WINDOW *win, char xflag)
{
	w_putc('\r', win);
	do {
		if(*p) {
			w_putc(xflag || *p, win);
			++p; }
		else
			w_putc(Fill, win); }
	while(--w);
}

/*
 * Wait for a key with rotating prompts
 */
int wait_key_prompt()
{
	unsigned i, l, ts, dp, dt;
	unsigned char *ptr, *ptr1, c, buffer[50];

	host_data.key_flag = 0;
	ts = BIOS_TICK & ~7;
	ptr = host_data.display;

next_string:
	if(!*ptr) {
		if(itimer) {
			display_width(buffer, 8, pdwin, 0);
			do {
				while((i = BIOS_TICK & ~7) == ts);
				ts = i; }
			while((wtstc() == 0) && --itimer);
			return 0; }
		ptr = host_data.display; }
	ptr1 = buffer;
	l = 0;
	while(c = *ptr) {
		++ptr;
		if(c == '\n')
			break;
		*ptr1++ = c;
		++l; }
	*ptr1 = 0;

	display_width(buffer, 8, pdwin, 0);
	dp = 0;
	dt = 2;

	do {
		i = BIOS_TICK & ~7;
		if(i != ts) {
			ts = i;
			if(l <= 8) {
				if(!--dt)
					goto next_string; }
			else {
				if((dp + 4) > l)
					goto next_string;
				display_width(buffer+dp++, 8, pdwin, 0); } } }
	while(!(i = testc()));

	return i;
}

/*
 * Send the pinblock back to the terminal
 */
send_pinblock(unsigned l)
{
	unsigned char epinblock[17], workkey[17];

	append_char(Lrc = STX);
	if(*dukpt) {
		DesEncrypt(epinblock, pinblock, host_data.masterkey, 1);
		append_string("710");
		append_string(dukpt); }
	else {
		DesEncrypt(workkey, eworkkey, host_data.masterkey, 0);	// Decrypt working key
		DesEncrypt(epinblock, pinblock, workkey, 1);			// Encrypt pinblock
		append_string("71.0");
		sprintf(buffer, "%02u", l);
		append_string(buffer);
		append_string("01"); }
	append_string(epinblock);
	append_char(ETX);
	append_char(Lrc);
	exit(0);
}

update_host()
{
	unsigned i;

	ptr = host_data;
	for(i=0; i < sizeof(host_data); ++i)
		poke(seg, off+i+HOST_DATA, *ptr++);
}

draw_pinpad()
{
	unsigned i;

	ppwin = wopen(X, Y, 19, 11, WSAVE|WCOPEN|WBOX2|NORMAL);
	pdwin = wopen(X+6, Y+1, 8, 1, WSAVE|WCOPEN|REVERSE);

	for(i=0; i < 12; ++i)
		draw_p_key(i, NORMAL);
}

int match(char *p, char *p1)
{
	while(*p)
		if(toupper(*p++) != *p1++)
			return 0;
	return -1;
}
		
main(int argc, char *argv[])
{
	unsigned i, o;
	unsigned char c;

	if(argc < 3) {
		printf("\nCurrent PINPAD settings:\n\n");
		printf("\tMasterkey: %s\n", host_data.masterkey);
		printf("\tDUKPT    : %s\n", *dukpt ? dukpt : "OFF");
		abort(help); }

	i = 0;
	ptr = argv[2];

	if(match(argv[1], "DUKPT")) {
		if(match(ptr, "OFF"))
			memset(dukpt, 0, sizeof(dukpt));
		else {
			while(isxdigit(*ptr))
				dukpt[i++] = toupper(*ptr++);
			if((i < 10) || (i > 20) || *ptr)
				abort(help); }
		goto saveimage; }
	if(match(argv[1], "KEY")) {
		while(isxdigit(*ptr))
			host_data.masterkey[i++] = toupper(*ptr++);
		if(i != 16)
			abort(help);
	saveimage:
		if(o = open(argv[0], F_WRITE|F_BINARY)) {
			write(0x100, PSP-0x102, o);
			close(o); }
		return; }

	seg = atoi(argv[1]);
	off = offr = offw = atoi(argv[2]);
	if(!(seg && off)) abort(help);

	/* Perform data storage with host */
	if(peekw(seg, off+HOST_DATA) == HOST_FLAG) {
		ptr = host_data;
		for(i=0; i < sizeof(host_data); ++i)
			*ptr++ = peek(seg, off+i+HOST_DATA); }
	else
		update_host();

	while((c = peek(seg, offr++)) != STX) {
		if(c == ACK)
			return;
		if(c == SI)
			break;
		if(c == EOT)
			return;
		if(!c) {
			Nak = 0;
			error("No STX in packet"); } }

	o = offr;
	Lrc = 0;
	while(c = peek(seg, o++)) {
		Lrc ^= c;
		if((c == ETX) || (c == SO)) {
			if(peek(seg, o) == Lrc)
				goto packet_ok;
			error("LRC mismatch: RX:%02x COMP:%02x", peek(seg, o), Lrc); } }
	error("No ETX in packet");
packet_ok:

// We have received packet from SIM330
	extract_string(buffer, 2);
	if(buffer[0] == 'Z') {
		switch(buffer[1]) {
//		default:	goto badmsg;
		case '2' : display_string(0);	break;
		case '8' : display_string(-1);	break;
		case '4' : switch(peek(seg, offr++)) {
//			default:	goto badmsg;
			case '0' :	get_keypress(-1);	break;
			case '2' : 	get_keypress(0);	} } }
	else switch(atoi(buffer)) {
//		default:	goto badmsg;
		case 02 : transfer_key02(); break;
		case 04 : test_key04();		break;
		case 70 : request_pin_70(); }

//rok:
	poke(seg, off, ACK);
	poke(seg, off+1, 0);
	return;
//badmsg:
//	error("Unknown message: %s", buffer);
//	goto rok;
}

/*
 * Get a keypress
 */
get_keypress(char flag)
{
	unsigned i;


	i = host_data.key_flag;
	if((i-1) & 0xFF00) {
		draw_pinpad();
		beep(1000, 300);
		i = wait_key_prompt();
		wclose();
		wclose(); }
	else
		host_data.key_flag = 0;

	append_char(Lrc = STX);
	append_string(flag ? "Z41" : "Z43");

	if(flag) switch(i) {
		case '1' : i = 1;	goto savekey;
		case '2' : i = 2;	goto savekey;
		case '3' : i = 3;	goto savekey;
		case '4' : i = 5;	goto savekey;
		case '5' : i = 6;	goto savekey;
		case '6' : i = 7;	goto savekey;
		case '7' : i = 9;	goto savekey;
		case '8' : i = 10;	goto savekey;
		case '9' : i = 11;	goto savekey;
		case '*' : i = 13;	goto savekey;
		case '0' : i = 14;	goto savekey;
		case '#' : i = 15;
		savekey:
			sprintf(buffer, "%u", i);
			append_string(buffer);
			break;
		default:
			append_char('?'); }
	else
		append_char(i);

	append_char(ETX);
	append_char(Lrc);

	exit(0);
}

request_pin_70()
{
	unsigned i, l;
	char pin[13];

	if(!*dukpt)
		expect_char('.');
	extract_string(account, 19);
	expect_char(FS);
	if(*dukpt) switch(i = peek(seg, offr++)) {
		default: error("Expected C/D - got: %02x", i);
		case 'C' :
		case 'D' : }
	else
		extract_string(eworkkey, 16);
	extract_string(buffer, 10);
	expect_char(ETX);
	append_char(ACK);

	draw_pinpad();
	beep(1000, 300);

	sprintf(host_data.display, " Total\n$%s\nEnter PIN\nPUSH \"ENTER\"", buffer);
reloop:
	i = wait_key_prompt();
	l = 0;
nextc:
	switch(i) {
		case '*' : goto reloop;
		case '#' :
			if(l < 4) goto dobeep;
			pinblock[0] = 0;
			pinblock[1] = l;
			for(i=0; i < 14; ++i)
				pinblock[i+2] = (i < l) ? (pin[i]-'0') : 0x0F;
			ptr = account;
			while(strlen(ptr) > 13)
				++ptr;
			sprintf(buffer, "%013s", ptr);
			for(i=0; i < 12; ++i)
				pinblock[i+4] ^= (buffer[i] - '0');
			for(i=0; i < 16; ++i) {
				if((pinblock[i] += '0') > '9')
					pinblock[i] += 7; }
			pinblock[16] = 0;
			wclose();
			wclose();
			send_pinblock(l); }
	if(l < 8) {
		pin[l++] = i;
		pin[l] = 0;
		display_width(pin, 8, pdwin, '*'); }
	else {
	dobeep:
		beep(500, 200); }
	while(!(i = testc()));
	goto nextc;
}

transfer_key02()
{
	if(*dukpt) {
		append_char(ACK);
		append_char(EOT); }
	else {
		expect_char('0');
		extract_string(host_data.masterkey, 16);
		expect_char(SO);
		update_host();
		append_char(ACK);
		Lrc = SI;
		append_char(SI);
		append_string("020");
		append_string(host_data.masterkey);
		append_char(SO);
		append_char(Lrc); }
	exit(0);
	
}

test_key04()
{
	unsigned char c;
	if(*dukpt) {
		append_char(ACK);
		append_char(EOT); }
	else {
		append_char(ACK);
		append_char(Lrc = SI);
		append_string("04");
		append_char(c = peek(seg, offr));
		append_char((c == '0') ? 'F' : 0);
		append_char(SO);
		append_char(Lrc); }
	exit(0);
}

display_string(unsigned char iflag)
{
	if(peek(seg, offr) == 0x1A)
		++offr;
	extract_string(host_data.display, sizeof(host_data.display)-1);
	expect_char(ETX);

	if(!*host_data.display) {
		if(iflag) {
			Fill = '-';
			strcpy(host_data.display, "       <---"); }
		else
			strcpy(host_data.display, " "); }

	draw_pinpad();

	itimer = 2;
	host_data.key_flag = wait_key_prompt();

	wclose();
	wclose();
	update_host();
}
