/*
 * EDT: a small/simple text editor
 *
 * ?COPY.TXT 1983-2008 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * To compile under linux/OS-X:
 *	Insure "#define UNIX" below is NOT commented out
 *		gcc edt.c -o edt
 *	below: Insure "#define UNIX" *IS* commented out
 * To compile under Turbo-C 2.0:
 *		geva -T			// Generate VIDEO.ASM for Turbo-C
 *		tcc -ms edt.c video.asm
 * To compile with Micro-C:
 *		geva -M			// Generate VIDEO.ASM for Micro-C
 *		cc edt -pofm
 *		masm/ml video;
 *		lc -s edt video
 *
 * Please note: This was one of the earliest significant applications I
 *	created - I didn't know C terrible well at that time!!!
 */
/*#define	UNIX*/

#include <stdio.h>				/* Standard I/O definitions */
#include <ctype.h>				/* Character classification macros */
#include <stdarg.h>				/* Variable argument definitions */
#ifdef UNIX
	#include <termios.h>
	#include <stdlib.h>
	#include <unistd.h>
	#include <fcntl.h>
	#include <sys/signal.h>
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <sys/ioctl.h>
	#include <asm/ioctls.h>		/* Comment out for OS-X */
#endif

#define EDIT_SIZE	256			/* max. size of line to edit */
#define CMD_SIZE	50			/* max. size of input commands */
#define TAB_WIDTH	4			/* Default tab width */

/* output translations */
#define _CL 	0x85			/* clear entire display */
#define _CD		0x86			/* clear to end of display */
#define _CE 	0x87			/* clear to end of line */
#define _SO 	0x88			/* Begin standout mode */
#define _SE 	0x89			/* end standout mode */

/* input translations */
#define	_KUA	0x80		/* Up arrow */
#define	_KDA	0x81		/* Down arrow */
#define	_KLA	0x82		/* Left arrow */
#define	_KRA	0x83		/* Right arrow */
#define	_KPU	0x84		/* PageUp */
#define	_KPD	0x85		/* PageDn */
#define	_KHO	0x86		/* Home */
#define	_KEN	0x87		/* End */
#define	_KKP	0x88		/* Keypad '+' */
#define	_KKM	0x89		/* Keypad '-' */
#define	_KIN	0x8A		/* Ins */
#define	_KDL	0x8B		/* Del */
#define	_KBS	0x8C		/* Backspace */
#define	_K1		0x8D
#define	_K2		0x8E
#define	_K3		0x8F
#define	_K4		0x90
#define	_K5		0x91
#define	_K6		0x92
#define	_K7		0x93
#define	_K8		0x94
#define	_K9		0x95
#define	_K10	0x96
#define	_CPU	0x97		/* CTRL-PageUp */
#define	_CPD	0x98		/* CTRL-PageDn */
#define	_CHO	0x99		/* CTRL-Home */
#define	_CEN	0x9A		/* CTRL-End */
#define	_CLA	0x9B		/* CTRL-Left arrow */
#define	_CRA	0x9C		/* CTRL-Right arrow */
#if 0
#define	_AltA	0x9D
#define	_AltB	0x9E
#define	_AltC	0x9F
#define	_AltD	0xA0
#define	_AltE	0xA1
#define	_AltF	0xA2
#define	_AltG	0xA3
#define	_AltH	0xA4
#define	_AltI	0xA5
#define	_AltJ	0xA6
#define	_AltK	0xA7
#define	_AltL	0xA8
#define	_AltM	0xA9
#define	_AltN	0xAA
#define	_AltO	0xAB
#define	_AltP	0xAC
#define	_AltQ	0xAD
#define	_AltR	0xAE
#define	_AltS	0xAF
#define	_AltT	0xB0
#define	_AltU	0xB1
#define	_AltV	0xB2
#define	_AltW	0xB3
#define	_AltX	0xB4
#define	_AltY	0xB5
#define	_AltZ	0xB6
#endif
extern unsigned char *refresh_line();
extern unsigned char *find_line(unsigned n);
extern void v_putc(unsigned char c);
#ifdef _MICROC_
	register v_printf(unsigned args);
#else
	extern void v_printf(unsigned char *format, ...);
#endif

int
	cx,						/* Virtual cursor 'x' position */
	cy,						/* Virtual cursor 'y' position */
	horz,					/* Real cursor 'x' position */
	offset,					/* Offset of screen from start of line */
	edlen,					/* Length of line being edited (0 = no line edit) */
	delen = 0,				/* Length of line in delete buffer */
	tab_width = TAB_WIDTH;	/* Current tab width */

unsigned char
	*t_start,				/* Start of edit buffer */
	*t_end,					/* End of text in memory */
	*txtpos,				/* Position of current line */
	*scrtop,				/* Address of top of screen */
	*actpos,				/* Address of actual current line (txtpos or edit_buff) */
	*tag = 0,				/* Pointer to start of tagged lines */
	*tag1 = 0,				/* Pointer to end of tagged lines */
	*errmsg = 0,			/* Error message to output flag */
	*newpos = 0,			/* New cursor position */
	*ep,					/* Startup command execute pointer */
	changed = 0,			/* Line changed flag */
	CHANGED = 0,			/* File changed flag */
	refresh = 0,			/* Screen refresh required flag */
	insflg = 0,				/* Insert mode flag */
	eolflg = 0,				/* End of line display flag */
	video = 0,				/* Video attribute on/off indicator */
	window = 255,			/* Window edit mode enable flag */
	*fname,					/* Name of file being edited */
	command[CMD_SIZE+1],	/* Command line input buffer */
	edit_buff[EDIT_SIZE],	/* Line edit buffer */
	del_buff[EDIT_SIZE];	/* Line deleted buffer */

#ifdef UNIX
	unsigned
		Lines = 25,				/* # lines on screen */
		Lines1,					/* # lines-1 */
		Lines2,					/* # lines-2 */
		Cols = 80,				/* # columns on screen */
		Buffer_size;			/* Buffer size */

	unsigned char *ext8[] = { "s", "mac", "tcl" };

unsigned char Help[] = { "\n\
Use: edt <file> [c=initial_cmd -video_inhibit b=bufsize h=height w=width]\n\n\
?COPY.TXT 1983-2008 Dave Dunfield\n\
 -- see COPY.TXT --.\n\n" };

unsigned char Help1[] = { "\
Special keys                   Commands                   Line ranges\n\
------------                   --------                   -----------\n\
PgDn  = Page forward             C = Copy line(s)          * = Current line\n\
PgUp  = Page backward            D = Delete line(s)        / = Entire file\n\
F11   = Start of file            F = File info             n = line # (1+)\n\
F12   = End of file             nH = Set Htab size         0 = End of file\n\
Home  = Start of line            I = Insert new line(s)    = = Tagged lines\n\
End   = End of line              L = List (unformatted)  r,r = range to range\n\
F8    = Redraw screen            M = Move line(s)\n\
sF3   = Word right               P = Print (formatted)\n\
sF4   = Word left                Q = Quit editor\n\
INS   = Insert/Overwrite        QQ = Quit with no save\n\
DEL   = Delete character     Rfile = Read & insert file   Examples:\n\
BKSPC = Delete previous  S/old/new = Substitute text      ---------\n\
F1    = EOL display              T = Tag lines             D\n\
F2    = Cursor position          V = Visual mode       1,10C\n\
F3    = Move line to top   W[file] = Write file           =M\n\
F4    = Tag line(s)        X[file] = eXit/write file      0R my.fil\n\
F5    = Del to end of line   ?text = Search for text      /W my.fil\n\
F6    = Del end of line     $[cmd] = Execute DOS command\n\
F7    = Insert deleted line (null) = Move to line range\n\
F10   = Line mode command\n\
F9    = Re-execute command\n" };

#else
	#define	Lines		25			/* Fixed width */
	#define	Lines1		(Lines-1)
	#define	Lines2		(Lines-2)
	#define	Cols		80			/* Fixed height */
	#define	Buffer_size	51201		/* Fixed buffer */

	unsigned char
		buffer[Buffer_size];		/* Edit buffer */

	unsigned char *ext8[] = { "ASM", "MAC", "TCL" };
#endif

unsigned Strlen(str)
	unsigned char *str;
{
	unsigned length;
	length = 0;
	while(*str++)
		++length;
	return length;
}

/*
 * Display a line of text
 */
unsigned char *display_line(ptr)
	unsigned char *ptr;
{
	register int h;
	register unsigned char c;

	h = 0;

	do {
		if((c = *ptr++) == 9) {
			do
				putchr(' ');
			while(++h % tab_width); }
		else {
			putchr(c);
			++h; } }
	while(c != '\n');
	return ptr;
}

/*
 * Main program
 */
main(argc, argv)
	int argc;
	unsigned char *argv[];
{
/*	Main pgm variables:
 *	i			- General unsigned int
 *	j			- Same as above
 *	tmptr		- General temporary pointer
 *	edtpos		- Misc pointer used in editing functions
 *	chr			- General purpose character variable
 *	chr1		- Same as above
 *	cchr		- Character entered from keyboard (command or text)
 */
	unsigned i, j;
	FILE *fp;
	unsigned char *tmptr, *edtpos, chr, chr1, cchr;
#ifdef UNIX
	struct stat fs;
#endif

	for(i=1; i < argc; ++i) {
		tmptr = argv[i];
		j = *tmptr++;
		switch((j << 8) | *tmptr++) {
		case ('-'<<8)|'v':			/* Visual mode disable */
		case ('-'<<8)|'V' : window = 0;	continue;
		case ('c'<<8)|'=' :			/* pre-Command  */
		case ('C'<<8)|'=' : ep = tmptr;	continue;
#ifdef UNIX
		case ('b'<<8)|'=' :			/* preallocate Buffer */
		case ('B'<<8)|'=' :
			if((Buffer_size = atoi(tmptr) * 1024) < 32768)
				Buffer_size = 32768;
			continue;
		case ('w'<<8)|'=' :			/* screen Width */
		case ('W'<<8)|'=' :
			if((Cols = atoi(tmptr)) < 40)
				Cols = 40;
			continue;
		case ('h'<<8)|'=' :			/* screen Height */
		case ('H'<<8)|'=' :
			if((Lines = atoi(tmptr)) < 8)
				Lines = 8;
			continue;
#endif
		} if(fname)
			goto help;
		fname = argv[i]; }
	if(!fname) {
#ifdef UNIX
	help: fputs(Help, stdout);
#else
	help: window = 0;
		x_help();
#endif
		exit(255); }

#ifdef UNIX
	Lines2 = (Lines1 = Lines - 1) - 1;
#endif

	edtpos = 0;
	tmptr = fname;

#ifdef UNIX
	/* Scan for file extension */
	while(chr = *tmptr++) {
		if(chr == '.')
			edtpos = tmptr; }

	/* Determine size of edit buffer */
	if(!Buffer_size) {
		Buffer_size = 65535;
		if(!stat(fname, &fs)) {
			if((i = fs.st_size + 32768) > Buffer_size)
				Buffer_size = i; } }
#else
	/* Convert to upper case & scan for extension */
	while(chr = *tmptr) {
		*tmptr++ = ((chr >= 'a') && (chr <= 'z')) ? (chr - 0x20) : chr;
		if(chr == '.')
			edtpos = tmptr; }

	t_start = t_end = buffer;
#endif

	/* Recognize extensions requireing 8-char tabs */
	if(edtpos) {
		for(i=0; i < (sizeof(ext8)/sizeof(ext8[0])); ++i) {
			if(partial_match(ext8[i], edtpos))
				tab_width = 8; } }

	/* Read file into edit buffer */
	if(fp = fopen(fname, "rt")) {
#ifdef UNIX
		t_start = malloc(Buffer_size+1)+1;
#endif
		i = fread(t_start, 1, Buffer_size, fp);
		if(i >= Buffer_size)
			errmsg = "File too large - Truncated\007";
		t_end = t_start + i;
		fclose(fp); }
	else {
#ifdef UNIX
		t_start = t_end = malloc(Buffer_size+1)+1;
#endif
		errmsg = "New file"; }

	*t_end = '\n';

	if(window)
		v_init();

home:	cx = cy = offset = horz = 0;
	scrtop = txtpos = t_start;
	--refresh;

	if(ep) {
		execute(ep);
		ep = 0; }

cmd: if(window) {
		if(newpos) {
			reposition(newpos);
			newpos = 0; }
		j = position_cursor();
		if(refresh)			/* refresh screen if nessary */
			refresh_screen(scrtop, refresh = 0);
		if(errmsg) {		/* issue error message if any */
			error_message(errmsg);
			errmsg = 0; }
		v_gotoxy(horz - offset, cy);
		cchr = v_getc();
		if((cchr != /*_KDO*/_KDA) && (cchr != /*_KUP*/_KUA))
			cx = j;
		actpos += cx;
		if(cchr & 0x80) {		/* special command */
		switch(cchr) {
			case /*_KHO*/_CPU:					/* start of file */
				update_changes();
				goto home;
			case /*_KEN*/_CPD:					/* end of file */
				update_changes();
				scrtop = txtpos = t_end;
				back_page(Lines1);
				newpos = t_end;
				break;
			case /*_KPU*/_KPU:					/* page up */
				back_page(Lines);
				break;
			case /*_KND*/_KRA:					/* cursor forward */
				fwd_chr();
				break;
			case /*_KDO*/_KDA:					/* down key */
				fwd_line();
				break;
			case /*_KPD*/_KPD:					/* page down */
				fwd_page();
				break;
			case /*_KBS*/_KLA:					/* backspace key */
				back_chr();
				break;
			case /*_KUP*/_KUA:					/* up key */
				back_line();
				break;
			case /*_K6*/_K3:					/* current line to top of screen */
				update_changes();
				scrtop = txtpos;
				cy = 0;
				--refresh;
				break;
			case /*_K3*/_KIN:					/* toggle insert flag */
				if(insflg = !insflg)
					errmsg = "Overwrite";
				else
					errmsg = "Insert";
				break;
			case /*_KPL*/_KHO:					/* cursor to start of line */
				if(!cx)
					back_line();
				cx = 0;
				break;
			case /*_KPR*/_KEN:					/* cursor to end of line */
				cx = 32767;
				if(*actpos == '\n')
					fwd_line();
				break;
			case /*_K11*/_CRA:					/* word right */
				if(*actpos == '\n')
					fwd_chr();
				else {
					while(*actpos > ' ') {
						++actpos;
						++cx; }
					while(((chr = *actpos) != '\n') && (chr <= ' ')) {
						++actpos;
						++cx; } }
				break;
			case /*_K12*/_CLA:					/* word left */
				if(cx) {
					while((cx > 0) && (*--actpos <= ' '))
						--cx;
					while((cx > 0) && (*actpos > ' ')) {
						--actpos;
						--cx; } }
				else
					back_chr();
				break;
			case /*_KDP*/_KBS:					/* backspace and delete */
				back_chr();
				cx = position_cursor();
				v_gotoxy(horz-offset, cy);
			case /*_KDC*/_KDL:					/* delete character key */
				if(txtpos < t_start)
					break;
				start_edit();
				chr = *(tmptr = edtpos = (edit_buff + cx));
				for(i=cx; i <= edlen; ++i)
					*tmptr++ = *++edtpos;
				--edlen;
				if(chr == '\n') {
					update_changes();
					if(!refresh)
						refresh_screen(txtpos, cy); }
				else
					refresh_line(edit_buff + cx);
				break;
			case /*_K7*/_K4:			/* tag lines(s) command */
				update_changes();
				if(!tag) {
					tag = tag1 = txtpos;
					v_gotoxy(horz = 0, cy);
					refresh_line(txtpos); }
				else {
					if(txtpos < tag)
						tag = txtpos;
					else if(txtpos > tag1)
						tag1 = txtpos;
					else
						tag = tag1 = 0;
					--refresh; }
				break;
			case /*_K8*/_K5:			/* delete line */
				update_changes();
				if((edtpos = txtpos + cx) < t_end) {
					tmptr = del_buff;
					delen = 0;
					do {
						*tmptr++ = (chr = *edtpos++);
						++delen; }
					while(chr != '\n');
					delete(txtpos + cx, delen);
					refresh_screen(txtpos, cy); }
				break;
			case /*_K9*/_K6:			/* delete to end of line */
				update_changes();
				tmptr = del_buff;
				edtpos = txtpos + cx;
				for(delen = 0; (chr = *edtpos++) != '\n'; ++delen)
					*tmptr++ = chr;
				delete(txtpos + cx, delen);
				putchr(_CE);
				break;
			case /*_K10*/_K7:			/* insert deleted */
				update_changes();
				insert(txtpos + cx, delen, del_buff, delen);
				refresh_screen(txtpos, cy);
				break;
			case /*_K4*/_K1:			/* toggle eol display on/off */
				eolflg = !eolflg;
			case /*_K13*/_K8:
#ifdef DEBUG
				v_gotoxy(0, Lines1);
				putchr(_SO);
				v_printf("S=%04x E=%04x P=%04x A=%04x, EL=%u DL=%u c=%u y=%u j=%u, H=%u o=%u",
					t_start, t_end, txtpos, actpos, edlen, delen, cx, cy, j, horz, offset);
				putchr(_SE);
				putchr(_CE);
				break;
#endif
			case /*_KCL*/_CHO:			/* refresh screen */
				update_changes();
				--refresh;
				break;
			case /*_K5*/_K2:			/* Display cursor position */
				v_gotoxy(0, Lines1);
				putchr(_SO);
				v_printf(" %u down, %u over, at character %u in line ",
					cy + 1, horz + 1, cx + 1);
				putchr(_SE);
				break;
			case /*_K1*/_KKP:			/* line mode command */
			case /*_K15*/_K10:
				v_gotoxy(0, Lines1);
				putchr(_CD);
				get_input("Command: ", command, CMD_SIZE);
			case /*_K2*/_KKM:		/* execute last command */
			case /*_K14*/_K9:
				v_gotoxy(0, Lines1);		/* display command line */
				putchr(_SO);
				video = 255;
				for(tmptr = command; *tmptr; ++tmptr)
					display_chr(*tmptr);
				putchr(_SE);
				video = 0;
				putchr(_CE);
				fflush(stdout);
				update_changes();
				execute(command);
				if(!errmsg)				/* clear displayed command */
					errmsg = "";
				break;
			default:
				errmsg = "Invalid KEY\007"; } }
	else {					/* normal key pressed */
		start_edit();		/* grab the line to edit */
		tmptr = edit_buff + cx;
		if((!insflg) || (*tmptr == '\n') || (cchr == '\n')) {
			chr = *(edtpos = tmptr);
			do {
				chr1 = chr;
				chr = *++edtpos;
				*++tmptr = chr1; }
			while(chr1 != '\n');
			++edlen;
			tmptr = edit_buff + cx; }
		*tmptr = cchr;
		++cx;
		if(cchr == '\n') {
			update_changes();
			refresh_screen(txtpos, cy);
			cx = 0;
			fwd_line(); }
		else
			refresh_line(tmptr); } }
	else {
		if(errmsg) {
			v_printf("%s\n",errmsg);
			errmsg = 0; }
		if(newpos) {
			reposition(newpos);
			--refresh;
			newpos = 0; }
		if(refresh) {
			if((tmptr = txtpos) >= t_end)
				v_printf("*EOF*\n");
			else display_line(tmptr);
			refresh = 0; }
		get_input("* ", command, CMD_SIZE);
		execute(command); }
	goto cmd;
}

/* backup up one character */
back_chr()
{
	if(0 > --cx) {
		cx = 32767;
		back_line(); }
}

/* backup a line */
back_line()
{
	register unsigned i;

	update_changes();
	--txtpos;
	while((txtpos >= t_start) && (*--txtpos != '\n'));
	++txtpos;
	if(0 > --cy) {
		for(i=Lines/2; i; --i)
			while((scrtop >= t_start) && (*--scrtop != '\n'));
		++scrtop;
		for(i=0; (i < cx) && (*txtpos != '\n'); ++i)
			++txtpos;
		newpos = txtpos;
		--refresh; }
}

/* move back a page */
back_page(n)
	unsigned n;
{
	register unsigned i;

	update_changes();
	for(i=0; i < n; ++i)
		while((scrtop >= t_start) && (*--scrtop != '\n'));
	txtpos = ++scrtop;
	cy = 0;
	--refresh;
}

/* move forward one character */
fwd_chr()
{
	if(*actpos != '\n')
		++cx;
	else if(fwd_line())
		cx = 0;
}

/* move forward one line */
fwd_line()
{
	register unsigned char *ptr;

	update_changes();
	ptr = txtpos;
	do  {
		if(ptr >= t_end)
			return(0); }
	while(*ptr++ != '\n');

	txtpos = ptr;

	if(cy < (Lines2))				/* next line in on screen */
		++cy;
	else {							/* advance screen a line */
		while(*scrtop++ != '\n');
		v_gotoxy(0, Lines1);
		putchr(_CE);
	/* some terminals may not perform forward scrolling when '\n' is */
	/* printed on the bottom line.... if so, replace next two lines */
	/* with code similar to that found in 'back_line'.              */
#ifdef UNIX
		refresh = 255; }
#else
		putchr('\n');
		refresh_screen(txtpos, cy); }
#endif
	return(-1);
}

/* move forward one page */
fwd_page() {
	register unsigned i;

	update_changes();
	for(i=0; i < (Lines1); ++i)
		while((scrtop < t_end) && (*scrtop++ != '\n'));
	txtpos = scrtop;
	cy = 0;
	--refresh;
}

/* function to position the cursor at the right location	*/
/* and sets up the global variable 'actpos' to point to		*/
/* the actual character under the cursor (in the text file	*/
/* or edit buffer. Returns actual size of line 				*/
int position_cursor() {
	register char *tmptr;
	register int i;

	horz = 0;
	if(changed)
		actpos = tmptr = edit_buff;
	else
		actpos = tmptr = txtpos;

	for(i=0; (i < cx) && (*tmptr != '\n'); ++i) {
		++horz;
		if(*tmptr++ == 9)
			while(horz % tab_width)
				++horz; }

	if(horz < offset) {		/* scroll screen right */
		update_changes();
		while(horz < offset)
			offset -= Cols/2;
		--refresh; }
	else if(horz >= (offset + Cols)) {	/* scroll screen left */
		update_changes();
		while(horz >= (offset + Cols))
			offset += Cols/2;
		--refresh; }

	return(i);
}

/* position cursor at an address */
reposition(addr)
	unsigned char *addr;
{
	register unsigned char *ptr;

/* calculate new 'Y' address */
	cy=0;
	for(ptr = scrtop; ptr < addr; ++ptr) {
		if(*ptr == '\n')
			if(++cy > Lines2)
				ptr = addr; }		/* no sence looking farther */

/* calculate new 'X' address */
	cx = 0;
	txtpos = addr;
	while((txtpos > t_start) && (*(txtpos-1) != '\n')) {
		--txtpos;
		++cx; }

	if((addr < scrtop) || (cy > Lines2)) {
		scrtop = txtpos;
		cy = 0;
		--refresh; }
}

/* function to refresh the screen */
refresh_screen(ptr, vert)
	unsigned char *ptr;
	unsigned vert;

{
	unsigned savhorz;

	savhorz = horz;
	do {
		v_gotoxy(horz = 0, vert);
		if(ptr >= t_end) {			/* end of file reached */
			putchr(_SO);
			putstr("*EOF*");
			putchr(_SE);
			break; }
		ptr = refresh_line(ptr); }	/* display the line */
	while(++vert < Lines);
	putchr(_CD);
	horz = savhorz;
}

/* refresh a line */
unsigned char *refresh_line(ptr)
	unsigned char *ptr;
{
	register unsigned eol;

	eol = offset + Cols;

/* if within tagged lines, display in inverse video */
	if((ptr >= tag) && (ptr <= tag1)) {
		video = 255;
		putchr(_SO); }

/* skip any data which preceeds the horizontal scrolling window */
	while((horz < offset) && (*ptr != '\n')) {
		if(*ptr == 9)						/* skip over tab */
			while(++horz % tab_width);
		else
			++horz;
		++ptr; }

/* output any data which is within the horizontal scrolling window */
	if(horz >= offset) {
		while(horz < eol) {
			if(*ptr == '\n') {					/* newline */
				if(eolflg)
					display_chr('\n');
				break; }
			else if(*ptr == 9) {				/* tab */
				do
					putchr(' ');
				while((horz < eol) && (++horz % tab_width)); }
			else								/* all others */
				display_chr(*ptr);
			++ptr; } }

/* skip past the window to the end of the line */
	while(*ptr++ != '\n');

/* if lines were tagged, turn off special video */
	if(video) {
		video = 0;
		putchr(_SE); }

/* if not at end, clear end of line */
	if(horz < eol)
		putchr(_CE);

	return(ptr);
}


/* display character in special video modes */
display_chr(chr)
	unsigned char chr;
{
	if(chr < ' ') {
		putchr((video) ? _SE : _SO);
		putchr(chr+'@');
		putchr((video) ? _SO : _SE); }
	else
		putchr(chr);
	++horz;
}

/* display error message */
error_message(text)
	unsigned char *text;
{
	v_gotoxy(0, Lines1);
	putchr(_SO);
	putstr(text);
	putchr(_SE);
	putchr(_CE);
}

/* grab current line for editing */
start_edit()
{
	register unsigned char *ptr, *ptr1;

	if(!changed) {		/* starting update for a new line */
		CHANGED = changed = -1;
		edlen = 0;
		ptr = edit_buff;
		ptr1 = txtpos;
		do {
			++edlen;
			if(ptr1 >= t_end) {
				*ptr = '\n';
				refresh_screen(t_end, cy+1);
				v_gotoxy(horz-offset, cy);
				break; }
			*ptr++ = *ptr1; }
		while(*ptr1++ != '\n'); }
}

/* update edited line to text file if required */
update_changes()
{
	unsigned buflen;
	register unsigned char *ptr;

	if(changed) {
		buflen = 0;
		ptr = txtpos;
		for(ptr = txtpos; ptr < t_end; ++ptr) {
			++buflen;
			if(*ptr == '\n')
				break; }
	/* insert or delete space as required */
		if(buflen < edlen) 	/* have to insert space */
			insert(txtpos, edlen - buflen, edit_buff, edlen);
		else {
			if(buflen > edlen)
				delete(txtpos, buflen - edlen);
			cpymem(txtpos, edit_buff, edlen); }
		edlen = changed = 0; }
}

/* delete space from the text file */
delete(pos, len)
	unsigned char *pos;
	unsigned len;
{
	register unsigned char *ptr;

	ptr = pos + len;

	if(pos < txtpos) {		/* adjust pointers if deleting below */
		if((txtpos -= len) < pos)
			txtpos = pos;
		newpos = txtpos; }
	if(pos < scrtop) {
		if((scrtop -= len) < pos) { 	/* top of screen deleted */
			scrtop = pos;
			newpos = txtpos;
			--refresh; } }
	if(pos < tag)
		tag -= len;
	if(pos < tag1)
		tag1 -= len;

	while(ptr < t_end)
		*pos++ = *ptr++;
	*(t_end = pos) = '\n';
	CHANGED = -1;

}

/* insert space into the text file */
insert(pos, len, src, slen)
	unsigned char *pos, *src;
	unsigned len, slen;
{
	register unsigned char *ptr, *ptr1;

	if((t_end+len) >= t_start + Buffer_size) {		/* ran out of memory */
		errmsg = "Out of memory - Changes lost\007";
		--refresh;
		return; }

	if(pos < scrtop)			/* adjust pointers if inserting below */
		scrtop += len;
	if(pos < txtpos)
		txtpos += len;
	if(pos < tag)
		tag += len;
	if(pos < tag1)
		tag1 += len;
	if((src > pos) && (src <= t_end))
		src += len;

	ptr = t_end;			/* move text to make room */
	ptr1 = t_end += len;
	while(ptr > pos)
		*--ptr1 = *--ptr;
	while(slen--)			/* copy in destination string */
		*pos++ = *src++;
	*t_end = '\n';
	CHANGED = -1;
}

/* copy memory */
cpymem(dest, src, len)
	unsigned char *dest, *src;
	unsigned len;
{
	while(len--)
		*dest++ = *src++;
}

/* locate a line by number */
unsigned char *find_line(num)
	unsigned num;
{
	register unsigned char *ptr;

	if(num) {
		ptr = t_start;
		while(--num)
			while((ptr < t_end) && (*ptr++ != '\n')); }
	else
		ptr = t_end;
	return(ptr);
}

/* execute line mode commands */
execute(cmd)
	unsigned char *cmd;
{
	unsigned char *start, *end, *optr, *tmptr;
	unsigned i, j;
	int k;
	unsigned char deflg, tgflg, chr;
	FILE *fp;
	start = tgflg = 0;

/* get input line range */
	do {
		while((chr = *cmd++) == ' ');		/* skip leading banks */
		tmptr = txtpos;
		deflg = i = 0;
		end = 0;
		switch(chr) {
			case '=':		/* tagged lines */
				if(!(tmptr = tag)) {
					errmsg = "No tagged lines\007";
					return; }
				end = tag1;
				--tgflg;
				break;
			case '/':		/* entire file */
				tmptr = t_start;
				end = t_end;
			case '*':		/* current line (already set) */
				break;
			default:		/* unknown could be numeric */
				--cmd;
				if(isdigit(chr)) {
					while(isdigit(*cmd))
						i = (i * 10) + (*cmd++ - '0');
					tmptr = find_line(i); }
				else
					--deflg; }

/* handle '+' and '-' from range */
		while((chr = *cmd++) == ' ');		/* skip leading blanks */
		if((chr == '+') || (chr == '-')) {
			i = 0;
			while(isdigit(*cmd))
				i = (i * 10) + (*cmd++ - '0');
			if(chr == '+') {
				while((tmptr < t_end) && i)
					if(*tmptr++ == '\n')
						--i; }
			else {
				++i;
				while((tmptr >= t_start) && i)
					if(*--tmptr == '\n')
						--i;
				++tmptr; }
			end = tmptr;
			deflg = 0;
			while((chr = *cmd++) == ' '); }

		if(!start)
			start = tmptr;
		if(!end)
			end = tmptr; }
	while(chr == ',');

	if(end < start) {
		tmptr = start;
		start = end;
		end = tmptr; }
	while((end < t_end) && (*end++ != '\n'));
	j = end - start;

/* get command character */
	optr = cmd;							/* pointer to operands */
	while(*cmd == ' ')					/* skip trailing blanks */
		++cmd;
	switch(chr = tolower(chr)) {
		case 0 :			/* goto line */
			newpos = start;
			break;
		case '?':			/* find */
			if(deflg) {		/* default to cursor position */
				start = txtpos + cx + 1;
				end = t_end; }
			while(start < end) {
				if(partial_match(optr, start))			/* found it */
					end = newpos = start;
				++start; }
			if(!newpos)
				errmsg = "Not found";
			break;
		case 's':			/* replace */
			tmptr = edit_buff;
			chr = *cmd++;
			while(*cmd && (*cmd != chr))
				*tmptr++ = *cmd++;
			*tmptr = 0;
			if(!*cmd++) {
				errmsg = "Invalid search string\007";
				break; }
			i = Strlen(edit_buff);
			j = Strlen(cmd);
			while(start < end) {
				if(partial_match(edit_buff, start)) {		/* found it */
					if(i < j) {
						insert(start, k=j-i, cmd, j);
						end += k; }
					else {
						if(j < i) {
							delete(start, k=i-j);
							end -= k; }
						cpymem(start, cmd, j); }
					newpos = start;		/* point to last one found */
					start += j;
					CHANGED = -1; }
				else					/* not here, advance to next */
					++start; }
			if(newpos)
				--refresh;
			else
				errmsg = "Not found\007";
			break;
		case 't' :		/* tag lines */
			tag = start;
			tag1 = end - 1;
			tgflg = 0;			/* incase tagged were used */
			--refresh;
			break;
		case 'c' :		/* copy lines */
			if((txtpos > start) && (txtpos < end)) {
				errmsg = "Invalid destination\007";
				break; }
			insert(txtpos, j, start, j);
			--refresh;
			break;
		case 'm' :		/* move lines */
			if((txtpos >= start) && (txtpos < end)) {
				errmsg = "Invalid destination\007";
				break; }
			insert(txtpos, j, start, j);
			if(start >= txtpos)
				start += j;
		case 'd' :		/* delete lines */
			delete(start, j);
			--refresh;
			break;
		case '$':		/* shell command */
			start_output();
			if(window) {
				v_printf("$ %s\n", cmd);
				v_gotoxy(0, 1);
#ifdef UNIX
				v_close(); }
			system(*cmd ? cmd : "sh");
			if(window)
				v_init();
#else
			} system(cmd);
#endif
			end_output();
			break;
		case 'f' :		/* display file statistics */
			start_output();
			i = j = 0;
			for(tmptr = t_start; tmptr <= t_end; ++tmptr) {
				if(tmptr == start)
					j = i;
				if(*tmptr == '\n')
					++i; }
			v_printf("Filename: %s, %u Lines, %u Characters\n",
				fname, i, t_end - t_start);
			i = 0;
			for(tmptr = start; tmptr <= end; ++tmptr) {
				if(*tmptr == '\n')
					++i; }
			v_printf("Position: %u, %u Lines, %u Characters\n",
				j+1, i, end - start);
#ifdef UNIX
			v_printf("Buffer  : %u\n", Buffer_size);
#endif
			v_printf("There are%s unsaved changes.\n",
				(CHANGED) ? "" : " no");
			end_output();
			break;
		case 'l':		/* list lines */
			start_output();
			while(start < end)
				start = display_line(start);
			end_output();
			break;
		case 'p':		/* print lines */
			start_output();
			i = 1;
			for(tmptr = t_start; tmptr < end;) {
				if(tmptr >= start) {
					if(tmptr == txtpos)
						chr ='*';
					else if((tmptr >= tag) && (tmptr <= tag1))
						chr = '=';
					else
						chr = ' ';
					v_printf("%c%5u ", chr, i);
					display_line(tmptr); }
				while(*tmptr++ != '\n');
				++i; }
			end_output();
			break;
		case 'i':			/* input lines */
			start_output();
			v_printf("Input:\n");
			while(i=get_input("",edit_buff, EDIT_SIZE - 1)) {
				edit_buff[i++] = '\n';
				insert(start, i, edit_buff, i);
				start += i; }
			end_output();
			break;
		case 'r':			/* read file */
			if(fp = fopen(cmd, "rt")) {
				do {
					i = fread(edit_buff, 1, EDIT_SIZE, fp);
					insert(start, i, edit_buff, i);
					start += i; }
				while( i == EDIT_SIZE);
				fclose(fp);
				--refresh; }
			else
				errmsg = "Can't open input file\007";
			break;
		case 'w':			/* write file */
		case 'x':			/* write file & exit */
			if(deflg) {		/* default to entire file */
				start = t_start;
				end = t_end; }
			if(!*cmd)
				cmd = fname;
			if(fp = fopen(cmd, "wt")) {
				fwrite(start, 1, end - start, fp);
				fclose(fp);
				CHANGED = 0; }
			else {
				errmsg = "Can't open output file\007";
				break; }
			if(chr != 'x')
				break;
		case 'q' :			/* quit command */
			if(CHANGED && ('q' != tolower(*optr)))
				errmsg = "Unsaved changes, use 'qq' to quit anyway\007";
			else {
				if(window) {
					putchr(_CL);
#ifdef UNIX
					v_close();
#endif
				}
			exit(chr == 'q'); }
			break;
		case 'v':		/* change visual modes */
			cx = cy = 0;
			newpos = start;
			--refresh;
			if(window = !window)
				v_init();
			else {
#ifdef UNIX
				v_close();
#endif
				v_putc(_CL); }
			break;
		case 'h':		/* Horizontal TAB size + help request */
			if(i) {			/* Set htab size */
				tab_width = i;
				--refresh; }
			else {			/* Help request */
				start_output();
#ifdef UNIX
				putstr(Help1);
#else
				/* v_help() displays the help text from the code segment */
				/* avoiding using up the data segment for help text */
				v_help();
#endif
				end_output(); }
			/*	errmsg = "Invalid tab size\007"; */
			break;
		default:
			errmsg = "Unknown command\007"; }

	if(tgflg && !errmsg) {		/* clear tags if no errors */
			tag = tag1 = 0;
			refresh = window; }
}

/* prepare for large output command */
start_output()
{
	if(window)
		putchr(_CL);
}

/* terminate large output command */
end_output()
{
	if(window) {
		v_gotoxy(0, Lines1);
		putstr("Press any key to continue... ");
		v_getc();
		--refresh; }
}

/* get a command line */
get_input(prompt, dest, length)
	unsigned char *prompt, dest[];
	unsigned length;
{
	register unsigned i;
	register unsigned char chr;

	i = 0;
	v_printf("%s", prompt);
	if(window) {		/* full screen mode */
		do {
			if(/*_KDP*/_KBS == (chr = v_getc())) {
				if(i) {
					putstr("\010 \010");
					--i; } }
			if((i < length) && !(chr & 0x80))
				display_chr(dest[i++] = chr); }
		while((chr != /*_K1*/_KKP) && (chr != /*_K15*/_K10));
		dest[i] = 0;
		if(!*prompt)
			putchr('\n'); }
	else {				/* line by line mode */
		fflush(stdout);
		fgets(dest, length, stdin);
		while(dest[i]) {
			if(dest[i] == '\n') {
				dest[i] = 0;
				break; }
			++i; } }
	return(i);
}

/*
 * Detect a partial match
 */
partial_match(string, source)
	unsigned char *string, *source;
{
	while(*string)
		if(*string++ != *source++)
			return 0;
	return 1;
}

/*
 * Write a character to the output device
 */
putchr(chr)
	unsigned char chr;
{
	if(window)
		v_putc(chr);
	else
		putc(chr, stdout);
}

/*
 * Write a string to the output device
 */
putstr(ptr)
	unsigned char *ptr;
{
	register unsigned char c;

	while(c = *ptr++)
		putchr(c);
}

/*
 * Formatted print to console device.
 */
#ifdef _MICROC_
register v_printf(unsigned args)
{
	unsigned char tmp[200];
	_format_(nargs()*2+&args, tmp);
	putstr(tmp);
}
#else
void v_printf(unsigned char *format, ...)
{
	va_list ap;
	unsigned char outstk[11], *ptr, justify, zero, minus, c;
	unsigned width, value, i;

	va_start(ap, format);

	while(c = *format++) {
		if(c == '%') {						/* format code */
			c = *format++;
			*(ptr = &outstk[10]) = justify = minus = width = value = i = 0;
			zero = ' ';
			if(c == '-') {					/* left justify */
				--justify;
				c = *format++; }
			if(c == '0')					/* leading zeros */
				zero = '0';
			while(isdigit(c)) {				/* field width specifier */
				width = (width * 10) + (c - '0');
				c = *format++; }

			switch(c) {
				case 'u' :					/* unsigned number */
					i = 10;
					goto donum;
				case 'x' :					/* hexidecimal number */
					i = 16;
		donum:		value = va_arg(ap, unsigned);
					break;
				case 'c' :					/* character data */
					*--ptr = va_arg(ap, int);
					break;
				case 's' :					/* string */
					ptr = va_arg(ap, char*);
					break;
				default:					/* all others */
					*--ptr = c; }

			if(i)		/* for all numbers, generate the ASCII string */
				do {
					if((c = (value % i) + '0') > '9')
						c += 7;
					*--ptr = c; }
				while(value /= i);

/* output sign if any */
			if(minus) {
				putchr('-');
				if(width)
					--width; }

/* pad with 'zero' value if right justify enabled  */
			if(width && !justify) {
				for(i = Strlen(ptr); i < width; ++i)
					putchr(zero); }

/* move in data */
			i = 0;
			value = width - 1;
			while((*ptr) && (i <= value)) {
				putchr(*ptr++);
				++i; }

/* pad with 'zero' value if left justify enabled */
			if(width && justify) {
				while(i < width) {
					putchr(zero);
					++i; } } }
		else
/* not a format code, simply display the character */
			putchr(c); }
}
#endif

#ifdef UNIX
/*
 * UNIX I/O routines
 * (see VIDEO.ASM for DOS I/O functions)
 */
int
	tty;					// Handle for "/dev/tty"
unsigned
	X, Y;
unsigned char
	outf;					// flush pending (output has occured)
struct termios
	oldkey,					// Original tty settings
	newkey;					// New tty settings

/*
 * Initialize the video sub-system
 */
v_init()
{
	tty = open("/dev/tty", O_RDWR | O_NOCTTY | O_NONBLOCK); //set the user console port up
	tcgetattr(tty, &oldkey); // save current port settings   //so commands are interpreted right for this program
	// set new port settings for non-canonical input processing  //must be NOCTTY
	newkey.c_cflag = B38400 | CRTSCTS | CS8 | CLOCAL | CREAD;
	newkey.c_iflag = IGNPAR;
	newkey.c_oflag = 0;
	newkey.c_lflag = 0;       //ICANON;
	newkey.c_cc[VMIN]=1;
	newkey.c_cc[VTIME]=0;
	tcflush(tty, TCIFLUSH);
	tcsetattr(tty, TCSANOW, &newkey);
	X = Y = 0;
}

/*
 * Close the video sub-system
 */
v_close()
{
	tcsetattr(tty, TCSANOW, &oldkey);
	close(tty);
}

/*
 * Write character to tty with output translactions
 */
void v_putc(unsigned char c)
{
	unsigned x;
	outf = 255;
	switch(c) {
	case _CL: fputs("\x1B[H", stdout);					// Clear screen
		X = Y = 0;
	case _CD: fputs("\x1B[J", stdout);		return;		// Clear to EOS
	case _CE: fputs("\x1B[K", stdout);		return;		// Clear to EOL
	case _SO: fputs("\x1B[7m", stdout);		return;		// Hilight-ON
	case _SE: fputs("\x1B[m", stdout);		return;		// Hilight-OFF
	case'\n': c = '\r';
	case'\r':	++Y; X = 0;					break;		// \n = \r\n
	default	:	++X; }
	if(Y >= Lines) return;
	x = Cols;
	if(Y >= Lines1)
		--x;
	if(X < x)
		putc(c, stdout);
}

/*
 * Position cursor on TTY
 */
v_gotoxy(unsigned x, unsigned y)
{
	v_printf("\x1B[%u;%uH", (Y = y)+1, (X = x)+1);
}

/*
 * Read single character from TTY
 */
unsigned char v_readc()
{
	unsigned char c;
	if(!fread(&c, 1, 1, stdin)) {
		v_close();
		fputs("Read error", stdout);
		exit(0); }
	return c;
}

/*
 * Get character/key from TTY with input translations
 */
v_getc()
{
	unsigned v, v1;
	unsigned char c;

	if(outf) {
		fflush(stdout);
		outf = 0; }

noesc:
	switch(c = v_readc()) {
	default: return c;
	case '\r' : return '\n';				// Translate return to NL
	case '\b' :								// Backspace
	case 0x7F : return /*_KDP*/_KBS;				// Delete
	case 0x1B : switch(c = v_readc()) {		// Escape (lead-in)
		default: return c;
		case 'O' :
			switch(c = v_readc()) {
			case 'P' :	return /*_K4*/_K1;			// F1
			case 'Q' :	return /*_K5*/_K2;			// F2
			case 'R' :	return /*_K6*/_K3;			// F3
			case 'S' :	return /*_K7*/_K4;			// F4
			case '2' : switch(c = v_readc()) {
				case 'P' :
				case 'R' : return /*_K12*/_CLA;		// ^Left
				case 'Q' :
				case 'S' : return /*_K11*/_CRA;		// ^Right
			} } goto noesc;
		case '[' :			// Further lead-in
			v = v1 = 0;
	next:	switch(c = v_readc()) {
			default:
				if(isdigit(c)) {
					v = (v * 10) + (c - '0');
					goto next; }
				goto noesc;
			case 'A' :	return /*_KUP*/_KUA;		// Up
			case 'B' :	return /*_KDO*/_KDA;		// Down
			case 'C' :	return (v == 5) ? /*_K11*/_CRA : /*_KND*/_KRA;	// Right (Wd-Right OS-X)
			case 'D' :	return (v == 5) ? /*_K12*/_CLA : /*_KBS*/_KLA;	// Left  (Wd-Left OS-X)
			case 'H' :	return /*_KPL*/_KHO;		// Home
			case 'F' :	return /*_KPR*/_KEN;		// End
			case 'Z' :	return /*_K3*/_KIN;			// Insert (OS-X)
			case ';' :	v1 = v; v = 0; goto next;
			case '[' :			// Further lead-in
				switch(v_readc()) {
				case 'A' :	return /*_K4*/_K1;		// F1
				case 'B' :	return /*_K5*/_K2;		// F2
				case 'C' :	return /*_K6*/_K3;		// F3
				case 'D' :	return /*_K7*/_K4;		// F4
				case 'E' :	return /*_K8*/_K5; }	// F5
				goto noesc;
			case '~' :
				switch((v1 << 8) | v) {
				case 0x0505:	return /*_KHO*/_CPU;	// ^PgUp
				case 0x0605:	return /*_KEN*/_CPD; }	// ^PgDn
				switch(v) {
				default: goto noesc;
				case 1 :	return /*_KPL*/_KHO;	// Home
				case 2 :	return /*_K3*/_KIN;		// INS
				case 3 :	return /*_KDC*/_KDL;	// DEL
				case 4 :	return /*_KPR*/_KEN;	// End
				case 5 :	return /*_KPU*/_KPU;	// PgUp
				case 6 :	return /*_KPD*/_KPD;	// PgDn
				case 15 :	return /*_K8*/_K5;		// F5
				case 17 :	return /*_K9*/_K6;		// F6
				case 18 :	return /*_K10*/_K7;	// F7
				case 19 :	return /*_KCL*/_CHO;	// F8
				case 20 :	return /*_K2*/_KKM;		// F9
				case 21 :	return /*_K1*/_KKP;		// F10
				case 23 :	return /*_KHO*/_CPU;	// F11 (Home)
				case 24 :	return /*_KEN*/_CPD;	// F12 (End)
				case 25 :
				case 28 :	return /*_K12*/_CLA;	// WD-Left
				case 26 :
				case 29 :	return /*_K11*/_CRA;	// WD-Right
	} } } }
}
#endif
