/*
 * main.c
 */
#include "F103reg.h"
#include <stdarg.h>
#include <string.h>
#include <setjmp.h>

typedef unsigned char	U8;
typedef unsigned short	U16;
typedef	unsigned		U32;
typedef int				S32;

/* --- SOME USEFUL MACROS --- */

// Macro to convert a bit number into a mask, eg: B(8) becomes 0x0100
#define	B(x)		(1<<x)			// Bit# to mask

// Macros to easily set/clear bits. Note, that due to the way the preprocessor
// evaluates macro arguments, a second level is needed if you want to specify
// both the port and bit with a single macro definition, therefore:
//	Use setio and clrio if you want specify both the port and the bit.
//		eg: 	setio(C,6);		clrio(A,12);
#define	setio(p,b) GPIO##p##_BSRR=(1<<b)
#define clrio(p,b) GPIO##p##_BRR=(1<<b)
#define tstio(p,b) (GPIO##p##_IDR&(1<<b))

#define	Setio(p,b) (GPIO##p##_BSRR=b)
#define Clrio(p,b) (GPIO##p##_BRR=b)
#define Tstio(p,b) (GPIO##p##_IDR&b)

unsigned
	Tick;				// 1ms clock tick
unsigned short
	DBmask,				// Debug output mask
	TKmark;				// Tick marker
unsigned char
	*Ptr,				// General pointer
	Buffer[256];		// General buffer
jmp_buf
	jmpsav;

void BG(void);
void Debug(unsigned short mask, const char *format, ...);
int ADCread(unsigned ch);
void udelay(unsigned short t);
void mdelay(unsigned short t);

#define	DB_ADC	0x0001		// Debug ADC
#define	DB_STEP	0x0002		// Debug STEP

#include "library.ch"
#include "step.ch"
#include "monitor.ch"

void Debug(unsigned short mask, const char *format, ...)
{
	char buffer[128];
	if(DBmask & mask) {
		va_list a;
		va_start(a, format);
		_format_(buffer, 127, format, a);
		va_end(a);
		Puts(buffer); }
}

// Set GPIO mode
//	in = input control regist er
// bit = Bit to change
//   m = Mode: 0=Input	1=Out10mhz	2=Out2mhz		3=Out50mhz
//	 c = Config:
//		I: 0=Analog		1=Floating	2=Pullup/down	3=n/a
//		O: 0=PushPull	1=OpenDrain	2=AltPushPull	2=AltOpenDrain
// Pull Up/Down selected via GPIOx_ODR register.
unsigned GPIOmode(unsigned in, unsigned bit, unsigned m, unsigned c)
{
	if((bit *= 4) > 30)
		bit -= 32;
	m |= (c << 2);		// Combine C & m
	in &= ~(0x0F << bit);
	in |= m << bit;
	return in;
}

// Background processing
void BG(void)
{
	unsigned short a;

	// Advance system time
	a = TIM4_CNT - TKmark;
	if(a < 1000)
		return;

	++Tick;
	TKmark += 1000;
	IWDG_KR = 0xAAAA;		// Kick dog

	// Handle stepper
	if((Tick - Slast) >= Srate) {
		Slast = Tick;
		switch(Sstate) {
		default: return;
		case 1 :		// Ramp UP
			Srate -= RSTEP;
			if(!--RampUp)
				Sstate = 2;
			break;
		case 2 :		// Ramp LEVEL
			if(!--RampLevel)
				Sstate = RampDown ? 3 : 4;
			break;
		case 3 :		// Ramp DOWN
			Srate += RSTEP;
			if(!--RampDown)
				Sstate = 4;
			break;
		case 4 :		// Shutting down
			SdriveMask = ~(0xFF << 3);	// All motors OFF
			switch(Sactive) {
			case 1 : Sposition1 = Sposition;	break;
			case 2 : Sposition2 = Sposition;	}
			Sstate = 5;
			return;
		case 5 :		// Going idle
			GPIOB_ODR = Smirror & SdriveMask;
			Sstate = 0;
			return; }
		if(Sdir < 0) {		// Stepping IN
			if(GPIOA_IDR & SzeroMask) {
				Slimit = 'Z';
				Sposition = 0;
				Sstate = 4;
				return; } }
		else {				// Stepping OUT
			if(GPIOA_IDR & SendMask) {
				Slimit = 'E';
				Sstate = 4;
				return; } }
		// Write to stepper
		GPIOB_ODR = (Smirror = StepTable[StepPos++ & 7]) & SdriveMask;
		Sposition += Sdir; }
}

// Delay for a number of microseconds
void udelay(unsigned short t)
{
	unsigned short a, b;
	static unsigned char c;
	b = TIM4_CNT;
	for(;;) {
		a = TIM4_CNT - b;
		if(a > t)
			return;
		if(!--c)
			BG(); }
}

// Delay for a number of milliseconds
void mdelay(unsigned short t)
{
	while(t) {
//		poll();
		udelay(1000);
		--t; }
}

/*
 * Read an analog-to-digital converter
 */
int ADCread(unsigned ch)
{
	unsigned x;

	if(!(ADC1_CR2 & B(0))) {	// ADC is powered-down
		ADC1_CR2 = B(23)|B(20)|(7<<17)|B(0);
		udelay(1000);
		ADC1_CR1 = 0;
		ADC1_SMPR1	= 0x00FFFFFF;
		ADC1_SMPR2	= 0xEFFFFFFF;
		ADC1_HTR	= 0x00000FFF;
		ADC1_LTR	= 0x00000000;
		ADC1_CR2 |= B(2);		// Start calibration
		x = 10;
		do {
			udelay(1000);
			Debug(DB_ADC, "\n%08x", ADC1_CR2);
			if(!--x) {
				ADC1_CR2 = 0;
				Debug(DB_ADC, "\nADC cal fail");
				return -1; } }
		while(ADC1_CR2 & B(2)); }

	x = ADC1_DR;				// Insure previous cleared
	ADC1_SQR1 = 0<<20;			// 1 conversion
	ADC1_SQR2 = 0;
	ADC1_SQR3 = ch & 0x1F;		// Channel number
	ADC1_CR2 |= B(22);			// Start conversion
	x = 100;
	do {
		udelay(10);
		if(!--x) {
			ADC1_CR2 = 0;
			Debug(DB_ADC, "\nADC read fail");
			return -2; } }
	while(!(ADC1_SR & B(1)));	// Wait till end
	return ADC1_DR & 0xFFF;		// Return value
}

int main()
{
	unsigned a;

	// Switch to high-speed on external crystal
	RRC_CR |= B(16);			// Enable HSE
	while(!(RRC_CR & B(17)));	// Wait for stable
	RRC_CFGR =	// 48Mhz
			(10 << 18)			// PLL x12 (4Mhz * 12) 48Mhz
			| B(17)				// PLL = HSE/2 (8Mhz = 4Mhz)
			| B(16)				// PLL input is HSE
			| (3 << 14)			// ADC = /8
			| (0 << 11)			// APB2-PCLK2 = /1
			| (4 << 8)			// APB1-PCLK1 = /2
			| (0 << 4)			// AHB = /1
			;
	RRC_CR |= B(24);			// Enable PLL
	while(!(RRC_CR & B(25)));	// Wait for PLL stable
	RRC_CFGR |= (3<<14)|2;		// ADC prescale=8, Sysclk=PLL

	// Enable peripheral clocks
	RRC_APB1ENR |= B(17)|B(2);								// USART2, TIM4
	RRC_APB2ENR |= B(14)|B(10)|B(9)|B(4)|B(3)|B(2)|B(0);	// USART1,ADC1,GPIOC,GPIOB,GPIOA,AFIO

	AFIO_MAPR = 0x02000000;			// Turn off JTAG, enable B3, B4 and A15

	GPIOA_ODR = B(15)|B(11)|B(10)|B(8)|B(7)|B(6);
	a = GPIOA_CRH;
	// mode(0=Input 1=Out10mhz 2=Out2mhz 3=Out50mhz), cfg:
	//	 in:	0=Analog		1=Floating		2=Pullup		3=n/a
	//	out:	0=PushPull		1=OpenDrain		2=AltPushPull	3=AltOpenDrain
	a = GPIOmode(a,15, 0, 2);		// 15 I,PU		Digital in
	a = GPIOmode(a,12, 0, 2);		// 12 I,PU		Digital in
	a = GPIOmode(a,11, 0, 2);		// 11 I,PU		Digital in
	a = GPIOmode(a,10, 0, 1);		// 10 I,F		Main serial RX
	a = GPIOmode(a, 9, 1, 2);		//  9 O10,APP	Main serial TX
	a = GPIOmode(a, 8, 0, 2);		//	8 I, PU
	GPIOA_CRH = a;
	a = GPIOA_CRL;
	a = GPIOmode(a, 7, 0, 2);		// 7 I,PU		Digital in
	a = GPIOmode(a, 6, 0, 2);		// 6 I,PU		Digital in
	a = GPIOmode(a, 5, 0, 0);		// 5 I,A		Analog in
	a = GPIOmode(a, 4, 0, 0);		// 4 I,A		Analog in
	a = GPIOmode(a, 3, 0, 0);		// 3 I,A		Analog in
	a = GPIOmode(a, 2, 0, 0);		// 2 I,A		Analog in
	a = GPIOmode(a, 1, 0, 0);		// 1 I,A		Analog in
	a = GPIOmode(a, 0, 0, 0);		// 0 I,A		Analog in
	GPIOA_CRL = a;

	GPIOB_ODR = Smirror = B(15)|B(14)|B(13)|B(1)|B(0);
	// mode(0=Input 1=Out10mhz 2=Out2mhz 3=Out50mhz), cfg:
	//	 in:	0=Analog		1=Floating		2=Pullup		3=n/a
	//	out:	0=PushPull		1=OpenDrain		2=AltPushPull	3=AltOpenDrain
	a = GPIOB_CRH;
	a = GPIOmode(a,15, 2, 1);		// 15 O2,OD		
	a = GPIOmode(a,14, 2, 1);		// 14 O2,OD		
	a = GPIOmode(a,13, 2, 1);		// 13 O2,OD		
	a = GPIOmode(a,12, 2, 0);		// 12 O2,PP		Solenoid2
	a = GPIOmode(1,11, 2, 0);		// 11 O2,PP		Solenoid1
	a = GPIOmode(a,10, 2, 0);		// 10 O2,PP		Motor2 Phase4
	a = GPIOmode(a, 9, 2, 0);		//  9 O2,PP		Motor2 Phase3
	a = GPIOmode(a, 8, 2, 0);		//  8 O2,PP		Motor2 Phase2
	GPIOB_CRH = a;
	a = GPIOB_CRL;
	a = GPIOmode(a, 7, 2, 0);		// 7 O2,PP		Motor2 Phase1
	a = GPIOmode(a, 6, 2, 0);		// 6 O2,PP		Motor1 Phase4
	a = GPIOmode(a, 5, 2, 0);		// 5 O2,PP		Motor1 Phase3
	a = GPIOmode(a, 4, 2, 0);		// 4 O2,PP		Motor1 Phase2
	a = GPIOmode(a, 3, 2, 0);		// 3 O2,PP		Motor1 Phase1
	GPIOB_CRL = a;

	GPIOC_ODR = B(13);
	// mode(0=Input 1=Out10mhz 2=Out2mhz 3=Out50mhz), cfg:
	//	 in:	0=Analog		1=Floating		2=Pullup		3=n/a
	//	out:	0=PushPull		1=OpenDrain		2=AltPushPull	3=AltOpenDrain
	a = GPIOC_CRH;
	a = GPIOmode(a,13, 2, 1);		// 13 O2,OD		LED
	GPIOC_CRH = a;

	// Setup USART1
	USART1_GTPR = 1;					// Set Prescale-1
//	USART1_BRR = (2500*16) + 0;			// 48000000/1200/16= 2500.00
//	USART1_BRR = (312*16) + 8;			// 48000000/9600/16= 312.50
	USART1_BRR = (156*16) + 4;			// 48000000/19200/16= 156.25
//	USART1_BRR = (78*16) + 2;			// 48000000/38400/16= 78.125
	USART1_CR1 = B(13) | B(3) | B(2);	// Enable UART, TX and RX

	// Setup TIMER4 for simple timeing
	// Advance rate is 48,000,000 / PSC
	TIM4_PSC = 48-1;				// 48M/48 = 1M count rate
	TIM4_ARR = 0x10000-1;			// Nice round (binary) upcount
	TIM4_CR1 = 0x0001;				// Enable counter

	IWDG_KR = 0xCCCC;

restart:
	TKmark = TIM4_CNT;
	mdelay(500);

	monitor();
	goto restart;
}

// Handlers to identify fault sources (debug loops)
void NMI_Handler(void) { for(;;); }
void HardFault_Handler(void) { for(;;); }
void MemManage_Handler(void) { for(;;); }
void BusFault_Handler(void) { for(;;); }
void UsageFault_Handler(void) { for(;;); }
void SVC_Handler(void) { for(;;); }
void DebugMon_Handler(void) { for(;;); }
void PendSV_Handler(void) { for(;;); }
void SysTick_Handler(void) { for(;;); }
