/*
 * UT81 ScopeMeter Communications Test Program
 *
 * Allows for easy examination of the status records transmitted by the UT81.
 *
 * - This is a 16-bit DOS application.
 * - Requires RS-232 connection to UT81 (see RS232.TXT)
 *
 * Compile with DDS Micro-C/PC available from www.dunfield.com
 * Compile command: cc ut81test -fop
 *
 * Dave Dunfield, Feb 2010
 */
#include <stdio.h>			// Standard IO
#include <window.h>			// Text windowing
#include <comm.h>			// Serial comunications

#define	TICK	peekw(0x40,0x6C)	// BIOS clock tick

#define	BASEMSG		41
#define	SCOPEMSG	320

unsigned
	Cport = 1;						// Comm port
	Ctop;							// Top of capture buffer

unsigned char
	Msg,							// Message displayed flag
	Capture[BASEMSG+SCOPEMSG];		// Capture buffer

struct WINDOW
	*mwin,							// Main window
	*swin;							// Status window

// Command line help text
unsigned char Help[] = { "\n\
UT81 ScopeMeter Communications Test Program\n\n\
Use:	UT81TEST [options]\n\n\
opts:	C=1-4	set Com port				[1]\n\
\nDave Dunfield - "#__DATE__"\n" };

/*
 * Display message in status window
 */
register message(unsigned args)
{
	unsigned char buf[81];

	_format_(nargs() * 2 + &args, buf);
	w_clwin(swin);
	w_puts(buf, swin);
	Msg = 255;
}

/*
 * Display content of capture buffer
 */
void display(void)
{
	int c;
	unsigned i;
	unsigned char f;

	for(i=f=0; i < Ctop; ++i) {
		c = Capture[i];
		if((c >= ' ') && (c < 0x7F) && (i >= 21) && (i <= 38)) {
			*W_OPEN = NORMAL;
			wputc(c);
			continue; }
		*W_OPEN = (++f & 1) ? 0x17 : 0x13;
		wprintf("%02x", c); }
	*W_OPEN = NORMAL;
	wputc('\n');
}

/*
 * Display scale on screen
 */
void scale(void)
{
	unsigned i, j;
	static unsigned char Colors[] = { 0x60, 0x20, 0x50, 0x40 };

	for(i=j=0; j < 79; ++i) {
		*W_OPEN = Colors[(i/10)&3];
		wputc((i % 10)+'0');
		++j;
		if( ((i < 21) || (i > 38)) || (i == 30) ) {
			wputc(' ');
			++j; } }
	*W_OPEN = NORMAL;
	wputc('\n');
}

/*
 * Display main prompt
 */
void prompt(void)
{
	w_clwin(swin);
	w_puts("UT81:  SPACE=Capture 'S'=Scale 'C'=Clear ESC=Exit ", swin);
	Msg = 0;
}

/*
 * Perform a transaction with the ScopeMeter
 */
int transaction(void)
{
	int c;
	unsigned h, t, t1, t2;
	unsigned char f;

	message("Communicating with UT81");
	while(Ctestc() != -1);		// Flush input
	Cputc(0x5A);
	Ctop = f = 0;
	t = TICK;
	t1 = 2*18;
	h = BASEMSG;
more:
	while(Ctop < h) {
		t2 = TICK;
		if((c = Ctestc()) != -1) {
			if(Ctop || (c == 0x5A)) {
				Capture[Ctop++] = c;
				t = t2;
				f = 255;
				t1 = 5; }
			continue; }
		if(kbtst() == 0x1B)
			goto abort;
		if((t2-t) > t1) {		// Timeout
			if(!f) {
				message("Waiting for UT81 -press ESC to cancel");
				f = 255;
				t1 = 43*18;
				continue; }
abort:		message("Transfer aborted!");
			Ctop = 0;
			return 255; } }

	if((Capture[5] == 0x59) && (h == BASEMSG)) {
		h = BASEMSG+SCOPEMSG;		// Add scope data
		goto more; }

	prompt();
	return 0;
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	int c;
	unsigned char *p;
	unsigned char f;

	for(c=1; c < argc; ++c) {
		p = argv[c];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case 'C=' : Cport = atoi(p);	continue;
		} abort(Help); }

	if((Cport < 1) || (Cport > 4))
		abort(Help);

	// Open Communications port
	if(Copen(Cport, _9600, PAR_NO|DATA_8|STOP_1, SET_DTR|OUTPUT_2))
		abort("Cannot open COM port");
	Cflags |= TRANSPARENT;

	// Open display windows
	swin = wopen(0,24, 80,  1, WSAVE|WCOPEN|0x17);
	mwin = wopen(0, 0, 80, 24, WSAVE|WCOPEN|WSCROLL|NORMAL);
	wcursor_off();
	scale();
	message("UT81 ScopeMeter Communications Test Program - Dave Dunfield - "#__DATE__"");

	// Main processing loop
	f = 0;
	for(;;) {
		if(c = wtstc()) {
			if(Msg)
				prompt(); }
		switch(c) {
		case 0x1B:			// Exit program
			wclose();
			wclose();
			Cclose();
			return;
		case ' ' :			// Capture data from meter
			transaction();
			continue;
		case 's' :			// Display a Scale
		case 'S' :
			scale();
			continue;
		case 'c' :			// Clear the screen
		case 'C' :
			wclwin();
			continue; }

		// If any data in buffer, display
		if(Ctop) {
			if(f)
				wputc('\n');
			display();
			Ctop = f = 0; }

		// Display any extranious input data
		if((c = Ctestc()) != -1) {
			wprintf("{%02x}", c);
			f = 255; } }
}
