/*
 * UT81 ScopeMeter demonstration program
 *
 * Implements display/storage of the UT81 ScopeMeter data
 *
 * - This is a 16-bit DOS application.
 * - Requires RS-232 connection to UT81 (see RS232.TXT)
 *
 * Compile with DDS Micro-C/PC available from www.dunfield.com
 * Compile command: cc ut81 -fop
 *
 * Dave Dunfield, Feb 2010
 */
#include <stdio.h>		// Standard IO
#include <hrg.h>		// High-res graphics
#include <keys.h>		// PC keyboard defs
#include <comm.h>		// Serial communications

#define	TICK peekw(0x40,0x6C)	// BIOS clock tick

#define	BASEMSG		41			// Base message width

// Screen layout parameters
#define	SCOPEW		320			// Width of SCOPE (dots)
#define	SCOPEH		384			// Height of SCOPE
#define	SCOPEX		10			// X position of scope display
#define	SCOPEY		60			// Y position of scope display
#define	LABELX		360			// X position of label display
#define	LABELY		120			// Y position of label display
#define	LABELH		40			// Height of label display
#define	LABELX1		(LABELX+90)	// X position of label fields
#define	PROMPTX		360			// X position of prompts
#define	PROMPTY		370			// Y position of prompts

unsigned
	Cport,						// COM port
	Ctop;						// Top of capture buffer

FILE
	*fp;						// General file pointer

unsigned char
	Msg,						// Message displayed flag
	Dscope,						// Scope display active
	Filename[65],				// Filename storage
	Capture[BASEMSG+SCOPEW];	// Capture buffer

unsigned
	Scope[SCOPEW];				// Scope display storage

// Lables shown on screen
unsigned char *label[] = {
	"Function",					// 0
	"Range",					// 1
	"TimeBase",					// 2
	"Trigger",					// 3
	0 };

// Table of ScopeMeter function names
unsigned char *Function[] = {
		"VOLT", "AMP", "HZ", "CAP", "OHM", "CONT", "DIODE" };

// Table of ScopeMeter timebase values & units
unsigned Time_value[] = { 1, 2, 5, 10, 20, 50, 100, 200, 500 };
unsigned char *Time_unit[] = { "ns", "us", "ms", "s" };

// Table of ScopeMeter trigger modes
unsigned char *Trigger_mode[] = { "AUTO", "NORM", "SHOT" };

// Command line help text
unsigned char Help[] = { "\n\
UT81 ScopeMeter Demo\n\n\
Use:	UT81 [filename] [options]\n\n\
opts:	C=1-4	set Com port				[none]\n\
\nDave Dunfield - "#__DATE__"\n" };

/*
 * Clear the message display area
 */
void clearmsg(void)
{
	if(Msg)
		hrg_fbox(0, 480-10, 640, 10, Msg = 0);
}

/*
 * Display a message
 */
register message(unsigned args)
{
	unsigned char buf[128];
	_format_(nargs() * 2 + &args, buf);
	clearmsg();
	hrg_puts(0, 480-10, 7, buf);
	Msg = 255;
}

/*
 * Prompt for input in message area
 */
int input(unsigned char *prompt, unsigned char *dest, unsigned length)
{
	int c;
	unsigned i, x, y;
	unsigned char local[80];

	strcpy(local, dest);
top:
	message("%s?", prompt);
	x = (strlen(prompt)*8) + 16;
	y = 480-10;
	for(i=0; c=local[i]; ++i) {
		hrg_putc(x, y, 7, c);
		x += 8; }
	for(;;) {
		hrg_fbox(x, y, 8, 8, 7);
		c = kbget();
		hrg_fbox(x, y, 8, 8, 0);
		switch(c) {
		case '\b' :
			if(i) {
				--i;
				x -= 8; }
			continue;
		case '\n' :
		case '\r' :
			local[i] = 0;
			strcpy(dest, local);
			clearmsg();
			return 255;
		case _HO :
			*local = 0;
			goto top;
		case 0x1B:
			clearmsg();
			return 0; }
		if((c >= ' ') && (c < 0x7F) && (i < length)) {
			hrg_putc(x, y, 7, local[i++] = c);
			x += 8; } }
}

/*
 * Draw the scope screen grid
 */
void draw_grid(void)
{
	unsigned x, y;

	// Draw horizontal grid lines
	for(y=(SCOPEY+32); y < (SCOPEY+SCOPEH); y += 32) {
		for(x=SCOPEX; x < (SCOPEX+SCOPEW); x += 8)
			hrg_plot(x, y, 1); }

	// Draw vertical grid lines
	for(x=(SCOPEX+40); x < (SCOPEX+SCOPEW); x += 40) {
		for(y=SCOPEY; y < (SCOPEY+SCOPEH); y += 8)
			hrg_plot(x, y, 1); }

	Dscope = 255;
}

/*
 * Get string field, indicate if out of bounds
 */
unsigned char *string(unsigned char *data[], unsigned size, unsigned index)
{
	if(index >= size)
		return "???";
	return data[index];
}

/*
 * Range is complicated to determine as it's based on function
 * and auto/manual - so we dedicate a function to working it out.
 */
void range(unsigned x, unsigned y)
{
	unsigned char *p1, *p2, r;

	static unsigned Rvalue1[] = {	// Scope ranges
		20, 50, 100, 200, 500,						// mv/ua
		1, 2, 5, 10, 20, 50, 100, 200, 500,			// v/ma
		2, 5 };										// a
	static unsigned Rvalue2[] = {	// Meter ranges
		400, -1, 400,			// 00 01 02
		4, -1, 4,				// 03 04 05
		40, -1, 40,				// 06 07 08
		400, -1, 400,			// 09 0A 0B
		1000, 1000, 4,			// 0C 0D 0E
		-1, 10 };				// 0F
	static unsigned Rvalue3[] = {	// Ohmmeter range
		400,					// Ohm
		4, 40, 400,				// kOhm
		4, 40 };				// mOhm

	switch(Capture[6]) {
	case 0x80:			// Scope volts
	case 0x81:			// Scope amps
	case 0x82:			// Scope Hz
		r = Capture[11];
		if(r < 5)		{ p1 = "mv"; p2 = "ua"; }
		else if(r < 14)	{ p1 = "v" ; p2 = "ma"; }
		else			{ p1 = "?" ; p2 = "a";  }
		break;
	case 0x00:			// Meter volts
	case 0x01:			// Meter amps
	case 0x02:			// Meter Hz
		r = Capture[11];
		if(r < 3)		{ p1 = "mv"; p2 = "ua"; }
		else if(r < 12)	{ p1 = "v" ; p2 = "ma"; }
		else 			{ p1 = "v" ; p2 = "a";  }
		break;
	case 0x04:			// Meter Ohms
		r = Capture[18];
		if(r < 1)		{ p1 = "ohm"; }
		else if(r < 4)	{ p1 = "kohm"; }
		else			{ p1 = "Mohm"; }
	}

	switch(Capture[6]) {
	case 0x00:		// Meter volts
	case 0x02:		// Meter Hz
		hrg_printf(x, y, 7, "%s %u %s", Capture[8] ? "AUTO" : "RANG", Rvalue2[r], p1);
		break;
	case 0x01:		// Meter amps
		hrg_printf(x, y, 7, "%s %u %s", Capture[8] ? "AUTO" : "RANG", Rvalue2[r], p2);
		break;
	case 0x04:
		hrg_printf(x, y, 7, "%s %u %s", Capture[19] ? "AUTO" : "RANG", Rvalue3[r], p1);
		break;
	case 0x80:		// Scope volts
	case 0x82:		// Scope Hz
		hrg_printf(x, y, 7, "%s %u %s", Capture[8]?"AUTO":"RANG", Rvalue1[r], p1);
		break;
	case 0x81:		// Scope AMPS
		hrg_printf(x, y, 7, "%s %u %s", Capture[8]?"AUTO":"RANG", Rvalue1[r], p2);
	}
}

/*
 * Display scope data
 */
void display(void)
{
	unsigned i, oy, y;
	char *p;
	static unsigned
		om = (SCOPEH/2)+SCOPEY,
		ot = (SCOPEH/2)+SCOPEY,
		ob = (SCOPEW/2)+SCOPEX;

	// Check capture validity and report problems
	if(Capture[0] != 0x5A)
		message("***Data does not begin with 0x5A");
	else switch(Ctop) {
		default:
			message("***Data should be 41 or 361 bytes (length is %u)", Ctop);
		case 41 :
		case 361 : ; }

	hrg_fbox(LABELX1, LABELY, (639-LABELX1)-2, 4*LABELH, 0);

	// Display function
	hrg_puts(LABELX1, LABELY+0, 7, string(Function, sizeof(Function)/2, i=Capture[6]&127));
	switch(i) {
	case 0 :
	case 1 :
		hrg_puts(LABELX1+(5*8), LABELY, 7, Capture[10] ? "AC" : "DC");
		break;
	case 2 :
		hrg_puts(LABELX1+(5*8), LABELY, 7, Capture[10] ? "DUTY" : "FREQ"); }
	hrg_puts(LABELX1+(10*8), LABELY, 7, Capture[20] ? "REL" : "");
	hrg_puts(LABELX1+(14*8), LABELY, 7, Capture[7] ? "" : "HOLD");

	// Display range
	range(LABELX1, LABELY+(1*LABELH));

	// Display scope data if available
	if(Ctop > 41) {
		// Display readings
		hrg_printf(SCOPEX+20, SCOPEY-11, 7, "%-18s%18s", Capture+21, Capture+31);

		// Display Timebase
		i = Capture[13];
		hrg_printf(LABELX1, LABELY+(2*LABELH), 7, "%u%s %-3d %4s",
			Time_value[i%9], string(Time_unit, sizeof(Time_unit)/2, i/9),
			(char)Capture[14], Capture[9] ? "AUTO" : "MAN");

		// Display Trigger
		hrg_printf(LABELX1, LABELY+(3*LABELH), 7, "%d %c %s",
			(char)Capture[16], Capture[15] ? '-' : '+',
			string(Trigger_mode, sizeof(Trigger_mode)/2, Capture[17]));

		// Clear old patterm from scope window
		for(i=1; i < SCOPEW; ++i)
			hrg_line(i+(SCOPEX-1), Scope[i-1], i+SCOPEX, Scope[i], 0);

		// Erase old move-offset, trigger-level & timebase-offset markers
		hrg_hline(SCOPEX, om, 5, 0);
		hrg_hline(SCOPEX+SCOPEW-4, ot, 5, 0);
		hrg_vline(ob, SCOPEY, 5, 0);

		if(!Dscope)			// Clear meter display if present
			hrg_fbox(SCOPEX+20, SCOPEY+(SCOPEH/2)-8, (18+18)*8, 8, 0);

		draw_grid();		// Redraw grid in case in case erased

		// Draw new pattern on scope window
		p = Capture+41;
		oy = (2*-(int)*p++) + (SCOPEY+(SCOPEH/2));
		oy += 2*-(char)Capture[12];		// Move offset
		if(oy < SCOPEY)
			oy = SCOPEY;
		else if(oy > (SCOPEY+SCOPEH))
			oy = SCOPEY+SCOPEH;
		Scope[0] = oy;
		for(i=1; i < SCOPEW; ++i) {
			y = (2*-(int)*p++) + (SCOPEY+(SCOPEH/2));
			y += 2*-(char)Capture[12];	// Move offset
			if(y < SCOPEY)
				y = SCOPEY;
			else if(y > (SCOPEY+SCOPEH))
				y = SCOPEY+SCOPEH;
			hrg_line(i+(SCOPEX-1), oy, i+SCOPEX, y, 3);
			Scope[i] = oy = y; }

		// Draw new move-offset, trigger-level & timebase-offset markers
		hrg_hline(SCOPEX, om=(-(char)Capture[12])*2+((SCOPEH/2)+SCOPEY), 5, 4);
		hrg_hline(SCOPEX+SCOPEW-4, ot=(-(char)Capture[16])*2+((SCOPEH/2)+SCOPEY), 5, 4);
		hrg_vline(ob=((char)Capture[14])*2+((SCOPEW/2)+SCOPEX), SCOPEY, 5, 4);

		return; }

	// No scope data - clear fields if Scope was displayed
	if(Dscope) {
		hrg_fbox(SCOPEX+20, SCOPEY-11, (18+18)*8, 8, 0);
		hrg_fbox(SCOPEX, SCOPEY, SCOPEW, SCOPEH, Dscope = 0); }

	// Display reading
	hrg_printf(SCOPEX+20, SCOPEY+(SCOPEH/2)-8, 7, "%-18s%18s", Capture+21, Capture+31);
}

/*
 * Perform a transaction with the ScopeMeter
 */
int transaction(void)
{
	int c;
	unsigned h, t, t1, t2;
	unsigned char f;

	clearmsg();
	while(Ctestc() != -1);		// Flush input
	Cputc(0x5A);				// Trigger ScopeMeter output
	Ctop = f = 0;
	t = TICK;
	t1 = 18*2;					// Initial timeout
	h = BASEMSG;
more:
	while(Ctop < h) {
		t2 = TICK;
		if((c = Ctestc()) != -1) {
			if(Ctop || (c == 0x5A)) {	// 1st must be 0x5A
				Capture[Ctop++] = c;
				t = t2;
				f = 255;
				t1 = 5; }
			continue; }
		if(kbtst() == 0x1B)				// User abort
			goto abort;
		if((t2-t) > t1) {				// Timeout
			if(!f) {
				message("Waiting for UT81 -press ESC to cancel");
				f = 255;
				t1 = 18*43;
				continue; }
abort:		Ctop = 0;
			message("Transfer Aborted!");
			return 255; } }
	// Base message captured - if Scope output, increase to full scope message
	if((Capture[6] & 0x80) && (h == BASEMSG)) {
		h = BASEMSG+SCOPEW;			// Add scope data
		goto more; }

	clearmsg();			// Success - remove messages (if any)
	return 0;
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i;
	unsigned char *p;

	// Parse command line arguments
	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case 'C=' : Cport = atoi(p);			continue;
		} p -= 2;
		if(!*Filename) {
			strcpy(Filename, p);
			continue; }
		abort(Help); }

	// If filename supplied, preload it
	if(*Filename) {
		fp = fopen(Filename, "rvqb");
		Ctop = fget(Capture, sizeof(Capture), fp);
		fclose(fp); }
	else if(!Cport)		// No file or COM - nothing to do
		abort(Help);

	// If COM port supplied, open it
	if(Cport) {
		if((Cport < 1) || (Cport > 4))
			abort(Help);
		if(Copen(Cport, _9600, PAR_NO|DATA_8|STOP_1, SET_DTR|OUTPUT_2))
			abort("Cannot open COM port");
		Cflags |= TRANSPARENT; }

	// Open high-res (VGA) graphics screen and draw static items
	if(hrg_open())
		abort("VGA required");
	hrg_fbox(242, 5, 156, 27, 4);
	hrg_puts(252, 15, 0x407, "UT81 - ScopeMeter");
	message("UT81 ScopeMeter Demo - Dave Dunfield - "#__DATE__"");

	// Display prompts
	if(Cport)
		hrg_puts(PROMPTX, PROMPTY+0 , 7, "SPACE = capture data from device");
	hrg_puts(PROMPTX, PROMPTY+10, 7, "  'L' = Load data from file");
	hrg_puts(PROMPTX, PROMPTY+20, 7, "  'S' = Save data to file");
	hrg_puts(PROMPTX, PROMPTY+30, 7, "  ESC = exit program");

	// Display field labels
	for(i=0; p = label[i]; ++i)
		hrg_printf(LABELX, (i*LABELH)+LABELY, 7, "%9s:", p);

	// Display scope
	hrg_box(SCOPEX-1, SCOPEY-1, SCOPEW+2, SCOPEH+2, 7);
	draw_grid();

	// Preset last display
	for(i=0; i < SCOPEW; ++i)
		Scope[i] = (SCOPEH/2)+SCOPEY;

	if(Ctop)		// Data was preloaded - display it
		display();

	for(;;) {
		i = kbget();
		clearmsg();
		switch(i) {
		case 0x1B:			// Exit program
			hrg_close();
			if(Cport)
				Cclose();
			return;
		case ' ' :			// Capture data
			if(!Cport) {
				message("No COM port specified (C=)");
				continue; }
			if(!transaction())
				display();
			continue;
		case 's' :			// Save data
		case 'S' :
			if(!input("Save to File", Filename, sizeof(Filename)-1))
				continue;
			if(!(fp = fopen(Filename, "wb"))) {
				message("Cannot WRITE: %s", Filename);
				continue; }
			fput(Capture, Ctop, fp);
			fclose(fp);
			continue;
		case 'l' :			// Load data
		case 'L' :
			if(!input("Load from File", Filename, sizeof(Filename)-1))
				continue;
			if(!(fp = fopen(Filename, "rb"))) {
				message("Cannot READ: %s", Filename);
				continue; }
			Ctop = fget(Capture, sizeof(Capture), fp);
			fclose(fp);
			display();
			continue; } }
}
