;
; Diagnostic ROM for SGC-230 Smart Antenna Tuner
;
; This program replaces the normal SGC-230 firmware ROM and communicates
; with a user via a bit-bash serial port implemented on the two SGC-230
; configuration jumper I/O pins. By connecting an RS-232 level convertor
; (such as a MAX-232), you can control all relays manually for diagnosis
; purposes, and also confirm that the status inputs to the CPU are working
; correctly. Please refer to S230DIAG.TXT for more information.
;
; SGC-230 I/O Address Map:
; PORT-A:
;	0 = MM5480 Data		4 = 4:1 input
;	1 = MM5480 Clock	5 = LO "Z" input
;	2 = !FWD IRQ input	6 = 2:1 input
;	3 = FWD input		7 = PHASE input
; PORT-B
;	0 = TIMER input
;	6 = Tune from memory jumper input (Serial data OUT)
;	7 = Tune out during receive input (Serial data IN)
;
; Dave Dunfield - www.dunfield.com - Nov 22, 2006
;
; Assemble with ASM05 from my XTOOLS package.
;

;
; 146805E2 internal registers
;
PADR	EQU	$0000		; Port-A data register
PBDR	EQU	$0001		; Port-B data register
PADDR	EQU	$0004		; Port-A direction register
PBDDR	EQU	$0005		; Port-B direction register
TIMDAT	EQU	$0008		; Timer data register
TIMCTL	EQU	$0009		; Timer control register
;
; Internal RAM locations
;
	ORG	$0010
TEMP	RMB	1		; Temporary storage
TEMP1	RMB	1		; ""
TEMP2	RMB	1		; ""
TEMP3	RMB	1		; ""
RELAY1	RMB	1		; K1-K7 (Transmitter capacitor)
RELAY2	RMB	1		; K8-K15 (Inductor)
RELAY3	RMB	1		; K16 (Antenna capacitor)
LED	RMB	1		; LED
BANK	RMB	1		; 0=K1-6 1=K7-K16 2=K17-K26
MAX	RMB	1		; Maximum bit value for selected bank
MAX1	RMB	1		; Maximum mask value for selected bank
HSPEED	RMB	1		; Count speed
;
; The SGC-230 2764 8K ROM has only 2K visible at the top of the device.
; Here I place a data value at address 0000 to set the base address of
; the output file, insuring that it will program correctly into a 2764
; from offset 0.
;
	ORG	$0000		; Start code at zero
	FDB	$FFFF		; Filler to anchor output file
;
; ROM becomes visible at 6K ($1800) into the memory map
;
	ORG	$1800
BOOT	SEI			; No interrupts
	RSP			; Reset stack
; Initialize I/O devices
	LDA	#%00000011	; MN5480 C+D=out, remaining status=In
	STA	PADDR		; Set PORT-A direction
	LDA	#%01111110	; RXD=In TXD=OUT 5-1=Unused/Out TIMER=IN
	STA	PBDDR		; Set PORT-B direction
	CLR	PADR		; Drop clock + data on MN5480
	LDA	#%01000000	; TXD to Mark state (high)
	STA	PBDR		; Set PORT-B outputs
; Zero memory
	LDX	#$10		; Start of RAM
clrram	CLR	,X		; Zero location
	INCX			; Next
	CPX	#$40		; At end?
	BNE	clrram		; Zero all of RAM
; Establish default values in RAM
	LDA	#RELAY1		; Transmitted bank
	STA	BANK		; Set active bank
	LDA	#5		; Maximum relay bit
	STA	MAX		; Set max bit
	LDA	#63		; Maximum hunt value
	STA	MAX1		; Set max value
; Issue welcome message
	LDX	#HELLO		; Welcome message
	JSR	PUTS		; Display
boot1	JSR	LFCR		; New line
;
; Main loop - update relays and wait for commands
;
main	JSR	relay		; Init relays
main1	RSP			; Reset stack (in case error)
	JSR	show		; Display status
	JSR	space		; Space for prompt
	JSR	getc		; Get character
	CMP	#'a'		; Lower-case?
	BLO	main2		; No, do not change
	AND	#$5F		; Convert to u pper
main2	CMP	#$0D		; carriage-return?
	BEQ	boot1		; Yes, Output blank line
	CMP	#' '		; Space?
	BEQ	main1		; Yes, Redisplay
	CMP	#'R'		; Reset?
	BEQ	boot		; Restart from beginning
;
; Select transmitter-capacitor relay bank
;
selt	CMP	#'T'		: Transmitter
	BNE	seli		; No, try next
	LDA	#63		; Maximum hunt value
	STA	MAX1		; Set max hunt
	LDA	#RELAY1		; TX relay bank
	LDX	#5		; Maximum bit value
sbank	STA	BANK		; Set relay bank
	STX	MAX		; Set maximum bit
	BRA	main1		; next command
;
; Select Inductor relay bank
;
seli	CMP	#'I'		; Inductor?
	BNE	sela		; No, try next
	LDA	#255		; Maximum hunt value
	STA	MAX1		; Set max hunt
	LDA	#RELAY2		; Inductor relay bank
	LDX	#7		; Maximum bit value
	BRA	sbank		; Save and continue
;
; Select Antenna-capacitor relay bank
;
sela	CMP	#'A'		; Antenna?
	BNE	sled		; No, try next
	LDA	#31		; Maximum hunt value
	STA	MAX1		; Set max hunt
	LDA	#RELAY3		; Antenna relay bank
	LDX	#4		; Maximum bit value
	BRA	sbank		; Save and continue
;
; Toggle LED (Tuned indicator)
;
sled	CMP	#'L'		; LED?
	BNE	speed		; No, try next
	LDA	LED		; Get LED value
	EOR	#1		; Toggle bit
	STA	LED		; Resave
	BRA	main		; Next command
;
; Set count speed
;
speed	CMP	#'S'		; Speed?
	BNE	up		; No, try next
	LDA	hspeed		; Get hunt speed
	INCA			; Advance
	CMP	#5		; End of range?
	BLO	speed1		; No, it's OK
	CLRA			; Reset to zero
speed1	STA	hspeed		; Resave
	ADD	#'1'		; Convert to printable
	JSR	putc		; Display
	JMP	main1		; And continue
;
; Increase selected bank capacitor/inductor value
;
up	CMP	#'U'		; Up?
	BNE	down		; No, try next
	BSR	bumpup		; Advance
	BRA	main1		; Next command
;
; Decrease selected bank capacitor/inductor value
;
down	CMP	#'D'		; Down?
	BNE	fwd		; No, try next
	BSR	bumpdn		; Decrease
down1	BRA	main1		; Next command
;
; Hunt forward in selected bank, stopping at end
;
fwd	CMP	#'F'		; Forward?
	BNE	back		; No, try next
fwd1	BSR	bumpup		; Increment
	BCS	down1		; Error - reverse
	LDX	#'F'		; Forward indicator
	BSR	direct		; Display status with indicator
	JSR	wait		; Delay
	BRA	fwd1		; And continue
;
; Hunt backward in selected bank, stopping at end
;
back	CMP	#'B'		; Back?
	BNE	hunt		; No, try next
back1	BSR	bumpdn		; Decrease
	BCS	down1		; Error - reverse
	LDX	#'B'		; Backward indicator
	BSR	direct		; Display status with indicator
	BSR	wait		; Delay
	BRA	back1		; And continue
;
; Hunt continuously up/down in selected bank
;
hunt	CMP	#'H'		; Hunt?
	BNE	tog		; No, try next
hunt1	BSR	bumpup		; Advance
	BCS	hunt4		; Hit end - switch to Back
hunt2	LDX	#'F'		; Forward indicator
	BSR	direct		; Display
	BRA	hunt1		; Do till end
hunt3	BSR	bumpdn		; Decrease
	BCS	hunt2		; Hit end - switch to Forward
hunt4	LDX	#'B'		; Backward indicator
	BSR	direct		; Display
	BRA	hunt3		; Do till end
;
; Bump the relay bank value to one higher
;
bumpup	LDX	bank		; Get address
	LDA	,X		; Get value
	CMP	MAX1		; In range?
	BHS	bup1		; At MAX - abort
	INCA			; Next
	BRA	bdn1		; And proceed
bup1	SEC			; Indicate at end of range
	RTS
;
; Decrement the relay values
;
bumpdn	LDX	bank		; Get bank
	LDA	,X		; Get value
	BEQ	bup1		; At 0 - abort
	DECA			; Decrement
bdn1	STA	,X		; Resave
	JSR	relay		; Update relays
	CLC			; Indicate no error
	RTS
;
; No command, check for 0-(maxbit) relay drive toggle
;
tog	SUB	#'0'		; convert digits
	CMP	MAX		; In range?
	BHI	error		; Yes
; In range, toggle bit
tog1	LDX	bank		; Get address
	CLR	temp		; Zero temp
	INC	temp		; Temp = 1
	TSTA			; At end?
	BEQ	tog3		; Yes, stop
tog2	LSL	temp		; Shift
	DECA			; Reduce count
	BNE	tog2		; Do them all
tog3	LDA	temp		; Get value
	EOR	,X		; Toggle bit
	STA	,X		; Resave
	JMP	main		; Update
;
; Display indictor '>' if selected, otherwise ':'
;
showind	JSR	PUTC		; Display label
	LDA	#':'		; Assume normal
	CPX	BANK		; Is this it?
	BNE	putb2		; No - not selected
	LDA	#'>'		; Change to selected
putb2	JSR	putc		; Display
	LDA	,X		; Load value
	RTS
;
; Indicate direction & show relay/status value
;
direct	BSR	lfcr		; New line
	TXA			; Get indicator
	JSR	putc		; Display
	JSR	space		; Seperator
	BSR	show1		; Display
;
; Wait for defined S)peed value between counts
;
wait	LDA	hspeed		; Set high
	LSLA			; Adjust
	LSLA			; To good
	LSLA			; Speed
	LSLA			; Range
	INCA			; Convert 0
	CLRX			; Zero count
wait1	BRCLR	7,PBDR,err1	; Key - abort
	DECX			; Reduce low
	BNE	wait1		; Not expired
	DECA			; Reduce high
	BNE	wait1		; Not expired
	RTS
;
; Error occured, issue message and restart
;
ERROR	LDA	#'?'		; Error indicator
	JSR	putc		; Display
	LDX	#CMDS		; Command prompt
	JSR	puts		; Output
err1	JMP	main1		; Reset
;
; Display relays/status
;
SHOW	BSR	LFCR		; New line
SHOW1	LDX	#RELAY1		; Get TX relay
	LDA	#'T'		; Prompt
	BSR	showind		; Display
	LDA	relay1		; K1-K6
	LDX	#6		; 6-bits
	BSR	putbin		; Display
	BSR	space		; Separator
	LDX	#relay2		; Get INDUCTOR relay
	LDA	#'I'		; Prompt
	BSR	showind		; Display
	LDA	relay2		; K7-K14
	LDX	#8		; 8-bits
	BSR	putbin		; Display
	BSR	space		; Separator
	LDX	#relay3		; Get ANTENNA relay
	LDA	#'A'		; Prompt
	BSR	showind		; Display
	LDA	relay3		; K17-K26
	LDX	#5		; 5-bits
	BSR	putbin		; Display
	BSR	space		; Separator
	LDX	#LED		; Get LED status
	LDA	#'L'		; Indicator
	BSR	showind		; Display
	LDX	#1		; 1 bit
	BSR	putbin		; Display
	BSR	space		; Separator
	LDA	PADR		; Get PORT-A
	RORA			; Skip DATA
	RORA			; Skip CLOCK
	LDX	#6		; 6-bits
;
; Display binary output on serial port
;
putbin	STA	temp2		; Save for later
putb1	CLRA			; Get zero
	ROR	temp2		; C = bit
	ADC	#'0'		; '0' or '1'
	BSR	putc		; Write it
	DECX			; Reduce count
	BNE	putb1		; Do them all
	RTS
;
; Write line-feed/carriage-return to serial port (new line)
;
LFCR	LDA	#$0A		; LF
	BSR	putc		; Output
	LDA	#$0D		; CR
	BRA	putc		; Output
;
; Output a string to the serial port
;
PUTS	LDA	TEXTSEG,X	; Get data
	BEQ	wbf2		; End, exit
	BSR	putc		; Write it
	INCX			; Skip
	BRA	PUTS		; And continue
;
; Write space to serial port
;
SPACE	LDA	#' '		; Get space
;
; Write character in A to bit-bash serial port
;
PUTC	STA	temp		;4 Save for later
	LDA	#8		;2 Shift 8 bits
	STA	temp1		;4 Save counter
	BCLR	6,PBDR		;5 Send start bit
	BSR	wbfull		;6+xx Wait 1/2 bit
; 30 cycles/loop (plus wbfull)
putc1	ROR	temp		;5 Get next bit
	BCS	putc2		;3 Bit is set
	BCLR	6,PBDR		;5 Send zero bit
	BRA	putc3		;3 And continue
putc2	BSET	6,PBDR		;5 Send one bit
	BRA	putc3		;3 3-cycle no-op
putc3	BSR	wbfull		;6+xx wait 1/2 bit time
	DEC	temp1		;5 reduce count
	BNE	putc1		;3 Do them all
	BSET	6,PBDR		;5 Send stop bit
; Wait full bit time (65 - 30 cycles)
wbfull	BRCLR	7,PBDR,abort	;5 Test for abort
	NOP			;2
	LDA	#5		;2 Get count (4)
wbf1	DECA			;3 Reduce counbt
	BNE	wbf1		;3 Do them all
wbf2	RTS			;6
; Key was pressed - abort to main prompt
abort	BSET	6,PBDR		; Insure TXD high
	CLRA			; Zero delay
abor1	BRCLR	7,PBDR,abort	; Low bit - restart
	DECA			; Reduce count
	BNE	abor1		; Wait for finish
	JMP	main1		; And restart
;
; Get character from bit-bash serial port
;
getc	BRSET	7,PBDR,getc	; Wait for it
	LDA	#8		;2 8 bits
	STA	temp1		;4 save count
	JSR	wbf2		;12
; 30 cycles/loop (plus wbfull)
getc1	LDA	#8		;2 Delay count
getc2	DECA			;3 Reduce count
	BNE	getc2		;3 Do them all
	NOP			;2
	LDA	PBDR		;2 Get data
	ROLA			;3 C = Data bit
	ROR	temp		;5 store in data
	NOP			;2
	NOP			;2
	NOP			;2
	DEC	temp1		;5 Entire count
	BNE	getc1		;3 Do them all
	LDA	temp		; Get data
	RTS
;
; Update relay latches from memory values
;
RELAY	BSR	rbit1		; Send zero
	BSR	rbit0		; Send one
	BSR	rbit1		; (1) Send zero
; Send K1-K6 bits
	LDA	relay1		; k1-k6
	LDX	#6		; Send 6 bits
	BSR	wrelay		; (1-6) 1-6
	CLRA			; Filler
	LDX	#5		; 5 filler
	BSR	wrelay		; (7-11) Write to relay
	LDA	RELAY2		; K7-K16
	LDX	#3		; 3 bits
	BSR	wrelay		; (12-14) 7-9
	BSR	rbit0		; (15) filler
	BSR	rbit0		; (16) filler
	BSR	rbit0		; (17) filler
	BSR	rbit0		; (18) filler
	LDX	#5		; K10-K14
	BSR	wrelay		; (19-23)
	LDA	relay3		; K17-TUNE
	BSR	rbitc		; (24) K17+K18
	BSR	rbit0		; (25) filler
	BSR	rbitc		; (26) K19+K20
	BSR	rbit0		; (27) filler
	BSR	rbit0		; (28) filler
	LDX	#3		; 3 bits
	BSR	wrelay		; (29-31) K21-K26
	BSR	rbit0		; (32) filler
	LDA	LED		; Get LED indicator
	BSR	rbitc		; (33) Tune
	LDA	#$FF		; All-0
	LDX	#4		; End of line
; Write to relay
wrelay	BSR	rbitc		; Write to relay
	DECX			; Reduce count
	BNE	wrelay		; Write them all
	RTS
; Send one relay bit
rbitc	RORA			; C = low bit
rbit	BCS	rbit1		; Send carry
; Relays are logically inverted from latch output
; Rather than keep track of that, I simply call a
; 1 a 0 and a 0 a 1 at this point.
rbit0	BSET	0,PADR		; Get logical zero
	BSET	1,PADR		; Raise clock
	BCLR	1,PADR		; Drop clock
	RTS
rbit1	BCLR	0,PADR		; Get logical one
	BSET	1,PADR		; Raise clock
	BCLR	1,PADR		; Drop clock
	RTS	
;
; Dummy interrupt handler (should never be called)
;
NOINT	RTI
;
; Position to top segment for text storage (within one page)
;
	ORG	$1F00
TEXTSEG	EQU	*
HELLO	FCB	$0A,$0D
	FCC	'SGC-230 diagnostic 1.0 - Dave Dunfield - www.dunfield.com - Nov 22, 2006'
	FCB	$0A
CMDS	FCB	$0A,$0D
	FCCZ	' 0-7 T)x I)nduct A)nt U)p D)own F)wd B)ack H)unt S)peed L)ed R)eset'
;
; At top of top segment goes interrupt vectors
;
	ORG	$1FF6
	FDB	NOINT		; Timer interrupt from wait
	FDB	NOINT		; Timer interrupt
	FDB	NOINT		; External interrupt
	FDB	NOINT		; SWI
	FDB	BOOT		; RESET
