/*
 * Telephone "distinctive ringing" detector.
 *
 * This program may be compiled with the DDS 8051 MICRO-C Compiler.
 *
 * Compile command: CC51 ringswit -fop m=t
 *
 * Permission granted for personal (non-commercial) use only.
 * Authors name and Copyright notice may not be altered or removed.
 *
 * Copyright 1993-2005 Dave Dunfield
 * All rights reserved.
 */
#include <8051io.h>				/* 8051 I/O definitions */
#include <8051reg.h>			/* 8051 register definitions */
#include <8051bit.h>			/* 8051 bit manipulation macros */

/*
 * If you remove the following define, the code will output progress
 * message to the 8051 serial port, which are useful when troubleshooting.
 */
#define	printf()				/* disable printf unless debugging */

#define	RECOUNT		2			/* Number of rings to perform recount */

/*
 * Hardware I/O bits in P1
 */
#define	TONEGEN		P1.7		// Tone generation			(output)
#define	RELAY		3			// Offset to relays (-1)	(output)
#define	OFFHK		0x0E		// Off-hook detectors		(input)
#define	RINGDET		0x01		// Ring detector 			(input)

/*
 * 8051 internal RAM locations
 */
register unsigned char P, LP1;	/* Active port mask & status */
register unsigned bflag;		/* Busy tone cadence counter */

/*
 * Read and de-bounce the input port P1
 */
unsigned char readp1(void)
{
	unsigned i, p, q;

	p = q = P1;
	for(i=0; i < 600; ++i) {
		if(p != P1) {
			q &= (p = P1);
			i = 0; } }

	return LP1 = p & (q | (~RINGDET & 255));
}

/*
 * Generate "busy" tone on local loop if any sets other than
 * the one involved in the current call are off-hook.
 */
void busy(void)
{
	unsigned i, j;

	if((~readp1() & ~P) & OFFHK) {
		printf("Busy\n");
		if(++bflag >= 8) {
			bflag = 0;
			for(i=0; i < 150; ++i) {
				setbit(TONEGEN);
				for(j=0; j < 15; ++j);
				clrbit(TONEGEN);
				for(j=0; j < 15; ++j); } } }
}

/*
 * Incomming call... count rings, and route to proper set
 */
void incomming(void)
{
	unsigned i, n, k;

	printf("Incomming... ");

	k = 0;								/* Count of recount cycles */

	/*
	 * First, count number of rings
	 */
count_rings:
	n = 0;
	for(;;) {
		++n;
		while(~readp1() & RINGDET);		/* Wait for ring to stop */
		i = 0;
		while(readp1() & RINGDET) {		/* Check for no-ring */
			if(++i > 12)
				goto rings_counted; } }

rings_counted:
	printf("%u rings\n", n);

	/*
	 * If this is a re-count, and the set has already gone off-hook,
	 * leave the connection intact (don't re-select).
	 */
	if(k && (~LP1 & P))
		goto connected;

	if((n < 1) || (n > 3))				/* Invalid... try again */
		return;

	P = 1 << n;			/* Set line mask */

	/* Select line (activate relay) */
	P1 = (P << RELAY) | (OFFHK|RINGDET);

	/*
	 * Sometimes the first ring is incorrect. Until we have counted
	 * RECOUNT times, we will keep listing and reselecting the output
	 * relay.
	 */
	if(++k < RECOUNT) {				/* Recount is due */
		/* Wait for ringing to stop */
		for(i=0; i < 100; ++i) {
			if(~readp1() & RINGDET) {
				printf("Recount... ");
				goto count_rings; } } }
	else {							/* Assume it is stable */
		for(i=0; i < 100; ++i) {
			busy();
			if(~LP1 & RINGDET) {
				printf("Ring\n");
				i = 0; } } }

connected:
	printf("Connected\n");

	/* Wait for line to go on-hook */
	for(i=0; i < 25; ++i) {
		busy();
		if(~LP1 & P)
			i = 0; }
}

/*
 * Local set has gone off-hook, connect to line, and indicate busy
 * to other sets until the call is completed.
 */
void outgoing(void)
{
	unsigned i;

	printf("Outgoing P=%02x\n", P);

	P1 = (P << RELAY) | (OFFHK|RINGDET);	/* Select line (activate relay) */
	delay(100);
	for(i=0; i < 25; ++i) {
		busy();
		if(~LP1 & P)
			i = 0; }
}

/*
 * Main program....
 *
 * Sit in idle loop, waiting for ringing, or any set off-hook.
 */
void main(void)		// Valid declaration for embedded system (no OS)
{
#ifndef printf
	serinit(28800);		/* For compatibility with DDS MONICA debugger */
#endif

	bflag = 0;
	for(;;) {
		P1 = OFFHK|RINGDET;		/* Set all lines idle */
		delay(100);
		printf("Idle\n");
		P = (~readp1() & (OFFHK|RINGDET));
		if(P & RINGDET)			/* Ringing - incoming call */
			incomming();
		else if(P & OFFHK)		/* Set off-hook - originate */
			outgoing(); }
}
