/*
 * C-FLEA Virtual Machine implementation in C with DDS MicroScope Support
 *
 * This is a version of the C virtual machine (See VMC.C) which has been
 * modified to include support for the DDS MicroScope Debugger. Use this
 * as an example/guide to providing a MicroScope debugging interface into
 * any C-FLEA virtual machine.
 *
 * Kernel interface commands and formats are compatible with the stock
 * CFLEA.DMS configuration delivered with MicroScope.
 *
 * Due to the use of serial I/O functions, this program compiles only
 * with DDS Micro-C/PC (available from www.dunfield.com).
 *
 * Compile command: cc vmcdms -pof
 *
 * Use "vmcdms ?" for a brief command summary.
 *
 * Copyright 1996-2005 Dave Dunfield
 * All rights reserved.
 */
#include <stdio.h>
#include <keys.h>
#include <comm.h>
#include <setjmp.h>

/* MicroScope Kernel Interface Definitions */
#define	K_ID		0xCFCF		/* Kernel ID */
#define	K_VER		1			/* Kernel version */
#define	BPCODE		0xFF		/* Breakpoint opcode */
#define	K_QUERY		0xA0		/* Query command */
#define	K_BREAK		0xA1		/* Break command */
#define	K_EXEC		0xA2		/* Execute command */
#define	K_STEP		0xA3		/* Step command */
#define	K_READ		0xB0		/* Read memory command */
#define	K_WRITE		0xB8		/* Write memory command */
#define	BR1			0x90		/* Break response code #1 */
#define	BR2			0xA5		/* Break response code #2 */
#define	BR3			0xB9		/* Break response code #3 */

#define	BIOS_TICK	peekw(0x40,0x6C)

/* C-FLEA VM definitions */
#define	BYTE_OP	0x08		/* Bit in opcode indicating byte operand */
#define	EQ		0x01		/* Bit in cflags indicating == */
#define	ULT		0x02		/* Bit in cflags indicating unsigned < */
#define	UGT		0x04		/* Bit in cflags indicating unsigned > */
#define	SLT		0x08		/* Bit in cflags indicating signed < */
#define	SGT		0x10		/* Bit in cflags indicating signed > */

/*
 * Public global variables
 */
unsigned
	ACC,					/* C-FLEA accumulator */
	INDEX,					/* C-FLEA Index register */
	SP,						/* C-FLEA Stack Pointer */
	PC,						/* C-FLEA Program Counter */
	CFSEG;					/* C-FLEA code image segment (if used) */

/*
 * Private global variables
 */
static unsigned
	Alt,					/* Alternate result from DIV */
	Address,				/* Address holding register */
	com_port = 1,			// Serial port to use
	com_speed = _19200;		// Serial port speed
static unsigned char
	keybuf[256],			// Keyboard buffer
	Wkey,					// Keyboard buffer WRITE pointer
	Rkey,					// Keyboard buffer READ pointer
	opcode,					/* current command opcode */
	cflags,					/* Compare result flags */
	mon = 255;				// Monitor enabled

jmp_buf
	reset_vector;			/* Reset to kernel */

/*
 * C-FLEA Function prototypes
 */
unsigned cf_run(void);			/* Main C-FLEA execution function */
void cf_output(unsigned port);	/* Called to output to an I/O port */
void cf_input(unsigned port);	/* Called to input from an I/O port */

/*
 * Functions to read and write C-FLEA memory
 *
 * For this demo, we will simply allocate a 64k segment and use the
 * host compilers peek/poke functions to access it, however you may
 * implement any memory and addressing scheme you wish.
 */
#define	read_mem(address)			peek(CFSEG, address)
#define	write_mem(address, data)	poke(CFSEG, address, data)

/*
 * Compute the address of the current instructions operand
 */
static void cf_get_address(void)
{
	switch(opcode & 7) {
		case 0 :		/* Immediate reference */
			Address = PC++;
			return;
		case 1 :		/* Direct address */
			Address = read_mem(PC++);
			Address |= read_mem(PC++) << 8;
			return;
		case 2 :		/* Indirect - no offset */
			Address = INDEX;
			return;
		case 3 :		/* Indirect with offset */
			Address = INDEX + read_mem(PC++);
			return;
		case 4 :		/* Indirect from SP with offset */
			Address = SP + read_mem(PC++);
			return;
		case 5 :		/* On top of stack, remove */
			Address = SP;
			SP += 2;
			return;
		case 6 :		/* Indirect through top of stack, remove */
			Address = read_mem(SP++);
			Address |= read_mem(SP++) << 8;
			return;
		case 7 :
			Address = read_mem(SP);
			Address |= read_mem(SP+1) << 8;
			return; }
}

/*
 * Main C-FLEA virtual machine execution function
 */
unsigned cf_step(void)
{
	// Make these static in this version so they won't be
	// allocated and released with every instruction step.
	static unsigned operand, t1, t2;

	opcode = read_mem(PC++);

	if(opcode < 0x98) {		/* Catagory 1 instruction */
		cf_get_address();
		operand = read_mem(Address);
		if(!(opcode & BYTE_OP)) {
			if(!(opcode & 7))
				++PC;
			operand |= read_mem(Address+1) << 8; }
		switch(opcode & 0xF0) {
		case 0x00 :	ACC = operand;					break;	/* LD */
		case 0x10 : ACC += operand;					break;	/* ADD */
		case 0x20 : ACC -= operand;					break;	/* SUB */
		case 0x30 : ACC *= operand;					break;	/* MUL */
		case 0x40 :
			Alt = ACC % operand; ACC /= operand;	break;	/* DIV */
		case 0x50 : ACC &= operand;					break;	/* AND */
		case 0x60 :	ACC |= operand;					break;	/* OR */
		case 0x70 :	ACC ^= operand;					break;	/* XOR */
		case 0x80 :											/* CMP */
			if(ACC == operand) {
				cflags = EQ;
				ACC = 1; }
			else {
				cflags = (ACC < operand) ? ULT : UGT;
				cflags |= ((int)ACC < (int)operand) ? SLT : SGT;
				ACC = 0; }
			break;
		case 0x90 :	INDEX = operand; }						/* LDI */
	} else if (opcode < 0xC8) {		/* Catagory 2 instruction */
		cf_get_address();
		switch(opcode & 0xF8) {
		case 0x98 :	INDEX = Address;				break;	/* LEAI */
		case 0xA0 :	write_mem(Address+1, ACC >> 8);			/* ST */
		case 0xA8 :	write_mem(Address, ACC);		break;	/* STB */
		case 0xB0 :	write_mem(Address, INDEX);				/* STI */
					write_mem(Address+1, INDEX >> 8); break;
		case 0xB8 : ACC >>= read_mem(Address);		break;	/* SHR */
		case 0xC0 :	ACC <<= read_mem(Address); }			/* SHL */
	} else switch(opcode) {		/* Catagory 3 instruction */
	case 0xC8 :	ACC = (cflags & (SLT))	  ? 1 : 0;	break;	/* LT */
	case 0xC9 : ACC = (cflags & (SLT|EQ)) ? 1 : 0;	break;	/* LE */
	case 0xCA : ACC = (cflags & (SGT))	  ? 1 : 0;	break;	/* GT */
	case 0xCB : ACC = (cflags & (SGT|EQ)) ? 1 : 0;	break;	/* GE */
	case 0xCC : ACC = (cflags & (ULT))	  ? 1 : 0;	break;	/* ULT */
	case 0xCD : ACC = (cflags & (ULT|EQ)) ? 1 : 0;	break;	/* ULE */
	case 0xCE : ACC = (cflags & (UGT))	  ? 1 : 0;	break;	/* UGT */
	case 0xCF : ACC = (cflags & (UGT|EQ)) ? 1 : 0;	break;	/* UGE */
	case 0xD0 :												/* JMP */
	doljmp:		Address = read_mem(PC++);
				PC = (read_mem(PC++) << 8) | Address;break;
	case 0xD1 : if(!ACC) goto doljmp; PC+=2;		break;	/* JZ */
	case 0xD2 :	if(ACC)  goto doljmp; PC+=2;		break;	/* JNZ */
	case 0xD3 :
	dosjmp:		Address = read_mem(PC++);					/* SJMP */
				if(Address & 0x80) Address |= 0xFF00;
				PC += Address;						break;
	case 0xD4 :	if(!ACC) goto dosjmp; ++PC;			break;	/* SJZ */
	case 0xD5 :	if(ACC)  goto dosjmp; ++PC;			break;	/* SJNZ */
	case 0xD6 :	PC = ACC;							break;	/* IJMP */
	case 0xD7 :												/* SWITCH */
		Address = INDEX;
		for(;;) {
			t1 = read_mem(Address++);
			t1 |= read_mem(Address++) << 8;
			t2 = read_mem(Address++);
			t2 |= read_mem(Address++) << 8;
			if(!t1) {
				PC = t2;
				break; }
			if(t2 == ACC) {
				PC = t1;
				break; } }
		break;
	case 0xD8 :												/* CALL */
		operand = PC + 2;
		write_mem(--SP, operand >> 8);
		write_mem(--SP, operand);
		goto doljmp;
	case 0xD9 :												/* RET */
		PC = read_mem(SP++);
		PC |= read_mem(SP++) << 8;					break;
	case 0xDA : SP -= read_mem(PC++);				break;	/* ALLOC */
	case 0xDB :	SP += read_mem(PC++);				break;	/* FREE */
	case 0xDC :												/* PUSHA */
		write_mem(--SP, ACC >> 8);
		write_mem(--SP, ACC);						break;
	case 0xDD :												/* PUSHI */
		write_mem(--SP, INDEX >> 8);
		write_mem(--SP, INDEX);						break;
	case 0xDE : SP = ACC;							break;	/* TAS */
	case 0xDF : ACC = SP;							break;	/* TSA */
	case 0xE0 : ACC = 0;							break;	/* CLR */
	case 0xE1 : ACC = ~ACC;							break;	/* COM */
	case 0xE2 : ACC = -ACC;							break;	/* NEG */
	case 0xE3 : ACC = !ACC;							break;	/* NOT */
	case 0xE4 : ++ACC;								break;	/* INC */
	case 0xE5 : --ACC;								break;	/* DEC */
	case 0xE6 : INDEX = ACC;						break;	/* TAI */
	case 0xE7 : ACC = INDEX;						break;	/* TIA */
	case 0xE8 : INDEX += ACC;						break;	/* ADAI */
	case 0xE9 : ACC = Alt;							break;	/* ALT */
	case 0xEA : cf_output(read_mem(PC++));			break;	/* OUT */
	case 0xEB : cf_input(read_mem(PC++));			break;	/* IN */
	default: return opcode; }			/* Unknown opcode */

	return 0;
}

/*
 * This function is called in response to an OUT instruction
 */
void cf_output(unsigned port)
{
	switch(port) {
		case 0 :	/* Console output */
			putc(ACC, stdout);
			return;
		/* More output ports can be defined here */
		}

	printf("\nC-FLEA: Unknown output port %02x at %04x\n", port, PC-2);
	exit(-1);
}

/*
 * This function is called in response to an IN instruction
 */
void cf_input(unsigned port)
{
	switch(port) {
		case 0 :	/* Console input */
			if(Rkey != Wkey)
				ACC = keybuf[Rkey++];
			else
				ACC = 0xFFFF;
			return;
		/* More input ports can be defined here */
		}

	printf("\nC-FLEA: Unknown input port %02x at %04x\n", port, PC-2);
	exit(-1);
}


/* --- DDS MicroScope Kernel Interface --- */

toggle_monitor()
{
	if(mon = mon ? 0 : 255)
		printf("Monitoring enabled (F1 to disable)\n");
	else
		printf("Monitoring disabled (F1 to enable)\n");
}

scan_keyboard()
{
	unsigned c;
	switch(c = kbtst()) {
	case _F1 :	toggle_monitor();	return 0;
	case _F2 :	return -1;
	case _F9 :	longjmp(reset_vector, 7);
	case _F10 :
		Cclose();
		free_seg(CFSEG);
		exit(0);
	default:
		if(!(c & 0xFF00))
			keybuf[Wkey++] = c;
	case 0 :
		return 0; }
}

/*
 * Get character from serial port with timeout
 */
static int Cgett(void)
{
	int c;
	unsigned t;
	t = BIOS_TICK;
	do {
		if((c = Ctestc()) != -1)
			return c;
		if(!(Csignals() & CTS))
			longjmp(reset_vector, 7);
		scan_keyboard(); }
	while((BIOS_TICK - t) < 18);
	printf("Timeout!\n");
	return 0;
}

/*
 * Get registers from host preceeding EXEC or STEP
 */
static void get_registers(void)
{
	cflags=	Cgett();					// Get condition flags
	Alt =	(Cgett() << 8) | Cgett();	// Get ALT result value
	SP =	(Cgett() << 8) | Cgett();	// Get SP
	PC =	(Cgett() << 8) | Cgett();	// Get PC
	INDEX =	(Cgett() << 8) | Cgett();	// Get Index Register
	ACC	=	(Cgett() << 8) | Cgett();	// Get Accumulator
}

/*
 * Scale to a fractional value
 */
static unsigned scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}
static char help[] = { "\n\
Use:	VMCDMS [options]\n\n\
opts:	C=1-4	- Specify COMM port		[1]\n\
	S=speed	- Specify baudrate (50-57600)	[19200]\n\
	-M	- Start with monitoring enabled\n\
\nCopyright 2001-2005 Dave Dunfield\nAll rights reserved.\n" };

/*
 * Demo program to show use of the VMC module.
 */
main(int argc, char *argv[])
{
	int i;
	unsigned a;
	unsigned char *ptr;

	/* Parse command line options */
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((*ptr++ << 8) | *ptr++) {
		case 'c=' :		// Comm port
		case 'C=' :	com_port = atoi(ptr);					continue;
		case 's=' :		// Comm speed
		case 'S=' : com_speed = scale(57600, 2, atoi(ptr));	continue;
		case '/m' :
		case '/M' :
		case '-m' :		// Monitoring enabled
		case '-M' : mon = 0;								continue;
		default: printf("\nUnknown option: %s\n", argv[i]);
		case '?'<<8 :
		case ('-'<<8)|'?':
		case ('/'<<8)|'?': }
		abort(help); }

	// Allocate code segment and zero it
	if(!(CFSEG = alloc_seg(4096)))		/* Allocate 64K run segment */
		abort("Cannot allocate memory");
	i = 0;
	do {
		pokew(CFSEG, i, 0); }
	while(i += 2);

	// Open the serial port
	if(Copen(com_port, com_speed, PAR_NO|DATA_8|STOP_1, SET_RTS|SET_DTR|OUTPUT_2))
		abort("Cannot open com port");
	Cflags |= TRANSPARENT;

	printf("C-FLEA - DDS MicroScope Remote Kernel (F10=Exit F9=Reset F2=HaltExec)\n");
	toggle_monitor();

	if(setjmp(reset_vector)) {
reset:
		if(mon)
			printf("RESET\n"); }

	while(!(Csignals() & CTS))
		scan_keyboard();

	/*
	 * Main kernel command loop
	 */
	for(;;) {
		do {
			scan_keyboard();
			if(!(Csignals() & CTS))
				goto reset; }
		while((i = Ctestc()) == -1);

		// Process host command
		switch(i) {

		case K_QUERY:		// Send kernel ID and version
			if(mon)
				printf("QUERY\n");
			Cputc(0xAA);		// Response byte 1
			Cputc(0x55);		// Response byte 2
			Cputc(K_ID >> 8);	// Kernel ID high byte
			Cputc(K_ID);		// Kernel ID low byte
			Cputc(K_VER);		// Kernel version
			continue;

		case K_BREAK:		// Set breakpoint
			a = (Cgett() << 8) | Cgett();	// Get address
			Cputc(i = peek(CFSEG, a));		// Send current value
			if(mon)
				printf("BREAK: %04x %02x\n", a, i);
			poke(CFSEG, a, BPCODE);			// Implant break opcode
			Cputc(0);						// Indicate success
			continue;

		case K_EXEC:		// Execute
			get_registers();				// Download registers
			if(mon)
				printf("EXEC: PC:%04x SP:%04x A:%04x I:%04x\n", PC, SP, ACC, INDEX);
			do {
				if((Csignals() & (DSR|CTS)) != (DSR|CTS)) {
					if(!(Csignals() & CTS))
						goto reset;
					i = 0;
					goto send_regs; }
				if(scan_keyboard()) {
					i = 0;
					goto send_regs; }
				if(i == ' ')
					toggle_monitor(); }
			while(!cf_step());				// Execute till failed/break
			i = 1;							// Adjust by breakpoint

			// Upload registers following completion of an exec/step
		send_regs:
			if(mon)
				printf("STOP: PC:%04x SP:%04x A:%04x I:%04x\n", PC, SP, ACC, INDEX);
			Cputc(BR1);						// Break prefix 1
			Cputc(BR2);						// Break prefix 2
			Cputc(BR3);						// Break prefix 3
			Cputc(i);						// PC Adjust factor
			Cputc(ACC);						// Send accumulator low
			Cputc(ACC >> 8);				// Send accumulator high
			Cputc(INDEX);					// Send index register low
			Cputc(INDEX >> 8);				// Send index register high
			Cputc(PC);						// Send PC low
			Cputc(PC >> 8);					// Send PC high
			Cputc(SP);						// Send SP low
			Cputc(SP >> 8);					// Send SP high
			Cputc(Alt);						// Send ALT result low
			Cputc(Alt >> 8);				// Send ALT result high
			Cputc(cflags);					// Send condition flags
			continue;

		case K_STEP:		// Single Step
			get_registers();				// Download registers
			if(mon)
				printf("STEP: PC:%04x SP:%04x A:%04x I:%04x\n", PC, SP, ACC, INDEX);
			cf_step();						// Exec one instruction
			i = 0;							// No PC adjustment
			goto send_regs;					// And send register back

		case K_READ:		// Read memory
			a = (Cgett() << 8) | Cgett();	// Get address
			if(!(i = Cgett()))				// Get size
				i = 256;					//	0=256
			if(mon)
				printf("READ: %04x %u\n", a, i);
			while(i--)
				Cputc(peek(CFSEG, a++));	// Read data (memory->host)
			continue;

		case K_WRITE:		// Write memory
			a = (Cgett() << 8) | Cgett();	// Get address
			if(!(i = Cgett()))				// Get size
				i = 256;					//	0=256
			if(mon)
				printf("WRITE: %04x %u\n", a, i);
			while(i--)
				poke(CFSEG, a++, Cgett());	// Write data (host->memory)
			continue; }

		printf("Unknown command: %02x\n", i); }
}
