;
; Dunfield MicroScope - Target Kernel for: AVR
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
; NOTE: The AVR does not have any way to WRITE to program memory,
;	which severely limits the debugging capabilities:
;	- You cannot LOAD a program - it must be preprogrammed into
;	  along with this debug kernel.
;	- You cannot set breakpoints dynamically! - You must preset
;	  breakpoints in your code via calls to the 'enter' address.
;	- You CAN single-step
;
ROM	EQU	$0000		; Code goes here
RAM	EQU	$0060		; Data memory goes here
STACK	EQU	$015F		; AVR stack initialized to here
BAUD	EQU	25		; Baudrate divisor for 9600 @ 4.00Mhz
; Special function register definitions for: AVR 90S8515
UBRR	EQU	$09		; UART Baud Rate Register
UCR	EQU	$0A		; UART Control Register
USR	EQU	$0B		; UART Status Register
UDR	EQU	$0C		; UART I/O Data Register
EECR	EQU	$1C		; EEPROM Control Register
EEDR	EQU	$1D		; EEPROM Data Register
EEAR	EQU	$1E		; EEPROM Address Register (word)
TCNT0	EQU	$32		; Timer/Counter 0
TCCR0	EQU	$33		; Timer/Counter 0 Control Register
MCUCR	EQU	$35		; MCU general Control Register
TIFR	EQU	$38		; Timer/Counter Interrupt FLAG register
TIMSK	EQU	$39		; Timer/Counter Interrupt MASK register
SP	EQU	$3D		; Stack Pointer
SREG	EQU	$3F		; Status REGister
; Command codes from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREADP	EQU	$B0		; Read Program memory
CREADD	EQU	$B1		; Read Data memory
CREADE	EQU	$B2		; Read EEPROM memory
CWRITD	EQU	$B8		; Write Data memory
CWRITE	EQU	$B9		; Write EEPROM memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$0A90		; Kernel ID
;
	ORG	ROM
	RJMP	BEGIN		; Execute start code
; Interrupt vectors (if you need them) go here!
; Unused vectors should point to the kernel entry point
	RJMP	enter		;1 External interrupt 0
	RJMP	enter		;2 External interrupt 1
	RJMP	enter		;3 Timer1 input capture
	RJMP	enter		;4 Timer1 output compareA
	RJMP	enter		;5 Timer1 output compareB
	RJMP	enter		;6 Timer1 overflow
	RJMP	enter		;7 Timer0 overflow
	RJMP	enter		;8 SPI transfer
	RJMP	enter		;9 UART RX
	RJMP	enter		;A UART Data reg empty
	RJMP	enter		;B UART TX complete
	RJMP	enter		;C Analog comparator
;
; Return control to DMS kernel
;
enter:	PUSH	R28		; Save R28
	IN	R28,SREG	; Get SREG
	PUSH	R28		; Save it
	LDI	R28,BRKF1	; Breakpoint flag 1
	RCALL	wrchr		; Send to host
	LDI	R28,BRKF2	; Breakpoint flag 2
	RCALL	wrchr		; Send to host
	LDI	R28,BRKF3	; Breakpoint flag 3
	RCALL	wrchr		; Send to host
	CLR	R28		; No adjust
	RCALL	wrchr		; Send to host
	POP	R28		; Get SREG
	RCALL	wrchr		; Send SREG to host
	MOV	R28,R29		; Get R29
	RCALL	wrchr		; Send R29 to host
	POP	R28		; Restore R28
	RCALL	wrchr		; Send R28 to host
	RCALL	wraddr		; Send R30:31
	MOV	R30,R26
	MOV	R31,R27
	RCALL	wraddr		; Send R26:27
	MOV	R30,R24
	MOV	R31,R25
	RCALL	wraddr		; Send R24:25
	MOV	R30,R22
	MOV	R31,R23
	RCALL	wraddr		; Send R22:23
	MOV	R30,R20
	MOV	R31,R21
	RCALL	wraddr		; Send R20:21
	MOV	R30,R18
	MOV	R31,R19
	RCALL	wraddr		; Send R18:19
	MOV	R30,R16
	MOV	R31,R17
	RCALL	wraddr		; Send R16:17
	MOV	R30,R14
	MOV	R31,R15
	RCALL	wraddr		; Send R14:15
	MOV	R30,R12
	MOV	R31,R13
	RCALL	wraddr		; Send R12:13
	MOV	R30,R10
	MOV	R31,R11
	RCALL	wraddr		; Send R10:11
	MOV	R30,R8
	MOV	R31,R9
	RCALL	wraddr		; Send R8:9
	MOV	R30,R6
	MOV	R31,R7
	RCALL	wraddr		; Send R6:7
	MOV	R30,R4
	MOV	R31,R5
	RCALL	wraddr		; Send R4:5
	MOV	R30,R2
	MOV	R31,R3
	RCALL	wraddr		; Send R2:3
	MOV	R30,R0
	MOV	R31,R1
	RCALL	wraddr		; Send R0:1
	POP	R31		; Get PC high
	POP	R30		; Get PC low
	LSL	R30		; *2
	ROL	R31		; for word entries
	RCALL	wraddr		; Send PC
	IN	R30,SP
	IN	R31,SP+1
	RCALL	wraddr		; Send to host
	RJMP	main		; re-enter monitor
; Initialize monitor
BEGIN	LDI	R30,STACK	; Get low hardware stack
	LDI	R31,=STACK	; Get high hardware stack
	OUT	SP,R30		; Set low hardware stack
	OUT	SP+1,R31	; Set high hardware stack
; Place additional initialization here
	CLR	R28		; Get a zero
	OUT	MCUCR,R28	; Disable external SRAM
; Setup serial port
	LDI	R28,BAUD	; Baud rate for 9600 & 4.0 Mhz
	OUT	UBRR,R28	; Write it
	LDI	R28,%00011000	; No interrupts, enable RX+TX, 8bit
	OUT	UCR,R28		; Write it
; Read input data until it expires
flush:	RCALL	getchr		; Get data
	RJMP	flush		; Until timeout
;
; Write address
;
wraddr:	MOV	R28,R31		; Get high
	RCALL	wrchr		; Write it
	MOV	R28,R30		; Get low
;
; Write character in R28 to serial port
;
wrchr	SBIS	USR,5		; Transmit empty?
	RJMP	wrchr		; No, wait
	OUT	UDR,R28		; Write to uart
	CPI	R28,$0A		; Newline?
	RET
;
; Get an address into R31:R30
;
getaddr	RCALL	getchr		; Get high
	MOV	R31,R28		; Save it
	RCALL	getchr		; Get low
	MOV	R30,R28		; Save it
	RET
;
; Get character with timeout
; re-enter monitor on timeout
;
getchr	LDI	R27,0		; Zero high
	MOV	R26,R27		; Zero high
getch1:	SBIS	USR,7		; Receive ready?
	RJMP	getch2		; No, try next
	IN	R28,UDR		; Read data
	RET
getch2:	SBIW	X,1		; Reduce count
	BRNE	getch1		; Wait till expiry
;
; Main monitor command loop
;
main:	LDI	R30,STACK	; Get low hardware stack
	LDI	R31,=STACK	; Get high hardware stack
	OUT	SP,R30		; Set low hardware stack
	OUT	SP+1,R31	; Set high hardware stack
main1:	RCALL	getchr		; Get next character
;
; Query command
;
	CPI	R28,CQUERY	; Query?
	BRNE	mreadp		; No, try next
	LDI	R30,qresp	; Point to response
	LDI	R31,=qresp	; ""
qloop:	LPM			; Get data from memory
	TST	R0		; End of data?
	BREQ	main1		; Yes, exit
	MOV	R28,R0		; Copy to accumulator
	RCALL	wrchr		; Write the data
	ADIW	Z,1		; Skip to next
	RJMP	qloop		; Do it all
qresp:	DB	RESP1,RESP2,=KID,KID,1,0
;
; Read Program memory
;
mreadp:	CPI	R28,CREADP	; Read memory?
	BRNE	mreadd		; No, try next
	RCALL	getaddr		; Get the address
	RCALL	getchr		; Get size
	MOV	R29,R28		; Save size
mrp1:	LPM			; Read byte from memory
	MOV	R28,R0		; Get data value
	RCALL	wrchr		; Send to host
	ADIW	Z,1		; Advance to next
	DEC	R29		; Reduce count
	BRNE	mrp1		; Do them all
	RJMP	main1		; And exit
;
; Read data memory
;
mreadd:	CPI	R28,CREADD	; Read data
	BRNE	mreade		; No, try next
	RCALL	getaddr		; Get address
	RCALL	getchr		; Get size
	MOV	R29,R28		; Save size
mrd1:	LD	R28,Z		; Get data
	RCALL	wrchr		; Send to host
	ADIW	Z,1		; Advance to next
	DEC	R29		; Reduce count
	BRNE	mrd1		; Do them all
	RJMP	main1		; And exit
;
; Read EEPROM memory
;
mreade:	CPI	R28,CREADE	; Read eeprom?
	BRNE	mwritd		; No, try next
	RCALL	getaddr		; Get address
	RCALL	getchr		; Get size
	MOV	R29,R28		; Save size
mre1:	OUT	EEAR,R30	; Write low address
	OUT	EEAR+1,R31	; WRite high address
	SBI	EECR,0		; Set to read
	ADIW	Z,1		; Skip to next
	IN	R28,EEDR	; Read data
	RCALL	wrchr		; Send to host
	DEC	R29		; Reduce count
	BRNE	mre1		; Do them all
	RJMP	main1		; And exit
;
; Write data memory
;
mwritd:	CPI	R28,CWRITD	; Write data memory?
	BRNE	mwrite		; No, try next
	RCALL	getaddr		; Get address
	RCALL	getchr		; Get size
	MOV	R29,R28		; Save size
mwd1:	RCALL	getchr		; Get data
	ST	Z,R28		; Write it
	ADIW	Z,1		; Skip to next
	DEC	R29		; Reduce count
	BRNE	mwd1		; Do them all
	RJMP	main1		; And exit
;
; Write EEPROM memory
;
mwrite:	CPI	R28,CWRITE	; Write EEPROM memory?
	BRNE	exec		; No, try next
	RCALL	getaddr		; Get address
	RCALL	getchr		; Get size
	MOV	R29,R28		; Save size
mwe1:	RCALL	getchr		; Get character
	SBIC	EECR,1		; Wait for EEWE=0
	RJMP	mwe2		; Skip this one
	OUT	EEAR,R30	; Write low address
	OUT	EEAR+1,R31	; Write high address
	OUT	EEDR,R28	; Write the data
	SBI	EECR,2		; Set EEMWE
	SBI	EECR,1		; Set EEWE
mwe2:	ADIW	Z,1		; Skip to next
	DEC	R29		; Reduce count
	BRNE	mwe1		; Do them all
	RJMP	main1		; And exit
;
; Execute user program
;
exec:	CPI	R28,CEXEC	; Execute?
	BRNE	step		; No, try next
	RCALL	getaddr		; Get SP
	OUT	SP,R30		; Set stack low
	OUT	SP+1,R31	; Set stack high
	RCALL	dnldreg		; Download general registers
	PUSH	R30		; Stack low
	PUSH	R31		; Stack high
	RCALL	getaddr		; Get R26:27
	PUSH	R30		; Save R26
	PUSH	R31		; Save R27
	RCALL	getaddr		; Get R30:31
	RCALL	getchr		; Get R29
	MOV	R29,R28		; Set it
	RCALL	getchr		; Get R28
	PUSH	R28		; Save it
	RCALL	getchr		; Get SREG
	OUT	SREG,R28	; Set it
	POP	R28		; Restore R28
	POP	R27		; Restore R27
	POP	R26		; Restore R26
	RET
;
; Single-step user program
;
step:	CPI	R28,CSTEP	; Single-Step?
	BRNE	gomain		; No, try next
	RCALL	getaddr		; Get SP
	OUT	SP,R30		; Set stack low
	OUT	SP+1,R31	; Set stack high
	RCALL	dnldreg		; Download general registers
	PUSH	R30		; Stack low
	PUSH	R31		; Stack high
	RCALL	getaddr		; Get R26:27
	PUSH	R30		; Save R26
	PUSH	R31		; Save R27
	RCALL	getaddr		; Get R30:31
	RCALL	getchr		; Get R29
	MOV	R29,R28		; Set it
	RCALL	getchr		; Get R28
	PUSH	R28		; Save it
	RCALL	getchr		; Get SREG
; Setup timer to cause an interrupt 1 cycle into user program
	ORI	R28,%10000000	; Enable interrupts in saved CC
	CLR	R26		; Get zero
	OUT	TCCR0,R26	; Stop timer
; This timer count must wait until all of the instructions up to the
; 'RET' at the end of this routine is executed
	LDI	R26,-11		; Get count value
	OUT	TCNT0,R26	; Set initial count value
	LDI	R26,%00000010	; Timer 0 overflow
	OUT	TIFR,R26	; Clear any pending interrupt
	IN	R26,TIMSK	; Get timer control
	ORI	R26,%00000010	; Enable timer0 overflow
	OUT	TIMSK,R26	; Re-write it
	LDI	R26,%00000001	; Prescale=0
	OUT	TCCR0,R26	; Start timer
; Continue with launching program (count cycles for timer value)
	OUT	SREG,R28	;1 Set it
	POP	R28		;2 Restore R28
	POP	R27		;2 Restore R27
	POP	R26		;2 Restore R26
	RET			;4 Launch program
gomain:	RJMP	main
;
; Download the general registers from the host
;
DNLDREG	RCALL	getaddr		; Get R0:1
	MOV	R0,R30
	MOV	R1,R31
	RCALL	getaddr		; Get R2:3
	MOV	R2,R30
	MOV	R3,R31
	RCALL	getaddr		; Get R4:5
	MOV	R4,R30
	MOV	R5,R31
	RCALL	getaddr		; Get R6:7
	MOV	R6,R30
	MOV	R7,R31
	RCALL	getaddr		; Get R8:9
	MOV	R8,R30
	MOV	R9,R31
	RCALL	getaddr		; Get R10:11
	MOV	R10,R30
	MOV	R11,R31
	RCALL	getaddr		; Get R12:13
	MOV	R12,R30
	MOV	R13,R31
	RCALL	getaddr		; Get R14:15
	MOV	R14,R30
	MOV	R15,R31
	RCALL	getaddr		; Get R16:17
	MOV	R16,R30
	MOV	R17,R31
	RCALL	getaddr		; Get R18:19
	MOV	R18,R30
	MOV	R19,R31
	RCALL	getaddr		; Get R20:21
	MOV	R20,R30
	MOV	R21,R31
	RCALL	getaddr		; Get R22:23
	MOV	R22,R30
	MOV	R23,R31
	RCALL	getaddr		; Get R24:25
	MOV	R24,R30
	MOV	R25,R31
	RCALL	getaddr		; Get PC & exit
	LSR	R31		; /2
	ROR	R30		; for word PC
	RET
