;
; Dunfield MicroScope - Target Kernel for: 8086
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
UART	EQU	$3F8		; Serial port address (8250)
BAUD	EQU	6		; 115200/6 = 19200
; Command code from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CUPDT	EQU	$A4		; Update registers
CREADC	EQU	$B0		; Read CS memory
CREADD	EQU	$B1		; Read DS memory
CREADE	EQU	$B2		; Read ES memory
CREADS	EQU	$B3		; Read SS memory	
CWRITC	EQU	$B8		; Write CS memory
CWRITD	EQU	$B9		; Write DS memory
CWRITE	EQU	$BA		; Write ES memory
CWRITS	EQU	$BB		; Write SS memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$8086		; Kernel ID

; Monitor data area
	ORG	$0		; Data goes here
SAVCS	DS	2		; Saved CS
SAVDS	DS	2		; Saved DS
SAVES	DS	2		; Saved ES
SAVSS	DS	2		; Saved SS
temp1	DS	2		; Temporary location
temp2	DS	2		; Temporary location
temp3	DS	2		; Temporary location
	DS	1000		; Some stack space
STACK	EQU	*		; Stack goes here
;
	ORG	$0100
; Initialize the 8250 uart device
	MOV	AL,#%10000011	; Access divisior latch
	MOV	DX,#UART+3	; Address Line Control register
	OUT	DX,AL		; Write data
	MOV	AL,#BAUD	; Get baudrate divisor
	MOV	DX,#UART	; Address latch LSB
	OUT	DX,AL		; Write data
	XOR	AL,AL		; Zero high
	INC	DX		; Advance to MSB
	OUT	DX,AL		; Write data
	MOV	AL,#%00000011	; Access data registers
	MOV	DX,#UART+3	; Address Line Control register
	OUT	DX,AL		; Write data
	XOR	AL,AL		; No interrupts
	MOV	DX,#UART+1	; Address Interrupt Enable register
	OUT	DX,AL		; Write data
	MOV	AL,#%00000011	; Assert DTR and RTS
	MOV	DX,#UART+4	; Address Modem Control register
	OUT	DX,AL		; Write data
; Initialize monitor and wait for commands
IMON:	MOV	AX,CS		; Get our segment
	ADD	AX,#4096	; Advance to next
	MOV	DS,AX		; Set Data seg
	MOV	SS,AX		; Set stack
	MOV	SP,#STACK	; Point to stack
; Hook INT1 and INT3 vectors
; Used for single-stepping and breakpoints
	XOR	AX,AX		; Get zero
	MOV	ES,AX		; Address interrupts
	MOV	AX,CS		; Get our code segment
	MOV	ES:>04,#ITRAP	; Set single-step vector
	MOV	ES:06,AX	; Set segment
	MOV	ES:>12,#IBREAK	; Set breakpoint vector
	MOV	ES:14,AX	; Set segment
;
;	call	db_str
;	strz	'Init!'
cmd:	call	db_exit		; Test for console exit
	CALL	CHKCHR		; Any command waiting
	JZ	cmd		; No, wait for it
;	call	db_hex
; We have received command, check for options
	CMP	AL,#CQUERY	; Query command?
	JNZ	readc		; No, try next
; Query command - send back kernel information
	MOV	BX,#qresp	; Point to response
qloop:	MOV	AL,CS:[BX]	; Get data
	INC	BX		; Skip to next
	AND	AL,AL		; End of table?
	JZ	cmd		; Yes, exit
	CALL	WRICHR		; Write the character
	JMP	<qloop		; Do them all
qresp:	DB	RESP1,RESP2,=KID,KID,1,0
; Read CS memory
readc:	CMP	AL,#CREADC	; Read CS memory?
	JNZ	readd		; No, try next
	MOV	ES,savcs	; Get saved CS
rdmem:	CALL	getaddr		; BX = address
	CALL	getchr		; Get a byte
	MOV	CL,AL		; Set count
rdm1:	MOV	AL,ES:[BX]	; Get a byte
	CALL	WRICHR		; Send it
	INC	BX		; Advance to next
	DEC	CL		; Reduce count
	JNZ	rdm1		; Do them all
	JMP	<cmd		; Next command
; Read DS memory
readd:	CMP	AL,#CREADD	; Read DS memory?
	JNZ	reade		; No, try next
	MOV	ES,savds	; Get saved DS
	JMP	<rdmem		; And do the read
; Read ES memory
reade:	CMP	AL,#CREADE	; Read ES memory?
	JNZ	reads		; No, try next
	MOV	ES,saves	; Get saved ES
	JMP	<rdmem		; And do the read
; Read SS memory
reads:	CMP	AL,#CREADS	; Read SS memory?
	JNZ	writec		; No, try next
	MOV	ES,savss	; Get saved SS
	JMP	<rdmem		; And do the read
; Write CS memory
writec:	CMP	AL,#CWRITC	; Write CS memory?
	JNZ	writed		; No, try next
	MOV	ES,savcs	; Get saved CS
wrmem:	CALL	getaddr		; BX = address
	CALL	getchr		; Get a byte
	MOV	CL,AL		; Set count
wrm1:	CALL	getchr		; Get a byte
	MOV	ES:[BX],AL	; Write it
	INC	BX		; Advance to next
	DEC	CL		; Reduce count
	JNZ	wrm1		; Do them all
	JMP	<cmd1
; Write DS memory
writed:	CMP	AL,#CWRITD	; Write DS memory?
	JNZ	writee		; No, try next
	MOV	ES,savds	; Get saved DS
	JMP	<wrmem		; Do the write
; Write ES memory
writee:	CMP	AL,#CWRITE	; Write ES memory?
	JNZ	writes		; No, try next
	MOV	ES,saves	; Get saved ES
	JMP	<wrmem		; Do the write
; Write SS memory
writes:	CMP	AL,#CWRITS	; Write SS memory?
	JNZ	break		; No, try next
	MOV	ES,saves	; Get saved SS
	JMP	<wrmem		; Do the write
; Set breakpoint on target system
break:	CMP	AL,#CBREAK	; Set breakpoint?
	JNZ	updt		; No, try next
	CALL	getaddr		; Get the address
	MOV	ES,savcs	; Point to code space
	MOV	AL,ES:[BX]	; Get the current data
	CALL	wrichr		; Send to host
	MOV	AL,#$CC		; Int-3 breakpoint
	MOV	ES:[BX],AL	; Install breakpoint
	SUB	AL,ES:[BX]	; Test it
	CALL	wrichr		; Return test result
cmd1:	JMP	cmd		; Next command
; Update monitor segment registers
updt:	CMP	AL,#CUPDT	; Update registers?
	JNZ	step		; No, try next
	CALL	getaddr		; Download CS
	MOV	savcs,BX
	CALL	getaddr		; Download DS
	mov	savds,BX
	CALL	getaddr		; Download ES
	MOV	saves,BX
	CALL	getaddr		; Download SS
	MOV	savss,BX
	JMP	<cmd1		; Next command
; Step one instruction
step:	CMP	AL,#CSTEP	; Step command?
	JNZ	exec		; No, try next
	CALL	GETADDR		; Download SS
	MOV	savss,BX	; Save it
	CALL	GETADDR		; Download SP
	CLI			; Insure no interrupts
	MOV	SP,BX		; Set sp
	MOV	SS,savss	; Set ss
	CALL	getaddr		; Get flags
	OR	BH,#$01		; Set TRAP flag
	JMP	<exec1		; and continue
; Execute program
exec:	CMP	AL,#CEXEC	; Execute command
	JNZ	badcmd		; No, try next
	CALL	GETADDR		; Download SS
	MOV	savss,BX	; Save it
	CALL	GETADDR		; Download SP
	CLI			; Insure no interrupts
	MOV	SP,BX		; Set sp
	MOV	SS,savss	; Set ss
	CALL	getaddr		; Get flags
	AND	BH,#$FE		; Clear TRAP flag
; Read remaining registers and launch program
exec1:	PUSH	BX		; Save flags
	CALL	getaddr		; Download CS
	PUSH	BX		; Save it
	CALL	getaddr		; Download PC
	PUSH	BX		; Save it
	CALL	getaddr		; Download DS
	MOV	DS,BX
	CALL	getaddr		; Download ES
	MOV	ES,BX
	CALL	getaddr		; Download SI
	MOV	SI,BX
	CALL	getaddr		; Download DI
	MOV	DI,BX
	CALL	getaddr		; Download BP
	MOV	BP,BX
	CALL	getaddr		; Download DX
	MOV	CX,BX
	CALL	getaddr		; Download CX
	MOV	CX,BX
	CALL	getaddr		; Download BX
	CALL	getchr		; Download AH
	MOV	AH,AL
	CALL	getchr		; Download AL
	IRET
;
badcmd:	CALL	db_hex
	CALL	db_str
	strz	'BadCmd!'
	JMP	cmd
;
; INT3 interrupt - breakpoint
;
IBREAK:	PUSH	AX		; Save AX
	PUSH	DS		; Save DS
	MOV	AX,CS		; Get code segment
	ADD	AX,#4096	; Offset to ours
	MOV	DS,AX		; Set new DS
	POP	>savds		; Restore saved DS
	POP	>temp1		; Save AX
	POP	>temp2		; Save PC
	POP	>savcs		; Save CS
	POP	>temp3		; Save flags
	MOV	savss,SS	; Save SS
	MOV	saves,SP	; Save SP (for now)
	MOV	SS,AX		; Switch to our
	MOV	SP,#stack	; own Internal stack
	MOV	AL,#BRKF1	; Breakpoint response 1
	CALL	wrichr		; Send it
	MOV	AL,#BRKF2	; Breakpoint response 2
	CALL	wrichr		; Send it
	MOV	AL,#BRKF3	; Breakpoint response 3
	CALL	wrichr		; Send it
	MOV	AL,#1		; Adjust by 1
	JMP	<intx		; and proceed
;
; TRAP interrupt - single-step return
;
ITRAP:	PUSH	AX		; Save AX
	PUSH	DS		; Save DS
	MOV	AX,CS		; Get code segment
	ADD	AX,#4096	; Offset to out DS
	MOV	DS,AX		; Set new DS
	POP	>savds		; Restore saved DS
	POP	>temp1		; Save AX
	POP	>temp2		; Save PC
	POP	>savcs		; Save CS
	POP	>temp3		; Save flags
	MOV	savss,SS	; Save SS
	MOV	saves,SP	; Save SP (for now)
	MOV	SS,AX		; Switch to our segment
	MOV	SP,#stack	; own Internal stack
	MOV	AL,#BRKF1	; Breakpoint response 1
	CALL	wrichr		; Send it
	MOV	AL,#BRKF2	; Breakpoint response 2
	CALL	wrichr		; Send it
	MOV	AL,#BRKF3	; Breakpoint response 3
	CALL	wrichr		; Send it
	MOV	AL,#0		; No adjust
intx:	CALL	wrichr		; Send it
	MOV	AX,savss	; Upload SS
	CALL	wriwrd
	MOV	AX,saves	; Upload SP
;;	ADD	AX,#6		; Adjust for overhead
	CALL	wriwrd
	MOV	AX,temp2	; Upload PC
	CALL	wriwrd
	MOV	AX,savcs	; Upload CS
	CALL	wriwrd
	MOV	AX,temp3	; Upload flags
	CALL	wriwrd
	MOV	AX,savds	; Upload DS
	CALL	wriwrd
	MOV	AX,ES		; Upload ES
	MOV	saves,AX	; Save for mem access
	CALL	wriwrd
	MOV	AX,SI		; Upload SI
	CALL	wriwrd
	MOV	AX,DI		; Upload DI
	CALL	wriwrd
	MOV	AX,BP		; Upload BP
	CALL	wriwrd
	MOV	AX,DX		; Upload DX
	CALL	wriwrd
	MOV	AX,CX		; Upload CX
	CALL	wriwrd
	MOV	AX,BX		; Upload BX
	CALL	wriwrd
	MOV	AX,temp1	; Upload AX
	CALL	wriwrd
	JMP	IMON		; And re-enter monitor
;
; Get address into BX
;
getaddr	CALL	getchr		; Get HIGH byte
	MOV	BH,AL		; Set it
	CALL	getchr		; Get LOW byte
	MOV	BL,AL		; Set it
	RET
;
; Get a character - if timeout, back to monitor
;
getchr	PUSH	BX		; Save BX
	PUSH	CX		; Save count
	XOR	BX,BX		; Zero counter
	MOV	CX,#10		; X count
getc1:	CALL	chkchr		; Check for character
	JNZ	getc2		; We have it
	DEC	BX		; Reduce count
	JNZ	getc1		; No timeout
	LOOP	getc1		; Full timeout
;	CALL	db_str
;	strz	'Time!'
	JMP	IMON
getc2:	POP	CX		; Restore CX
	POP	BX		; Restore BX
	RET
;
; Write a word (AL:AH) to system
;
WRIWRD:	CALL	wrichr		; Write the character
	MOV	AL,AH		; Get high
;
; Write char in AL to 8250 type UART device.
;
WRICHR	PUSH	DX		; Save DX
	PUSH	AX		; Save AX
	MOV	DX,#UART+5	; Address status register
wric1	IN	AL,DX		; Read status register
	TEST	AL,#$20		; Transmitter ready?
	JZ	wric1		; No, wait for it
	MOV	DX,#UART	; Address data register
	POP	AX		; Restore character
	OUT	DX,AL		; Write to uart
	POP	DX		; Restore DX
	RET
; Test 8250 type UART device.
CHKCHR	PUSH	DX		; Save DX
	MOV	DX,#UART+5	; Address status register
	IN	AL,DX		; Read status register
	AND	AL,#$01		; Receiver ready
	JZ	chkc1		; No, return with zero
	MOV	DX,#UART	; Address data register
	IN	AL,DX		; Read data
chkc1	POP	DX		; Restore DX
	RET
;
; DEBUG support functions
; -- Perform I/O via PC-BIOS calls to perform kernel debugging
; -- remote these functions & calls to them in production kernel
;
; Test for console ESC and exit to DOS if found
db_exit: mov	ah,#1
	int	$16
	jz	nochr
	xor	ah,ah
	int	$16
	cmp	al,#$1B
	jnz	nochr
	xor	ax,ax
	int	$21
nochr:	ret
; Display a string on the debug console
db_str:	POP	BX		; Get return address
dbs1:	MOV	AL,CS:[BX]	; Get data
	INC	BX		; Next
	PUSH	BX		; Save counter
	AND	AL,AL		; End?
	JZ	dbs2		; Yes, exit
	MOV	AH,#$0E		; Write console character
	MOV	BX,#$0007	; Page=0, Color=White
	INT	$10		; Call BIOS
	POP	BX		; Restore BX
	JMP	<dbs1
dbs2:	RET
; Write HEX character[AL] to debug console
db_hex:	PUSH	BX		; Save BX
	PUSH	AX
	MOV	AH,#$0E
	MOV	AL,#'['
	MOV	BX,#$0007
	INT	$10
	MOV	BX,#7		; Get 7
	POP	AX
	PUSH	AX
	ROL	AL,1
	ROL	AL,1
	ROL	AL,1
	ROL	AL,1
	CALL	hnib
	POP	AX
	PUSH	AX
	CALL	hnib
	MOV	AH,#$0E
	MOV	AL,#']'
	MOV	BX,#$0007
	INT	$10
	OR	AL,#-1
	POP	AX
	POP	BX
	RET
; Write a nibble
hnib:	AND	AL,#%00001111
	ADD	AL,#'0'
	CMP	AL,#'9'+1
	JB	hnok
	ADD	AL,#7
hnok:	MOV	AH,#$0E
	MOV	BX,#$0007
	INT	$10
	RET
