;
; Dunfield MicroScope - Target Kernel for: 8085
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
;
ROM	EQU	$0000		; Debugger goes here
STACK	EQU	$A000		; Stack goes here
UBASE	EQU	$4000		; User program base address
; Command codes from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREAD	EQU	$B0		; Read Program memory
CWRITE	EQU	$B8		; Write program memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$8085		; Kernel ID
;
	ORG	ROM		; Debugger code goes here
;
	LXI	SP,STACK	; Set up initial stack pointer
	JMP	begin		; Execute main program
	DS	2		; Filler bytes to first int
;
; Interrupt handlers for RESTART interrupts
;
; Although they RST 1.5, 2.5 and 3.5 vectors are not used by the
; 8085 hardware,  they are included since the space must contain
; SOMETHING,  and who knows,  perhaps someone uses them for jump
; table addresses etc...
;
; Restart 1 is the entry point for breakpoints
RST1	JMP	ENTRY		; Execute handler
	DS	1		; Filler to next int
RST15	CALL	RSTINT		; Invoke interrupt
	DB	12		; Offset to handler
RST2	CALL	RSTINT		; Invoke interrupt
	DB	16		; Offset to handler
RST25	CALL	RSTINT		; Invoke interrupt
	DB	20		; Offset to handler
RST3	CALL	RSTINT		; Invoke interrupt
	DB	24		; Offset to handler
RST35	CALL	RSTINT		; Invoke interrupt
	DB	28		; Offset to handler
RST4	CALL	RSTINT		; Invoke interrupt
	DB	32		; Offset to handler
TRAP	CALL	RSTINT		; Invoke interrupt
	DB	36		; Offset to handler
RST5	CALL	RSTINT		; Invoke interrupt
	DB	40		; Offset to handler
RST55	CALL	RSTINT		; Invoke interrupt
	DB	44		; Offset to handler
RST6	CALL	RSTINT		; Invoke interrupt
	DB	48		; Offset to handler
RST65	CALL	RSTINT		; Invoke interrupt
	DB	52		; Offset to handler
RST7	CALL	RSTINT		; Invoke interrupt
	DB	56		; Offset to handler
RST75	CALL	RSTINT		; Invoke interrupt
	DB	60		; Offset to handler
;
; Process a RESTART interrupt, get offset & vector to code
; To speed processing, it is assumed that the user program
; base address begins on a 256 byte page boundary.
;
RSTINT	XTHL			; Save HL, Get PTR to offset
	PUSH	PSW		; Save A and CC
	MOV	A,M		; Get offset
	LXI	H,UBASE		; Get user program address
	MOV	L,A		; Set low address
	POP	PSW		; Restore A & CC
	XTHL			; Restore HL, set 
	RET			; Vector to interrupt
; Initialize timer and uart
; 32=9600 16=19200 8=38400
begin:	LXI	H,16		; Time count for 19200
	MVI	A,%10110110	; Timer 2, Mode 3
	OUT	$8B		; Write to MODE port
	MOV	A,L		; Get LOW value
	OUT	$8A		; Write TIMER 2
	MOV	A,H		; Get HIGH value
	OUT	$8A		; Write TIMER 2
	XRA	A		; Get a ZERO
	OUT	$81		; Clear out 8251
	OUT	$81		; "" "" ""
	OUT	$81		; "" "" ""
	MVI	A,%01000000	; Internal RESET
	OUT	$81		; Write 8251 Control
	MVI	A,%01001110	; 8Bit, 1Stop, NoPar, x16
	OUT	$81		; Write 8251 control
	MVI	A,%00110111	; RTS, DTR, RxEn, TxEn, ErrRst
	OUT	$81		; Write 8251 control
flush:	CALL	getchr		; Get character
	JMP	flush		; Until timeout
;
; Write address in HL to host
;
wraddr:	MOV	A,H		; Get high
	CALL	wrchr		; Write it
	MOV	A,L		; Get low
;
; Write character in A to host
;
wrchr:	PUSH	PSW		; Save char
wrch1	IN	$81		; Get 8251 status
	RRC			; Test TX bit
	JNC	wrch1		; Not ready
	POP	PSW		; Restore char
	OUT	$80		; Write 8251 data
	RET
;
; Get an address into the HL register paid
;
getaddr	CALL	getchr		; Get high byte
	MOV	H,A		; Save it
	CALL	getchr		; Get low byte
	MOV	L,A
	RET
;
; Get character from host with timeout
; Re-enter monitor if timeout
;
getchr	LXI	B,0		; Reset timer
getch1:	IN	$81		; Get 8251 status
	ANI	%00000010	; Test for ready
	JZ	getch2		; No character
	IN	$80		; Get 8251 data
	RET
getch2:	DCX	B		; Reduce count
	MOV	A,B		; Get high
	ORA	C		; Test low
	JNZ	getch1		; No timeout yet
;
; Main monitor command loop
;
main:	LXI	SP,STACK	; Reset stack in case abort
main1:	CALL	getchr		; Get a character
	CPI	CREAD		; Read memory?
	JZ	mread		; Handle it
	CPI	CWRITE		; Write memory?
	JZ	mwrite		; Handle it
	CPI	CBREAK		; Set breakpoint?
	JZ	break		; Handle it
	CPI	CEXEC		; Execute?
	JZ	exec		; Handle it
	CPI	CQUERY		; Query kernel?
	JNZ	main1		; No, ignore
;
; Query kernel
;
	LXI	H,qresp		; Point top response
qloop:	MOV	A,M		; Get data
	ANA	A		; End of list?
	JZ	main1		; All done
	CALL	wrchr		; Write it
	INX	H		; Advance to next
	JMP	qloop		; And do them all
qresp:	DB	RESP1,RESP2,=KID,KID,1,0
;
; Read memory
;
mread:	CALL	getaddr		; Get address
	CALL	getchr		; Get size
	MOV	D,A		; Save size
mread1:	MOV	A,M		; Get data
	CALL	wrchr		; Write it
	INX	H		; Skip to next
	DCR	D		; Reduce count
	JNZ	mread1		; Do them all
	JMP	main1		; And exit
;
; Write memory
;
mwrite:	CALL	getaddr		; Get address
	CALL	getchr		; Get size
	MOV	D,A		; Save size
mwrit1:	CALL	getchr		; Get a character
	MOV	M,A		; Write to memory
	INX	H		; Skip to next
	DCR	D		; Reduce count
	JNZ	mwrit1		; Do them all
	JMP	main1		; And exit
;
; Set a breakpoint
;
break:	CALL	getaddr		; Get address
	MOV	A,M		; Get content
	CALL	wrchr		; Send to host
	MVI	M,$CF		; 'RST 1' = breakpoint
	MOV	A,M		; Get data
	SBI	$CF		; Did it work?
	CALL	wrchr		; Send to host
	JMP	main1		; And proceed
;
; Execute user program
;
exec:	CALL	getaddr		; Download SP
	SPHL			; Set SP
	CALL	getaddr		; Download PC
	PUSH	H		; Save it
	CALL	getaddr		; Download DE
	XCHG			; Set DE
	CALL	getaddr		; Download BC
	PUSH	H		; Save it
	CALL	getaddr		; Download A:PSW
	PUSH	H		; Save it
	CALL	getaddr		; Get HL
	POP	PSW		; Restore A:PSW
	POP	B		; Restore BC
	RET			; Execute
;
; Re-enter monitor after breakpoint
;
entry:	PUSH	PSW		; Save A:PSW
	MVI	A,BRKF1		; Breakpoint flag 1
	CALL	wrchr		; Send it
	MVI	A,BRKF2		; Breakpoint flag 2
	CALL	wrchr		; Send it
	MVI	A,BRKF3		; Breakpoint flag 3
	CALL	wrchr		; Send it
	MVI	A,1		; Adjust by 1 for breakpoint
	CALL	wrchr		; Send it
	CALL	wraddr		; Upload HL
	POP	H		; Get A:PSW
	CALL	wraddr		; Upload A:PSW
	MOV	H,B		; Get B
	MOV	L,C		; Get C
	CALL	wraddr		; Upload BC
	XCHG			; Get DE
	CALL	wraddr		; Upload DE
	POP	H		; Get PC
	CALL	wraddr		; Upload PC
	LXI	H,0		; Start with 0
	DAD	SP		; Get SP
	CALL	wraddr		; Upload SP
	JMP	main		; And enter monitor
