;
; Dunfield MicroScope - Target Kernel for: 8051
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
; System parameters
BAUD	EQU	3		; Baudrate (9600*2=19200)
ROM	EQU	$0000		; Location target monitor
USERAM	EQU	$0800		; Location of user program
STACK	EQU	8		; Debugger stack position
; Command code from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREADP	EQU	$B0		; Read Program memory
CREADD	EQU	$B1		; Read Data memory
CREADI	EQU	$B2		; Read Internal memory
CREADS	EQU	$B3		; Read SFR memory	
CWRITP	EQU	$B8		; Write program memory
CWRITD	EQU	$B9		; Write data memory
CWRITI	EQU	$BA		; Write internal memory
CWRITS	EQU	$BB		; Write SFR memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$8051		; Kernel ID
;
	ORG	ROM		; Target monitor begins
	AJMP	begin		; Start debugger
* Re-vector interrupts to user program area
	ORG	ROM+$0003	; EXT Interrupt 0
	LJMP	USERAM+$03
	ORG	ROM+$000B	; Timer 0 overflow
	LJMP	USERAM+$0B
	ORG	ROM+$0013	; EXT Interrupt 1
	LJMP	USERAM+$13
	ORG	ROM+$001B	; Timer1 - Single step
	LJMP	sshand
	ORG	ROM+$0023	; RI+TI interrupt
	LJMP	USERAM+$23
	ORG	ROM+$002B	; TF2+EXF2
	LJMP	USERAM+$2B
;
; Initialize debugger
;
Begin	MOV	TMOD,#%00100000	; T1=8-bit auto-reload
	MOV	TH1,#-BAUD	; Timer-1 reload value
	MOV	TH2,#-BAUD	; Timer-2 initial value
	MOV	TCON,#%01001001	; Run1, hold 0
	MOV	SCON,#%01010010 ; Mode 1, REN, TXRDY, RXEMPTY
	MOV	PCON,#%10000000	; Double baud-rate
; Loop reading data till it times out
bloop	ACALL	getchr		; Wait for character
	SJMP	bloop		; Till none
;
; Write character to serial port
;
Wrchr	JNB	SCON.1,*	; Wait for TX ready
	CLR	SCON.1		; Indicate sending
	MOV	SBUF,A		; Write it
	RET
;
; Get memory access operands (ADDR-SIZE)
;
getmemx	ACALL	getaddr		; Get high address
	ACALL	getchr		; Get size
	MOV	R7,A		; Set it
	RET
;
; Get an address
;
getaddr	ACALL	getchr		; Get high address
	MOV	DPH,A		; Set it
	ACALL	getchr		; Get low address
	MOV	DPL,A		; Set it
	MOV	R1,A		; Also for internal
	RET
;
; Get character from serial port with timeout
; Re-enter monitor if timeout
;
Getchr	MOV	R5,#100		; Timeout count
	MOV	R6,#0		; 512 cycles
getc1	JNB	SCON.0,getc2	; No data yet
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Get data
	RET
getc2	DJNZ	R6,getc1	; Wait for it
	DJNZ	R5,getc1	; And wait for it
;
; Main command execution loop
;
main	MOV	SP,#STACK	; Reset stack
main1	ACALL	Getchr		; Check for character
;
; Query command
;
	CJNE	A,#CQUERY,readp	; Try break command
	MOV	DPTR,#qresp	; Point to query response
qloop	CLR	A		; Zero offset
	MOVC	A,[A+DPTR]	; Get data byte
	JZ	main1		; End of list
	ACALL	wrchr		; Write it
	INC	DPTR		; Skip to next
	SJMP	qloop		; And proceed
qresp	DB	RESP1,RESP2,=KID,KID,1,0
;
; Read program memory
;
readp	CJNE	A,#CREADP,readd	; No, try next
	ACALL	getmemx		; Get memory operands
rdp1	CLR	A		; Zero A
	MOVC	A,[A+DPTR]	; Get byte of data
	ACALL	wrchr		; Send to host
	INC	DPTR		; Skip to next
	DJNZ	R7,rdp1		; Do them all
	AJMP	main1
;
; Read data memory
;
readd	CJNE	A,#CREADD,readi	; No, try next
	ACALL	getmemx		; Get memory operands
rdd1	MOVX	A,[DPTR]	; Get byte of data
	ACALL	wrchr		; Send to host
	INC	DPTR		; Skip to next
	DJNZ	R7,rdd1		; Do them all
	AJMP	main1
;
; Read internal memory
;
readi	CJNE	A,#CREADI,reads	; No, try next
	ACALL	getmemx		; Get memory operands
rdi1	MOV	A,[R1]		; Read byte of data
	ACALL	wrchr		; Send to host
	INC	R1		; Skip to next
	DJNZ	R7,rdi1		; Do them all
	AJMP	main1
;
; Read SFR memory
;
reads	CJNE	A,#CREADS,writp	; No, try next
	ACALL	getmemx		; Get memory operands
rds1	MOV	DPTR,#SFRRT	; Point to SFR read table
	ACALL	sfrx		; Get the value
	ACALL	wrchr		; Send to host
	INC	R1		; Skip to next
	DJNZ	R7,rds1		; Do them all
	AJMP	main1
;
; Write program memory
;
writp	CJNE	A,#CWRITP,writd	; No, try next
	AJMP	wrd0		; Same as data
;
; Write data memory
;
writd	CJNE	A,#CWRITD,writi	; No, try next
wrd0	ACALL	getmemx		; Get memory operands
wrd1	ACALL	getchr		; Get a data byte
	MOVX	[DPTR],A	; Write it
	INC	DPTR		; Skip to next
	DJNZ	R7,wrd1		; Do them all
	AJMP	main1
;
; Write internal memory
;
writi	CJNE	A,#CWRITI,writs	; No, try next
	ACALL	getmemx		; Get memory operands
wri1	ACALL	getchr		; Get data byte
	MOV	[R1],A		; Write it
	INC	R1		; Skip to next
	DJNZ	R7,wri1		; Do them all
	AJMP	main1
;
; Write SFR memory
;
writs	CJNE	A,#CWRITS,step	; No, try next
	ACALL	getmemx		; Get memory operands
wrs1	ACALL	getchr		; Get data byte
	MOV	R0,A		; Save for write
	MOV	DPTR,#SFRWT	; Point to write table
	ACALL	SFRX		; Access SFR
	INC	R1		; Advance
	DJNZ	R7,wrs1		; Do them all
	AJMP	main1
;
; Step one instruction
;
step	CJNE	A,#CSTEP,exec	; No, try next
	ACALL	GETCHR		; Download R0 value
	MOV	R0,#1		; Address
step1	JNB	SCON.0,*	; Wait for char
	CLR	SCON.0		; Indicate RX
	MOV	[R0],SBUF	; Write data
	INC	R0		; Advance
	CJNE	R0,#$20,step1	; Do them all
	MOV	0,A		; Write R0 value
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	SP,SBUF		; Read stack pointer
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Download PC.0
	PUSH	A		; Save for later
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Download PC.1
	PUSH	A		; Save for later
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Download ACC
	ORL	B,#%10001000	; Enable timer-1 interrupt
	CLR	IP.3		; Insure LOW priority
	PUSH	B		; Save program IE
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	B,SBUF		; Download B
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	DPL,SBUF	; Download DPTR.0
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	DPH,SBUF	; Download DPTR.1
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	PSW,SBUF	; Download PSW
* Set up Timer-1 for single step interrupt
	CLR	TCON.6		; Stop timer-1
	ANL	TMOD,#%00001111	; Zero timer-1 mode
	ORL	TMOD,#%00010000	; Timer-1 16 bit
	MOV	TH1,#-1		; Number of cycles
	MOV	TL1,#-4		; Till first user instruction
	CLR	TCON.7		; Clear timer-1 int pend.
* Enable timer for single-step interrupt
	SETB	TCON.6		; Enable timer-1
	POP	IE		; Set program IE
	RET			; Execute user program
;
; Launch program
;
exec	CJNE	A,#CEXEC,break	; No, try next
	ACALL	getchr		; Download R0 from host
	MOV	R0,#1		; memory address
exec1	JNB	SCON.0,*	; Wait for bit
	CLR	SCON.0		; Indicate received
	MOV	[R0],SBUF	; Read the data
	INC	R0		; Skip to next
	CJNE	R0,#$20,exec1	; Read them all
	MOV	0,A		; Write R0 value
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	SP,SBUF		; Read stack pointer
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Download PC.0
	PUSH	A		; Save for later
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Download PC.1
	PUSH	A		; Save for later
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	A,SBUF		; Download ACC
	PUSH	B		; Save program IE
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	B,SBUF		; Download B
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	DPL,SBUF	; Download DPTR.0
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	DPH,SBUF	; Download DPTR.1
	JNB	SCON.0,*	; Wait for data
	CLR	SCON.0		; Indicate received
	MOV	PSW,SBUF	; Download PSW
	POP	IE		; Restore program IE
	RET
;
; Set breakpoint on target system
;
break	CJNE	A,#CBREAK,xxx	; No, try next
	ACALL	getaddr		; Get an address
	MOV	R0,#0		; Indicate no prob
br1	MOV	R1,#$12		; LCALL instruction
	ACALL	brx		; Set and check
	MOV	R1,#=bphand	; Handler high-byte
	ACALL	brx		; Set and check
	MOV	R1,#bphand	; Handler low-byte
	ACALL	brx		; Set and check
;	MOV	A,R0		; Get flag
	MOV	A,#0
	ACALL	wrchr		; Send it
	AJMP	main1
brx	CLR	A		; Zero A
;	MOVC	A,[A+DPTR]	; Get byte from memory
	MOVX	A,[DPTR]
	ACALL	wrchr		; Send to host
	MOV	A,R1		; Get byte to send
	MOVX	[DPTR],A	; Write it
	CLR	A		; Zero A
;	MOVC	A,[A+DPTR]	; Read again
	MOVX	A,[DPTR]
	INC	DPTR		; Advance to next
	XRL	A,R1		; Does it match
	JZ	brx1		; Yes, it's OK
	INC	R0		; Indicate failure
brx1	RET
;
xxx	AJMP	main
;
; Execute SFR memory access function from DPTR/R1
SFRX	MOV	A,R1		; Get SFR address
	CLR	C		; Zero carry in
	RLC	A		; x2, drop high bit
	ADD	A,DPL		; Include in DP
	MOV	DPL,A		; Replace
	JNC	sfrx1		; No carry out
	INC	DPH		; Advance high
sfrx1	MOV	A,R1		; Get address again
	ANL	A,#%01111111	; Drop high bit
	JMP	[A+DPTR]	; Execute function
;
; Wait for TX complete
;
bpwtx	PUSH	A		Save ACC
	CLR	A		Zero ACC
; This DJNZ instruction should occur enough times to account for one
; character transmission time at the baud rate used by the target system
; Each DJNZ uses 512 cycles of CPU time (1 Cycle=12 XTAL periods)
; @12Mhz XTAL, each DJNZ provides .000512 sec delay
;
; 10x=0.00512 Sec delay, long enough for 2000 bps character
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	POP	A		Restore A
	SETB	SCON.1		Force TX ready
;
; Breakpoint handler
;
bphand	CLR	IE.3		; Disable Timer-1 interrupt
	JNB	SCON.1,bpwtx	; Wait for TX complete
	PUSH	IE		; Save Interrupt enable
	MOV	IE,#0		; Clear interrupts
* Restore timer-1 for serial I/O
	ANL	TMOD,#%00001111	; Zero timer-1 mode
	ORL	TMOD,#%00100000	; T1 = 8 bit auto-reload
	MOV	TH1,#-BAUD	; Timer-1 reload value
	MOV	TL1,#-BAUD	; Timer-1 initial value
* Signal a Breakpoint/Single-step return to the host
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#BRKF1	; Send breakpoint flag #1
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#BRKF2	; Send breakpoint flag #2
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#BRKF3	; Send breakpoint flag #3
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#3		; Adjust by -3
	SJMP	ctgo		; Continuewith contect switch
;
; Wait for TX complete
;
sswtx	PUSH	A		Save ACC
	CLR	A		Zero ACC
; This DJNZ instruction should occur enough times to account for one
; character transmission time at the baud rate used by the target system
; Each DJNZ uses 512 cycles of CPU time (1 Cycle=12 XTAL periods)
; @12Mhz XTAL, each DJNZ provides .000512 sec delay
;
; 10x=0.00512 Sec delay, long enough for 2000 bps character
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	DJNZ	A,*		Wait a bit
	POP	A		Restore A
	SETB	SCON.1		Force TX ready
;
; Single-Step interrupt handler
;
sshand	CLR	IE.3		; Disable Timer-1 interrupt
	JNB	SCON.1,sswtx	; Wait for TX complete
	PUSH	IE		; Save Interrupt enable
	MOV	IE,#0		; Clear interrupts
* Restore timer-1 for serial I/O
	ANL	TMOD,#%00001111	; Zero timer-1 mode
	ORL	TMOD,#%00100000	; T1 = 8 bit auto-reload
	MOV	TH1,#-BAUD	; Timer-1 reload value
	MOV	TL1,#-BAUD	; Timer-1 initial value
* Signal a Breakpoint/Single-step return to the host
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#BRKF1	; Send breakpoint flag #1
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#BRKF2	; Send breakpoint flag #2
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#BRKF3	; Send breakpoint flag #3
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,#0		; No adjust
* Upload registers to host
ctgo	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,PSW	; Upload PSW
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,A		; Upload ACC
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,B		; Upload B
	POP	B		; Save program IE
	POP	A		; Get HIGH PC
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,A		; Upload PC.1
	POP	A		; Get LOW PC
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,A		; Upload PC.0
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,DPH	; Upload DPTR.1
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,DPL	; Upload DPTR.0
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,SP		; Upload SP
	ANL	PSW,#%11100111	; Insure RB=0
	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,0		; Save R0
	MOV	R0,#1		; Point to location 1
UPLR1	JNB	SCON.1,*	; Wait for TX
	CLR	SCON.1		; Indicate send
	MOV	SBUF,[R0]	; Save memory
	INC	R0		; Advance
	CJNE	R0,#$20,UPLR1	; Do them all
	ACALL	IRET		; Insure INT system reset
	AJMP	MAIN		; Return to MAIN
IRET	RETI			; Reset interrupt system
;
; Indirect SFR read table
;
SFRRT	MOV	A,$80
	RET
	MOV	A,$81
	RET
	MOV	A,$82
	RET
	MOV	A,$83
	RET
	MOV	A,$84
	RET
	MOV	A,$85
	RET
	MOV	A,$86
	RET
	MOV	A,$87
	RET
	MOV	A,$88
	RET
	MOV	A,$89
	RET
	MOV	A,$8A
	RET
	MOV	A,$8B
	RET
	MOV	A,$8C
	RET
	MOV	A,$8D
	RET
	MOV	A,$8E
	RET
	MOV	A,$8F
	RET
	MOV	A,$90
	RET
	MOV	A,$91
	RET
	MOV	A,$92
	RET
	MOV	A,$93
	RET
	MOV	A,$94
	RET
	MOV	A,$95
	RET
	MOV	A,$96
	RET
	MOV	A,$97
	RET
	MOV	A,$98
	RET
	MOV	A,$99
	RET
	MOV	A,$9A
	RET
	MOV	A,$9B
	RET
	MOV	A,$9C
	RET
	MOV	A,$9D
	RET
	MOV	A,$9E
	RET
	MOV	A,$9F
	RET
	MOV	A,$A0
	RET
	MOV	A,$A1
	RET
	MOV	A,$A2
	RET
	MOV	A,$A3
	RET
	MOV	A,$A4
	RET
	MOV	A,$A5
	RET
	MOV	A,$A6
	RET
	MOV	A,$A7
	RET
	MOV	A,$A8
	RET
	MOV	A,$A9
	RET
	MOV	A,$AA
	RET
	MOV	A,$AB
	RET
	MOV	A,$AC
	RET
	MOV	A,$AD
	RET
	MOV	A,$AE
	RET
	MOV	A,$AF
	RET
	MOV	A,$B0
	RET
	MOV	A,$B1
	RET
	MOV	A,$B2
	RET
	MOV	A,$B3
	RET
	MOV	A,$B4
	RET
	MOV	A,$B5
	RET
	MOV	A,$B6
	RET
	MOV	A,$B7
	RET
	MOV	A,$B8
	RET
	MOV	A,$B9
	RET
	MOV	A,$BA
	RET
	MOV	A,$BB
	RET
	MOV	A,$BC
	RET
	MOV	A,$BD
	RET
	MOV	A,$BE
	RET
	MOV	A,$BF
	RET
	MOV	A,$C0
	RET
	MOV	A,$C1
	RET
	MOV	A,$C2
	RET
	MOV	A,$C3
	RET
	MOV	A,$C4
	RET
	MOV	A,$C5
	RET
	MOV	A,$C6
	RET
	MOV	A,$C7
	RET
	MOV	A,$C8
	RET
	MOV	A,$C9
	RET
	MOV	A,$CA
	RET
	MOV	A,$CB
	RET
	MOV	A,$CC
	RET
	MOV	A,$CD
	RET
	MOV	A,$CE
	RET
	MOV	A,$CF
	RET
	MOV	A,$D0
	RET
	MOV	A,$D1
	RET
	MOV	A,$D2
	RET
	MOV	A,$D3
	RET
	MOV	A,$D4
	RET
	MOV	A,$D5
	RET
	MOV	A,$D6
	RET
	MOV	A,$D7
	RET
	MOV	A,$D8
	RET
	MOV	A,$D9
	RET
	MOV	A,$DA
	RET
	MOV	A,$DB
	RET
	MOV	A,$DC
	RET
	MOV	A,$DD
	RET
	MOV	A,$DE
	RET
	MOV	A,$DF
	RET
	MOV	A,$E0
	RET
	MOV	A,$E1
	RET
	MOV	A,$E2
	RET
	MOV	A,$E3
	RET
	MOV	A,$E4
	RET
	MOV	A,$E5
	RET
	MOV	A,$E6
	RET
	MOV	A,$E7
	RET
	MOV	A,$E8
	RET
	MOV	A,$E9
	RET
	MOV	A,$EA
	RET
	MOV	A,$EB
	RET
	MOV	A,$EC
	RET
	MOV	A,$ED
	RET
	MOV	A,$EE
	RET
	MOV	A,$EF
	RET
	MOV	A,$F0
	RET
	MOV	A,$F1
	RET
	MOV	A,$F2
	RET
	MOV	A,$F3
	RET
	MOV	A,$F4
	RET
	MOV	A,$F5
	RET
	MOV	A,$F6
	RET
	MOV	A,$F7
	RET
	MOV	A,$F8
	RET
	MOV	A,$F9
	RET
	MOV	A,$FA
	RET
	MOV	A,$FB
	RET
	MOV	A,$FC
	RET
	MOV	A,$FD
	RET
	MOV	A,$FE
	RET
	MOV	A,$FF
	RET
;
; Indirect SFR write table
;
SFRWT	MOV	$80,R0
	RET
	MOV	$81,R0
	RET
	MOV	$82,R0
	RET
	MOV	$83,R0
	RET
	MOV	$84,R0
	RET
	MOV	$85,R0
	RET
	MOV	$86,R0
	RET
	MOV	$87,R0
	RET
	MOV	$88,R0
	RET
	MOV	$89,R0
	RET
	MOV	$8A,R0
	RET
	MOV	$8B,R0
	RET
	MOV	$8C,R0
	RET
	MOV	$8D,R0
	RET
	MOV	$8E,R0
	RET
	MOV	$8F,R0
	RET
	MOV	$90,R0
	RET
	MOV	$91,R0
	RET
	MOV	$92,R0
	RET
	MOV	$93,R0
	RET
	MOV	$94,R0
	RET
	MOV	$95,R0
	RET
	MOV	$96,R0
	RET
	MOV	$97,R0
	RET
	MOV	$98,R0
	RET
	MOV	$99,R0
	RET
	MOV	$9A,R0
	RET
	MOV	$9B,R0
	RET
	MOV	$9C,R0
	RET
	MOV	$9D,R0
	RET
	MOV	$9E,R0
	RET
	MOV	$9F,R0
	RET
	MOV	$A0,R0
	RET
	MOV	$A1,R0
	RET
	MOV	$A2,R0
	RET
	MOV	$A3,R0
	RET
	MOV	$A4,R0
	RET
	MOV	$A5,R0
	RET
	MOV	$A6,R0
	RET
	MOV	$A7,R0
	RET
	MOV	$A8,R0
	RET
	MOV	$A9,R0
	RET
	MOV	$AA,R0
	RET
	MOV	$AB,R0
	RET
	MOV	$AC,R0
	RET
	MOV	$AD,R0
	RET
	MOV	$AE,R0
	RET
	MOV	$AF,R0
	RET
	MOV	$B0,R0
	RET
	MOV	$B1,R0
	RET
	MOV	$B2,R0
	RET
	MOV	$B3,R0
	RET
	MOV	$B4,R0
	RET
	MOV	$B5,R0
	RET
	MOV	$B6,R0
	RET
	MOV	$B7,R0
	RET
	MOV	$B8,R0
	RET
	MOV	$B9,R0
	RET
	MOV	$BA,R0
	RET
	MOV	$BB,R0
	RET
	MOV	$BC,R0
	RET
	MOV	$BD,R0
	RET
	MOV	$BE,R0
	RET
	MOV	$BF,R0
	RET
	MOV	$C0,R0
	RET
	MOV	$C1,R0
	RET
	MOV	$C2,R0
	RET
	MOV	$C3,R0
	RET
	MOV	$C4,R0
	RET
	MOV	$C5,R0
	RET
	MOV	$C6,R0
	RET
	MOV	$C7,R0
	RET
	MOV	$C8,R0
	RET
	MOV	$C9,R0
	RET
	MOV	$CA,R0
	RET
	MOV	$CB,R0
	RET
	MOV	$CC,R0
	RET
	MOV	$CD,R0
	RET
	MOV	$CE,R0
	RET
	MOV	$CF,R0
	RET
	MOV	$D0,R0
	RET
	MOV	$D1,R0
	RET
	MOV	$D2,R0
	RET
	MOV	$D3,R0
	RET
	MOV	$D4,R0
	RET
	MOV	$D5,R0
	RET
	MOV	$D6,R0
	RET
	MOV	$D7,R0
	RET
	MOV	$D8,R0
	RET
	MOV	$D9,R0
	RET
	MOV	$DA,R0
	RET
	MOV	$DB,R0
	RET
	MOV	$DC,R0
	RET
	MOV	$DD,R0
	RET
	MOV	$DE,R0
	RET
	MOV	$DF,R0
	RET
	MOV	$E0,R0
	RET
	MOV	$E1,R0
	RET
	MOV	$E2,R0
	RET
	MOV	$E3,R0
	RET
	MOV	$E4,R0
	RET
	MOV	$E5,R0
	RET
	MOV	$E6,R0
	RET
	MOV	$E7,R0
	RET
	MOV	$E8,R0
	RET
	MOV	$E9,R0
	RET
	MOV	$EA,R0
	RET
	MOV	$EB,R0
	RET
	MOV	$EC,R0
	RET
	MOV	$ED,R0
	RET
	MOV	$EE,R0
	RET
	MOV	$EF,R0
	RET
	MOV	$F0,R0
	RET
	MOV	$F1,R0
	RET
	MOV	$F2,R0
	RET
	MOV	$F3,R0
	RET
	MOV	$F4,R0
	RET
	MOV	$F5,R0
	RET
	MOV	$F6,R0
	RET
	MOV	$F7,R0
	RET
	MOV	$F8,R0
	RET
	MOV	$F9,R0
	RET
	MOV	$FA,R0
	RET
	MOV	$FB,R0
	RET
	MOV	$FC,R0
	RET
	MOV	$FD,R0
	RET
	MOV	$FE,R0
	RET
	MOV	$FF,R0
	RET
