;
; Dunfield MicroScope - Target Kernel for: 6812
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
; This monitor is setup for use on a 68HC812A4, which is installed in the
; Motorola 68HC12 EVB, and is configured to run out of RAM under the D-Bug12
; monitor supplied on the EVB.
;
; Memory/Addressing setup
REGS	EQU	0		; 68HC12 internal registers
ROM	EQU	$6800		; Program code
RAM	EQU	$0960		; data area (160 bytes)
; Monitor features setup
NUMINT	EQU	31		; Number of interrupt vectors handled
; Command codes from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREAD	EQU	$B0		; Read Program memory
CWRITE	EQU	$B8		; Write program memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$6811		; Kernel ID
; Monitor internal RAM memory area
	ORG	RAM
INTVEC	RMB	NUMINT*2	; Interrupt vector table
STACK	EQU	*+16		; Some stack space
;
	ORG	ROM
; Monitor code starts here:
BEGIN	LDS	#STACK		; Set up stack
;
;** Patch D-Bug12 vector table to gain control of interrupts
;
; This is required only when testing under D-Bug12, the monitor
; provided by Motorola with the 68HC12 EVB. If not using D-Bug12, replace
; this block with code to disable the COP, initialize the chip selects,
; serial port and other on-chip peripherals in a manner appropriate to
; your target system hardware configuration.
;
	LDD	#28		; TRAP, D-Bug12's highest interrupt
	LDX	#FFF8		; Start with our TRAP handler
ptdbug	PSHD			; Save vector number
	PSHX			; Pass address on stack
	JSR	[$FE1A-*-4,PC]	; Execute via D-Bug12 function vector
	PULX			; Restore X
	PULD			; Restore D
	LEAX	4,X		; Handlers are 4 bytes in size
	DECB			; Reduce interrupt number
	CMPB	#7		; Lowest documented interrupt in D-Bug12
	BHS	ptdbug		; Do them all (End of D-Bug12 setup block)
; Patch our SWI vector to perform breakpoint processing
	LDD	#BREAKPT	; Point to breakpoint handler
	STD	INTVEC+6	; Install it
; Loop, reading data till it times out
flush:	BSR	getchr		; Get data
	BRA	flush		; till timeout
;
; Breakpoint has occured
;
BREAKPT	LDAB	#1		; Adjust by 1 for breakpt
break0:	LDAA	#BRKF1		; Response 1
	BSR	wrchr		; Send to host
	LDAA	#BRKF2		; Response 2
	BSR	wrchr		; Send to host
	LDAA	#BRKF3		; Response 3
	BSR	wrchr		; Send to host
	TBA			; Get adjust
	BSR	wrchr		; Send to host
	PULA			; Get CC
	BSR	wrchr		; Send to host
	LDAB	#9		; 9 bytes (CC,A,B,X,Y,PC)
break1:	PULA			; Get byte from stack
	BSR	wrchr		; Send to host
	DBNE	B,break1	; Do them all
	TFR	SP,D		; Get SP
	BSR	wrchr		; Send SHH
	TFR	B,A		; Get SPL
	BSR	wrchr		; Send SPL
	BRA	main		; re-enter monitor
;
; Write character (A) to serial port
;
Wrchr:	BRCLR	REGS+$C4;%10000000;*	; Wait for TX ready
	STAA	REGS+$C7	; Write character
	RTS
;
; Get an address into the X register
;
getaddr	BSR	getchr		; Read low
	TAB			; B = low byte
	BSR	getchr		; Read high
	TFR	D,X		; X = address
	RTS
;
; Get a character from the serial port with timeout
; Re-enter monitor if timeout
;
getchr:	LDY	#0		; Reset timeout
getch1:	BRCLR	REGS+$C4;%00100000;getch2	; No character ready
	LDAA	REGS+$C7	; Read character
	RTS
getch2:	DEY			; Reduce count
	DBNE	Y,getch1	; Timeout
;
; Main monitor command loop
:
main:	LDS	#STACK		; Reset stack in case failed
main1:	BSR	getchr		; Get command character
;
; Query command
;
	CMPA	#CQUERY		; Query?
	BNE	mread		; No, try next
	LDX	#QRESP		; Point to response
qloop:	LDAA	1,X+		; Get byte from table
	BEQ	main1		; End, exit
	BSR	wrchr		; Send to host
	BRA	qloop		; Do them all
qresp:	FCB	RESP1,RESP2,=KID,KID,1,0
;
; Read memory
;
mread:	CMPA	#CREAD		; Read memory?
	BNE	mwrite		; No, try next
	BSR	getaddr		; X = address
	BSR	getchr		; A = size
	TFR	A,B		; B = size
mread1:	LDAA	1,X+		; Read byte from memory
	BSR	wrchr		; Send to host
	DBNE	B,mread1	; Do them all
	BRA	main1		; And exit
;
; Write memory
;
mwrite:	CMPA	#CWRITE		; Write memory?
	BNE	break		; No, try next
	BSR	getaddr		; X = address
	BSR	getchr		; A = size
	TFR	A,B		; B = size
mwrit1:	BSR	getchr		; Get byte from host
	STAA	1,X+		; Write to memory
	DBNE	B,mwrit1	; Do them all
	BRA	main1		; And exit
;
; Set a breakpoint
;
break:	CMPA	#CBREAK		; Set breakpoint?
	BNE	exec		; No, try next
	BSR	getaddr		; Get the address
	LDAA	,X		; Get current data
	BSR	wrchr		; Send to host
	LDAA	#$3F		; 'SWI' = breakpoint
	STAA	,X		; Write to memory
	SUBA	,X		; Test for OK
	BSR	wrchr		; Send to host
	BRA	main1		; And exit
;
; Execute program
;
exec:	CMPA	#CEXEC		; Execute?
	BNE	main1		; No, ignore
	BSR	getaddr		; Get SP
	TXS			; Set SP
	LDAB	#9		; 9 bytes (PC,Y,X,B,A,CC)
exec1:	BSR	getchr		; Get byte from host
	PSHA			; Stack it
	DBNE	B,exec1		; Do them all
	RTI			; Launch user program
;
; Vector handlers
;
FFFC	LDAB	#0		; Clock monitor failire
	BRA	revect
FFFA	LDAB	#1		; COP timeout
	BRA	revect
FFF8	LDAB	#2		; Unimplemented opcode
	BRA	revect
FFF6	LDAB	#3		; SWI
	BRA	revect
FFF4	LDAB	#4		; XIRQ
	BRA	revect
FFF2	LDAB	#5		; IRQ
	BRA	revect
FFF0	LDAB	#6
	BRA	revect
FFEE	LDAB	#7
	BRA	revect
FFEC	LDAB	#8
	BRA	revect
FFEA	LDAB	#9
	BRA	revect
FFE8	LDAB	#10
	BRA	revect
FFE6	LDAB	#11
	BRA	revect
FFE4	LDAB	#12
	BRA	revect
FFE2	LDAB	#13
	BRA	revect
FFE0	LDAB	#14
	BRA	revect
FFDE	LDAB	#15
	BRA	revect
FFDC	LDAB	#16
	BRA	revect
FFDA	LDAB	#17
	BRA	revect
FFD8	LDAB	#18
	BRA	revect
FFD6	LDAB	#19
	BRA	revect
FFD4	LDAB	#20
	BRA	revect
FFD2	LDAB	#21
	BRA	revect
FFD0	LDAB	#22
	BRA	revect
FFCE	LDAB	#23
	BRA	revect
FFCC	LDAB	#24
	BRA	revect
FFCA	LDAB	#25
	BRA	revect
FFC8	LDAB	#26
	BRA	revect
FFC6	LDAB	#27
	BRA	revect
FFC4	LDAB	#28
	BRA	revect
FFC2	LDAB	#29
	BRA	revect
FFC0	LDAB	#30
; Handle interrupt
REVECT	LDX	#INTVEC		; Point to interrupt vector table
	ABX			; x1
	LDX	B,X		; Point to vector (x2)
	BEQ	UNEXINT		; Unexpected
	JMP	,X		; Execute user handler
UNEXINT	CLRB			; No adjust for bad int
	JMP	break0		; Handle it
;
; Interrupt vectors.
;
;** To add additional vectors, you must:
;
; -Change ORG below to new starting address & add new FDB statements
; -Add new handlers around the REVECT routine
; -Increase the equate NUMINT to the new number of vectors
; -Modify Monitor DATA area/STACK location to accomodate larger INTVEC
;
; NOTE when loading under D-Bug12, these vectors will cause errors with
; failed writes to memory (the vectors are in EPROM on the EVB). This is
; not a problem, because the startup code calls D-Bug12 to acquire the
; vectors, and these (physical) vectors are not used in this configuration.
; The vectors are *REQUIRED* when running on the "bare metal".
;
	ORG	$FFC0		Allow room for 31 vectors + reset

	FDB	FFC0
	FDB	FFC2
	FDB	FFC4
	FDB	FFC6
	FDB	FFC8
	FDB	FFCA
	FDB	FFCC
	FDB	FFCE
	FDB	FFD0
	FDB	FFD2
	FDB	FFD4
	FDB	FFD6
	FDB	FFD8
	FDB	FFDA
	FDB	FFDC
	FDB	FFDE
	FDB	FFE0
	FDB	FFE2
	FDB	FFE4
	FDB	FFE6
	FDB	FFE8
	FDB	FFEA
	FDB	FFEC
	FDB	FFEE
	FDB	FFF0
	FDB	FFF2		; IRQ
	FDB	FFF4		; XIRQ
	FDB	FFF6		; SWI
	FDB	FFF8		; Unimplemented Opcode
	FDB	FFFA		; COP timeout
	FDB	FFFC		; Clock monitor
	FDB	BEGIN		; Hard reset
