;
; Dunfield MicroScope - Target Kernel for: 6809
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
; HARDWARE INFORMATION
ROM	EQU	$E000		; Code goes here
;RAM	EQU	$DF60		; Data goes here
;UART	EQU	$00E0		; Serial Port address (6551)
RAM	EQU	$9FC0
UART	EQU	$C000
; Command codes from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREAD	EQU	$B0		; Read Program memory
CWRITE	EQU	$B8		; Write program memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$6809		; Kernel ID
;
	ORG	RAM
VSWI3	RMB	2		; SWI3 redirected vector
VSWI2	RMB	2		; SWI2 redirected vector
VFIRQ	RMB	2		; FIRQ redirected vector
VIRQ	RMB	2		; IRQ  redirected vector
VNMI	RMB	2		; NMI  redirected vector
STACK	EQU	RAM+$1F		; Monitor stack pointer
	ORG	ROM
;
RESET:	LDS	#STACK		; Setup stack
	LDD	#$0B1F		; Enable RX, TX, 19200, 8N1
	CLR	UART+1		; Reset UART
	STD	UART+2		; Configure UART
; Loop, reading serial port until timeout
flush:	BSR	getchr		; Get data
	BRA	flush		; Till timeout
;
; Breakpoint handler
;
SWIHND:	LDB	#1		; Adjust by -1 for breakpoint
swih0:	LDA	#BRKF1		; Breakpoint flag 1
	BSR	wrchr		; Write it
	LDA	#BRKF2		; Breakpoint flag 2
	BSR	wrchr		; Write it
	LDA	#BRKF3		; Breakpoint flag 3
	BSR	wrchr		; Write it
	TFR	B,A		; Get adjust value
	BSR	wrchr		; Write it
	LDB	#12		; Upload 12 bytes
swih1:	PULS	A		; Get byte from stack
	BSR	wrchr		; Upload it
	DECB			; Reduce count
	BNE	swih1		; Do them all
	TFR	S,D		; D = SP
	BSR	wrchr		; Send SPH
	TFR	B,A		; Get SPL
	BSR	wrchr		; Send SPL
	BRA	main		; re-enter monitor
;
; Write character to the serial port
;
WRCHR:	PSHS	A		; Save ACC
wrchr1	LDA	UART+1		; Get UART status
	BITA	#%00010000	; TX ready?
	BEQ	wrchr1		; No, try again
	PULS	A		; Restore ACC
	STA	UART		; Write to UART
	RTS
;
; Get an address into X register
;
getaddr	BSR	getchr		; Get character
	TFR	A,B		; B = LOW byte
	BSR	getchr		; Get character
	TFR	D,X		; D:X = data
	RTS
;
; Get a character from the serial port with timeout
;
getchr:	LDY	#0		; Zero timeout
getch1	LDA	UART+1		; Get status
	BITA	#%00001000	; RX ready?
	BEQ	getch2		; No, keep looking
	LDA	UART		; Read the data
	RTS			; And exit
getch2	LEAY	-1,Y		; Reduce timeout
	BNE	getch1		; Not expired
;
; Main monitor command loop
;
main:	LDS	#STACK		; Reset stack
main1:	BSR	getchr		; Get command character
;
; Query command
;
	CMPA	#CQUERY		; Query command?
	BNE	mread		; No, try next
	LDX	#QRESP		; Point to query response
qloop	LDA	,X+		; Read data
	BEQ	main1		; Exit at end
	BSR	wrchr		; Send to host
	BRA	qloop		; Do them all
qresp:	FCB	RESP1,RESP2,=KID,KID,1,0
;
; Read memory
;
mread:	CMPA	#CREAD		; Read memory command?
	BNE	mwrite		; No, try next
	BSR	getaddr		; X = memory address
	BSR	getchr		; A = size
	TFR	A,B		; B = size
mread1:	LDA	,X+		; Read data byte
	BSR	WRCHR		; Write it
	DECB			; Reduce count
	BNE	mread1		; Do them all
	BRA	main		; Exit
;
; Write memory
;
mwrite:	CMPA	#CWRITE		; Write memory command?
	BNE	break		; No, try next
	BSR	getaddr		; X = memory address
	BSR	getchr		; A = size
	TFR	A,B		; B = size
mwrit1:	BSR	getchr		; Read next data
	STA	,X+		; Write it
	DECB			; Reduce count
	BNE	mwrit1		; Do them all
	BRA	main		; Exit
;
; Set breakpoint
;
break:	CMPA	#CBREAK		; Breakpoint Command?
	BNE	exec		; No, try next
	BSR	getaddr		; Get the address
	LDA	,X		; Get current content
	BSR	wrchr		; Send to host
	LDA	#$3F		; 'SWI' = breakpoint
	STA	,X		; Write to memory
	SUBA	,X		; Test for valid
	BSR	wrchr		; Write it
	BRA	main1		; Exit
;
; Begin program execution
;
exec:	CMPA	#CEXEC		; Execute command?
	BNE	main1		; No, try next
	BSR	getaddr		; Get stack
	TFR	D,SP		; Set SP
	LDB	#11		; Transfer 12 bytes
exec1:	BSR	getchr		; Get a byte
	PSHS	A		; Stack it
	DECB			; Reduce count
	BNE	exec1		; Do them all
	BSR	getchr		; Get CC
	ORA	#%10000000	; Set "Entire" flag
	PSHS	A		; Stack it
	RTI			; Execute user program
;
; Vector redirection
;
SWI3:	LDX	VSWI3		; Get vector
	BEQ	unexi		; Not expected
	JMP	,X		; Execute it
SWI2:	LDX	VSWI2		; Get vector
	BEQ	unexi		; Not expected
	JMP	,X		; Execute
FIRQ:	LDX	VFIRQ		; Get vector
	BEQ	unexi		; Not expected
	JMP	,X		; Execute
IRQ:	LDX	VIRQ		; Get vector
	BEQ	unexi		; Not expected
	JMP	,X		; Execute
NMIHND:	LDX	VNMI		; Get vector
	BEQ	unexi		; Not expected
	JMP	,X		; Execute
;
; Unexpected interrupt handler
;
UNEXI:	CLRB			; No adjust on UNEX interrupt
	JMP	swih0		; And proceed
;
	ORG	$FFF2
	FDB	SWI3
	FDB	SWI2
	FDB	FIRQ
	FDB	IRQ
	FDB	SWIHND
	FDB	NMIHND
	FDB	RESET
