;
; Dunfield MicroScope - Target Kernel for: Cypher Mini-Eval
;
; To rebuild:
;   asm320 cypher -t
;   dmslfb cypher
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
; System parameters
BAUD	EQU	1		; Baudrate
ROM	EQU	$0000		; Location target monitor
USERAM	EQU	$0800		; Location of user program
STACK	EQU	8		; Debugger stack position
; Command code from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREADP	EQU	$B0		; Read Program memory
CREADD	EQU	$B1		; Read Data memory
CREADI	EQU	$B2		; Read Internal memory
CREADS	EQU	$B3		; Read SFR memory	
CWRITP	EQU	$B8		; Write program memory
CWRITD	EQU	$B9		; Write data memory
CWRITI	EQU	$BA		; Write internal memory
CWRITS	EQU	$BB		; Write SFR memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$CE01		; Kernel ID
; Cypher internal peripheral registers
MPAGE	EQU	$92		; Memory page register
;DPH1	EQU	$85		; Alternate DPH
;DPL1	EQU	$84		; Alternate DPL
;DPS	EQU	$86		; DP select
; Fixed memory addresses
RXBUF	EQU	$3FFE		; Receive  buffer (1 byte)
TXBUF	EQU	$3FFF		; Transmit buffer (1 byte)
USP	EQU	$3FFD		; User stack pointer
;
	ORG	ROM		; Target monitor begins
	AJMP	BEGIN		Begin program
; Re-vector interrupts
	ORG	ROM+$0003	EXT Interrupt 0
	LJMP	USERAM+$03
	AJMP	_WRCHR		; Entry point to write character
	AJMP	_TSTCHR		; Entry point to test for character
	ORG	ROM+$000B	Timer 0 overflow
	LJMP	USERAM+$0B
	AJMP	_GETCH		; Get character
	AJMP	xled		; Debug LED's
	ORG	ROM+$0013	EXT Interrupt 1
	LJMP	USERAM+$13
	ORG	ROM+$001B	Timer1 - Single step
	LJMP	sshand
	ORG	ROM+$0023	RX+TX interrupt from SERIAL0
	LJMP	USERAM+$23
	ORG	ROM+$002B	TF2+EXF2
	LJMP	USERAM+$2B
	ORG	ROM+$0033	Power Fail Interrupt
	LJMP	USERAM+$33
	ORG	ROM+$003B	RX+TX interrupt from SERIAL1
	LJMP	USERAM+$3B
	ORG	ROM+$0043	External interrupt 2
	LJMP	USERAM+$43
	ORG	ROM+$004B	External interrupt 3
	LJMP	USERAM+$4B
	ORG	ROM+$0053	External interrupt 4
	LJMP	USERAM+$53
	ORG	ROM+$005B	External interrupt 5
	LJMP	USERAM+$5B
	ORG	ROM+$0063	Watchdog timeout
	LJMP	USERAM+$63
	ORG	$0066
ENTER:	MOVX	[R1],A		; Entry to RAM sub
RAMTFR:	NOP			; Required for Sync
	AJMP	BEGIN		; Launch debugger
	MOV	A,R2		; Get return value
	MOVX	[R1],A		; Return to ROM
	RET
;
; Initialize debugger
;
Begin	MOV	TMOD,#%00100000	; T1=8-bit auto-reload
	MOV	TH1,#-BAUD	; Timer-1 reload value
	MOV	TH2,#-BAUD	; Timer-2 initial value
	MOV	TCON,#%01001001	; Run1, hold 0
	ACALL	CPWR		; Install write tunction
	MOV	DPTR,#USP	; Point to USER SP copy
	MOV	A,#7		; Default SP
	MOVX	[DPTR],A	; Set default SP
; Loop reading data till it times out
bloop	MOV	R5,#100		; Set short timeout
	ACALL	getcht		; Wait for character
	SJMP	bloop		; Till none
;
; Get memory access operands (ADDR-SIZE)
;
getmemx	ACALL	getaddr		; Get high address
	ACALL	getchr		; Get size
	MOV	R7,A		; Set it
	RET
;
; Get an address
;
getaddr	ACALL	getchr		; Get high address
	MOV	DPH,A		; Set it
	ACALL	getchr		; Get low address
	MOV	DPL,A		; Set it
	MOV	R1,A		; Also for internal
	RET
;
; Get character from serial port with timeout
; Re-enter monitor if timeout
;
Getchr	MOV	R5,#0		; Timeout count
getcht	MOV	R6,#0		; 512 cycles
getcr1	ACALL	TSTCHR		; Test for character
	JNC	getcr2		; No character
	RET
getcr2	DJNZ	R6,getcr1	; Wait for it
	DJNZ	R5,getcr1	; And wait for it
;
; Main command execution loop
;
main	MOV	SP,#STACK	; Reset stack
main1	ACALL	Getchr		; Check for character
;
; Query command
;
	CJNE	A,#CQUERY,readp	; Try break command
	MOV	DPTR,#qresp	; Point to query response
qloop	CLR	A		; Zero offset
	MOVC	A,[A+DPTR]	; Get data byte
	JZ	main1		; End of list
	ACALL	wrchr		; Write it
	INC	DPTR		; Skip to next
	SJMP	qloop		; And proceed
qresp	DB	RESP1,RESP2,=KID,KID,1,0
;
; Read program memory
;
readp	CJNE	A,#CREADP,readd	; No, try next
	ACALL	getmemx		; Get memory operands
rdp1	CLR	A		; Zero A
	MOVC	A,[A+DPTR]	; Get byte of data
	ACALL	wrchr		; Send to host
	INC	DPTR		; Skip to next
	DJNZ	R7,rdp1		; Do them all
	AJMP	main1
;
; Read data memory
;
readd	CJNE	A,#CREADD,readi	; No, try next
	ACALL	getmemx		; Get memory operands
rdd1	MOVX	A,[DPTR]	; Get byte of data
	ACALL	wrchr		; Send to host
	INC	DPTR		; Skip to next
	DJNZ	R7,rdd1		; Do them all
	AJMP	main1
;
; Read internal memory
;
readi	CJNE	A,#CREADI,reads	; No, try next
	ACALL	getmemx		; Get memory operands
rdi1	MOV	A,[R1]		; Read byte of data
	ACALL	wrchr		; Send to host
	INC	R1		; Skip to next
	DJNZ	R7,rdi1		; Do them all
	AJMP	main1
;
; Read SFR memory
;
reads	CJNE	A,#CREADS,writp	; No, try next
	ACALL	getmemx		; Get memory operands
rds1	MOV	DPTR,#SFRRT	; Point to SFR read table
	ACALL	sfrx		; Get the value
	ACALL	wrchr		; Send to host
	INC	R1		; Skip to next
	DJNZ	R7,rds1		; Do them all
	AJMP	main1
;
; Write program memory
;
writp	CJNE	A,#CWRITP,writd	; No, try next
	PUSH	MPAGE		; Save MPAGE
	ACALL	getmemx		; Get memory operands
	ACALL	PREPWP		; Prepare for write program
wrp1:	ACALL	getchr		; Get character
	ACALL	WXP		; Write it
	DJNZ	R7,wrp1		; Do them all
	POP	MPAGE		; Restore MPAGE
	AJMP	main1		; And proceed
;
; Write data memory
;
writd	CJNE	A,#CWRITD,writi	; No, try next
wrd0	ACALL	getmemx		; Get memory operands
wrd1	ACALL	getchr		; Get a data byte
	MOVX	[DPTR],A	; Write it
	INC	DPTR		; Skip to next
	DJNZ	R7,wrd1		; Do them all
	AJMP	main1
;
; Write internal memory
;
writi	CJNE	A,#CWRITI,writs	; No, try next
	ACALL	getmemx		; Get memory operands
wri1	ACALL	getchr		; Get data byte
	MOV	[R1],A		; Write it
	INC	R1		; Skip to next
	DJNZ	R7,wri1		; Do them all
	AJMP	main1
;
; Write SFR memory
;
writs	CJNE	A,#CWRITS,step	; No, try next
	ACALL	getmemx		; Get memory operands
wrs1	ACALL	getchr		; Get data byte
	MOV	R0,A		; Save for write
	MOV	DPTR,#SFRWT	; Point to write table
	ACALL	SFRX		; Access SFR
	INC	R1		; Advance
	DJNZ	R7,wrs1		; Do them all
	AJMP	main1
;
; Step one instruction
; *KM <SP <PC <A <B <DPTR <PSW
;
step	CJNE	A,#CSTEP,exec	; No, try next
	MOV	DPTR,#USP	; Point to user pointer
	MOVX	A,[DPTR]	; Read value
	MOV	SP,A		; Set stack
	ACALL	GETCHR		; Download R0 value
	MOV	DPH,A		; Save for later
	MOV	R0,#1		; Address
step1	ACALL	GETCH		; Read character
	MOV	[R0],A		; Write data
	INC	R0		; Advance
	CJNE	R0,#$20,step1	; Do them all
	MOV	R0,DPH		; Write R0
	ACALL	GETCH		; Read SP
	MOV	SP,A		; Read stack pointer
	ACALL	GETCH		; Read PC.0
	PUSH	A		; Save for later
	ACALL	GETCH		; Read PC.1
	PUSH	A		; Save for later
	ACALL	GETCH		; Read ACC
	ORL	B,#%10001000	; Enable timer-1 interrupt
	CLR	IP.3		; Insure LOW priority
	PUSH	B		; Save program IE
	PUSH	A		; Save A
	ACALL	GETCH		; Read B
	MOV	B,A		; Download B
	ACALL	GETCH		; Read DPTR.0
	MOV	DPL,A		; Save DPL
	ACALL	GETCH		; Read DPTR.1
	MOV	DPH,A		; Save DPH
	ACALL	GETCH		; Read DPTR1.0
	MOV	DPL1,A		; Save DPL1
	ACALL	GETCH		; Read DPTR1.1
	MOV	DPH1,A		; Save DPH1
	ACALL	GETCH		; Read DPS
	MOV	DPS,A		; Save
	ACALL	_GETCH		; Read PSW
	MOV	PSW,A		; Download PSW
* Set up Timer-1 for single step interrupt
	CLR	TCON.6		; Stop timer-1
	ANL	TMOD,#%00001111	; Zero timer-1 mode
	ORL	TMOD,#%00010000	; Timer-1 16 bit
	MOV	TH1,#-1		; Number of cycles
	MOV	TL1,#-4		; Till first user instruction
	CLR	TCON.7		; Clear timer-1 int pend.
	SETB	TCON.6		; Enable timer-1
	POP	A		; Restore A
	NOP			; Line up for interrupt
* Enable timer for single-step interrupt
	POP	IE		; Set program IE
	RET			; Execute user program
;
; Launch program
;
exec	CJNE	A,#CEXEC,break	; No, try next
	MOV	DPTR,#USP	; Point to user pointer
	MOVX	A,[DPTR]	; Read value
	MOV	SP,A		; Set stack
	ACALL	GETCHR		; Download R0 value
	MOV	DPH,A		; Save for later
	MOV	R0,#1		; Address
exec1	ACALL	GETCH		; Read character
	MOV	[R0],A		; Write data
	INC	R0		; Advance
	CJNE	R0,#$20,exec1	; Do them all
	MOV	R0,DPH		; Write R0
	ACALL	GETCH		; Read SP
	MOV	SP,A		; Read stack pointer
	ACALL	GETCH		; Read PC.0
	PUSH	A		; Save for later
	ACALL	GETCH		; Read PC.1
	PUSH	A		; Save for later
	ACALL	GETCH		; Read ACC
	PUSH	B		; Save program IE
	PUSH	A		; Save A
	ACALL	GETCH		; Read B
	MOV	B,A		; Download B
	ACALL	GETCH		; Read DPTR.0
	MOV	DPL,A		; Save DPL
	ACALL	GETCH		; Read DPTR.1
	MOV	DPH,A		; Save DPH
	ACALL	GETCH		; Read DPTR1.0
	MOV	DPL1,A		; Save DPL1
	ACALL	GETCH		; Read DPTR1.1
	MOV	DPH1,A		; Save DPH1
	ACALL	GETCH		; Read DPS
	MOV	DPS,A		; Save
	ACALL	_GETCH		; Read PSW
	MOV	PSW,A		; Download PSW
	POP	A		; Restore ACC
	POP	IE		; Restore program IE
	RET
;
; Set breakpoint on target system
;
break	CJNE	A,#CBREAK,xxx	; No, try next
	ACALL	getaddr		; Get an address
; Send date to host
	CLR	A		; Zero offset
	MOVC	A,[A+DPTR]	; Read byte
	ACALL	WRCHR		; Write it
	MOV	A,#1		; One offset
	MOVC	A,[A+DPTR]	; Read byte
	ACALL	WRCHR		; Write it
	MOV	A,#2		; Two offset
	MOVC	A,[A+DPTR]	; Read byte
	ACALL	WRCHR		; Write it
	ACALL	PREPWP		; Prepare to write program
	MOV	A,#$12		; LCALL instruction
	ACALL	WXP		; Write to code memory
	MOV	A,#=bphand	; Handler LOW address
	ACALL	WXP		; Write to code memory
	MOV	A,#bphand	; Handler HIGH address
	ACALL	WXP		; Write to code memory
	MOV	A,#0		; Success flag
	ACALL	WRCHR		; Send to host
	AJMP	main1		; And continue
;
xxx	AJMP	main
;
; Execute SFR memory access function from DPTR/R1
SFRX	MOV	A,R1		; Get SFR address
	CLR	C		; Zero carry in
	RLC	A		; x2, drop high bit
	ADD	A,DPL		; Include in DP
	MOV	DPL,A		; Replace
	JNC	sfrx1		; No carry out
	INC	DPH		; Advance high
sfrx1	MOV	A,R1		; Get address again
	ANL	A,#%01111111	; Drop high bit
	JMP	[A+DPTR]	; Execute function
;
; Breakpoint handler
;
bphand	CLR	IE.3		; Disable Timer-1 interrupt
	PUSH	IE		; Save Interrupt enable
	MOV	IE,#0		; Clear interrupts
	PUSH	A		; Save ACC
* Restore timer-1 for serial I/O
	ANL	TMOD,#%00001111	; Zero timer-1 mode
	ORL	TMOD,#%00100000	; T1 = 8 bit auto-reload
	MOV	TH1,#-BAUD	; Timer-1 reload value
	MOV	TL1,#-BAUD	; Timer-1 initial value
* Signal a Breakpoint/Single-step return to the host
	MOV	A,#BRKF1	; Send breakpoint flag #1
	ACALL	_WRCHR		; Send it
	MOV	A,#BRKF2	; Send breakpoint flag #2
	ACALL	_WRCHR		; Send it
	MOV	A,#BRKF3	; Send breakpoint flag #3
	ACALL	_WRCHR		; Send it
	MOV	A,#3		; Adjust by 3
	SJMP	ctgo		; Continuewith contect switch
;
; Single-Step interrupt handler
; #break $90 $A5 $B9 <PSW <A <B >PC >DPTR <SP *KM
;
sshand	CLR	IE.3		; Disable Timer-1 interrupt
	PUSH	IE		; Save Interrupt enable
	MOV	IE,#0		; Clear interrupts
	PUSH	A		; Save ACC
* Restore timer-1 for serial I/O
	ANL	TMOD,#%00001111	; Zero timer-1 mode
	ORL	TMOD,#%00100000	; T1 = 8 bit auto-reload
	MOV	TH1,#-BAUD	; Timer-1 reload value
	MOV	TL1,#-BAUD	; Timer-1 initial value
* Signal a Breakpoint/Single-step return to the host
	MOV	A,#BRKF1	; Get flag #1
	ACALL	_WRCHR		; Send it
	MOV	A,#BRKF2	; Send breakpoint flag #2
	ACALL	_WRCHR		; Send it
	MOV	A,#BRKF3	; Send breakpoint flag #3
	ACALL	_WRCHR		; Send  it
	MOV	A,#0		; No adjust
* Upload registers to host
ctgo	ACALL	_WRCHR		; Send it
	MOV	A,PSW		; Upload PSW
	ACALL	_WRCHR		; Send it
	POP	A		; Restore ACC
	ACALL	_WRCHR		; Send it
	MOV	A,B		; Upload B
	ACALL	_WRCHR		; Send it
	POP	B		; Save program IE
	POP	A		; Get HIGH PC
	ACALL	_WRCHR		; Send it
	POP	A		; Get LOW PC
	ACALL	_WRCHR		; Send it
	MOV	A,DPH		; Get DPH
	ACALL	_WRCHR		; Send it
	MOV	A,DPL		; Get DPL
	ACALL	_WRCHR		; Send it
	MOV	A,DPH1		; Get DPH1
	ACALL	_WRCHR		; Send it
	MOV	A,DPL1		; Get DPL1
	ACALL	_WRCHR		; Send it
	MOV	A,DPS		; Get DPS
	ACALL	_WRCHR		; send it
	MOV	A,SP		; Get SP
	MOV	DPTR,#USP	; Point to user
	MOVX	[DPTR],A	; Save copy
	ACALL	_WRCHR		; Send it
	ANL	PSW,#%11100111	; Insure RB=0
	MOV	A,R0		; Get R0
	ACALL	_WRCHR		; Send it
	MOV	R0,#1		; Point to location 1
UPLR1	MOV	A,[R0]		; Read data byte
	ACALL	_WRCHR		; Send it
	INC	R0		; Advance
	CJNE	R0,#$20,UPLR1	; Do them all
	ACALL	IRET		; Insure INT system reset
	ACALL	CPWR		; Install write function
	AJMP	MAIN		; Return to MAIN
IRET	RETI			; Reset interrupt system
;
; Indirect SFR read table
;
SFRRT	MOV	A,$80
	RET
	MOV	A,$81
	RET
	MOV	A,$82
	RET
	MOV	A,$83
	RET
	MOV	A,$84
	RET
	MOV	A,$85
	RET
	MOV	A,$86
	RET
	MOV	A,$87
	RET
	MOV	A,$88
	RET
	MOV	A,$89
	RET
	MOV	A,$8A
	RET
	MOV	A,$8B
	RET
	MOV	A,$8C
	RET
	MOV	A,$8D
	RET
	MOV	A,$8E
	RET
	MOV	A,$8F
	RET
	MOV	A,$90
	RET
	MOV	A,$91
	RET
	MOV	A,$92
	RET
	MOV	A,$93
	RET
	MOV	A,$94
	RET
	MOV	A,$95
	RET
	MOV	A,$96
	RET
	MOV	A,$97
	RET
	MOV	A,$98
	RET
	MOV	A,$99
	RET
	MOV	A,$9A
	RET
	MOV	A,$9B
	RET
	MOV	A,$9C
	RET
	MOV	A,$9D
	RET
	MOV	A,$9E
	RET
	MOV	A,$9F
	RET
	MOV	A,$A0
	RET
	MOV	A,$A1
	RET
	MOV	A,$A2
	RET
	MOV	A,$A3
	RET
	MOV	A,$A4
	RET
	MOV	A,$A5
	RET
	MOV	A,$A6
	RET
	MOV	A,$A7
	RET
	MOV	A,$A8
	RET
	MOV	A,$A9
	RET
	MOV	A,$AA
	RET
	MOV	A,$AB
	RET
	MOV	A,$AC
	RET
	MOV	A,$AD
	RET
	MOV	A,$AE
	RET
	MOV	A,$AF
	RET
	MOV	A,$B0
	RET
	MOV	A,$B1
	RET
	MOV	A,$B2
	RET
	MOV	A,$B3
	RET
	MOV	A,$B4
	RET
	MOV	A,$B5
	RET
	MOV	A,$B6
	RET
	MOV	A,$B7
	RET
	MOV	A,$B8
	RET
	MOV	A,$B9
	RET
	MOV	A,$BA
	RET
	MOV	A,$BB
	RET
	MOV	A,$BC
	RET
	MOV	A,$BD
	RET
	MOV	A,$BE
	RET
	MOV	A,$BF
	RET
	MOV	A,$C0
	RET
	MOV	A,$C1
	RET
	MOV	A,$C2
	RET
	MOV	A,$C3
	RET
	MOV	A,$C4
	RET
	MOV	A,$C5
	RET
	MOV	A,$C6
	RET
	MOV	A,$C7
	RET
	MOV	A,$C8
	RET
	MOV	A,$C9
	RET
	MOV	A,$CA
	RET
	MOV	A,$CB
	RET
	MOV	A,$CC
	RET
	MOV	A,$CD
	RET
	MOV	A,$CE
	RET
	MOV	A,$CF
	RET
	MOV	A,$D0
	RET
	MOV	A,$D1
	RET
	MOV	A,$D2
	RET
	MOV	A,$D3
	RET
	MOV	A,$D4
	RET
	MOV	A,$D5
	RET
	MOV	A,$D6
	RET
	MOV	A,$D7
	RET
	MOV	A,$D8
	RET
	MOV	A,$D9
	RET
	MOV	A,$DA
	RET
	MOV	A,$DB
	RET
	MOV	A,$DC
	RET
	MOV	A,$DD
	RET
	MOV	A,$DE
	RET
	MOV	A,$DF
	RET
	MOV	A,$E0
	RET
	MOV	A,$E1
	RET
	MOV	A,$E2
	RET
	MOV	A,$E3
	RET
	MOV	A,$E4
	RET
	MOV	A,$E5
	RET
	MOV	A,$E6
	RET
	MOV	A,$E7
	RET
	MOV	A,$E8
	RET
	MOV	A,$E9
	RET
	MOV	A,$EA
	RET
	MOV	A,$EB
	RET
	MOV	A,$EC
	RET
	MOV	A,$ED
	RET
	MOV	A,$EE
	RET
	MOV	A,$EF
	RET
	MOV	A,$F0
	RET
	MOV	A,$F1
	RET
	MOV	A,$F2
	RET
	MOV	A,$F3
	RET
	MOV	A,$F4
	RET
	MOV	A,$F5
	RET
	MOV	A,$F6
	RET
	MOV	A,$F7
	RET
	MOV	A,$F8
	RET
	MOV	A,$F9
	RET
	MOV	A,$FA
	RET
	MOV	A,$FB
	RET
	MOV	A,$FC
	RET
	MOV	A,$FD
	RET
	MOV	A,$FE
	RET
	MOV	A,$FF
	RET
;
; Indirect SFR write table
;
SFRWT	MOV	$80,R0
	RET
	MOV	$81,R0
	RET
	MOV	$82,R0
	RET
	MOV	$83,R0
	RET
	MOV	$84,R0
	RET
	MOV	$85,R0
	RET
	MOV	$86,R0
	RET
	MOV	$87,R0
	RET
	MOV	$88,R0
	RET
	MOV	$89,R0
	RET
	MOV	$8A,R0
	RET
	MOV	$8B,R0
	RET
	MOV	$8C,R0
	RET
	MOV	$8D,R0
	RET
	MOV	$8E,R0
	RET
	MOV	$8F,R0
	RET
	MOV	$90,R0
	RET
	MOV	$91,R0
	RET
	MOV	$92,R0
	RET
	MOV	$93,R0
	RET
	MOV	$94,R0
	RET
	MOV	$95,R0
	RET
	MOV	$96,R0
	RET
	MOV	$97,R0
	RET
	MOV	$98,R0
	RET
	MOV	$99,R0
	RET
	MOV	$9A,R0
	RET
	MOV	$9B,R0
	RET
	MOV	$9C,R0
	RET
	MOV	$9D,R0
	RET
	MOV	$9E,R0
	RET
	MOV	$9F,R0
	RET
	MOV	$A0,R0
	RET
	MOV	$A1,R0
	RET
	MOV	$A2,R0
	RET
	MOV	$A3,R0
	RET
	MOV	$A4,R0
	RET
	MOV	$A5,R0
	RET
	MOV	$A6,R0
	RET
	MOV	$A7,R0
	RET
	MOV	$A8,R0
	RET
	MOV	$A9,R0
	RET
	MOV	$AA,R0
	RET
	MOV	$AB,R0
	RET
	MOV	$AC,R0
	RET
	MOV	$AD,R0
	RET
	MOV	$AE,R0
	RET
	MOV	$AF,R0
	RET
	MOV	$B0,R0
	RET
	MOV	$B1,R0
	RET
	MOV	$B2,R0
	RET
	MOV	$B3,R0
	RET
	MOV	$B4,R0
	RET
	MOV	$B5,R0
	RET
	MOV	$B6,R0
	RET
	MOV	$B7,R0
	RET
	MOV	$B8,R0
	RET
	MOV	$B9,R0
	RET
	MOV	$BA,R0
	RET
	MOV	$BB,R0
	RET
	MOV	$BC,R0
	RET
	MOV	$BD,R0
	RET
	MOV	$BE,R0
	RET
	MOV	$BF,R0
	RET
	MOV	$C0,R0
	RET
	MOV	$C1,R0
	RET
	MOV	$C2,R0
	RET
	MOV	$C3,R0
	RET
	MOV	$C4,R0
	RET
	MOV	$C5,R0
	RET
	MOV	$C6,R0
	RET
	MOV	$C7,R0
	RET
	MOV	$C8,R0
	RET
	MOV	$C9,R0
	RET
	MOV	$CA,R0
	RET
	MOV	$CB,R0
	RET
	MOV	$CC,R0
	RET
	MOV	$CD,R0
	RET
	MOV	$CE,R0
	RET
	MOV	$CF,R0
	RET
	MOV	$D0,R0
	RET
	MOV	$D1,R0
	RET
	MOV	$D2,R0
	RET
	MOV	$D3,R0
	RET
	MOV	$D4,R0
	RET
	MOV	$D5,R0
	RET
	MOV	$D6,R0
	RET
	MOV	$D7,R0
	RET
	MOV	$D8,R0
	RET
	MOV	$D9,R0
	RET
	MOV	$DA,R0
	RET
	MOV	$DB,R0
	RET
	MOV	$DC,R0
	RET
	MOV	$DD,R0
	RET
	MOV	$DE,R0
	RET
	MOV	$DF,R0
	RET
	MOV	$E0,R0
	RET
	MOV	$E1,R0
	RET
	MOV	$E2,R0
	RET
	MOV	$E3,R0
	RET
	MOV	$E4,R0
	RET
	MOV	$E5,R0
	RET
	MOV	$E6,R0
	RET
	MOV	$E7,R0
	RET
	MOV	$E8,R0
	RET
	MOV	$E9,R0
	RET
	MOV	$EA,R0
	RET
	MOV	$EB,R0
	RET
	MOV	$EC,R0
	RET
	MOV	$ED,R0
	RET
	MOV	$EE,R0
	RET
	MOV	$EF,R0
	RET
	MOV	$F0,R0
	RET
	MOV	$F1,R0
	RET
	MOV	$F2,R0
	RET
	MOV	$F3,R0
	RET
	MOV	$F4,R0
	RET
	MOV	$F5,R0
	RET
	MOV	$F6,R0
	RET
	MOV	$F7,R0
	RET
	MOV	$F8,R0
	RET
	MOV	$F9,R0
	RET
	MOV	$FA,R0
	RET
	MOV	$FB,R0
	RET
	MOV	$FC,R0
	RET
	MOV	$FD,R0
	RET
	MOV	$FE,R0
	RET
	MOV	$FF,R0
	RET
;
; Write character[A] to serial port preserving active DPTR
;
_WRCHR:	PUSH	A		; Save ACC
	MOV	A,DPS		; Get current DPS
	JB	A.0,WRCH1	; DPTR-1 selected
	POP	A		; Restore A
; Write character[A] to serial port preserving DPTR-0
WRCHR:	PUSH	DPH		; Save DPH
	PUSH	DPL		; Save DPL
	MOV	DPTR,#TXBUF	; Point to TX buffer
	MOVX	[DPTR],A	; Write it
	MOV	DPTR,#$B0EB	; Serial 1 transmit length
	CLR	A		; Zero high
	MOVX	[DPTR],A	; Write high
	INC	DPTR		; Skip to low
	INC	A		; Packet length = 1
	MOVX	[DPTR],A	; Write low
; Start transfer
	MOV	DPTR,#$B0E8	; Serial DMA control
	MOV	A,#$01		; Start transfer
	MOVX	[DPTR],A	; Write it
; Wait for transfer to finish
	MOV	DPTR,#$B015	; Interrupt source register
wrch0a:	MOVX	A,[DPTR]	; Read the data
	ANL	A,#%00100000	; Wait for port-1 complete
	JZ	wrch0a		; Wait for it
	MOVX	[DPTR],A	; Reset interrupt
	MOV	DPTR,#TXBUF	; Point to data
	MOVX	A,[DPTR]	; Restore character
	POP	DPL
	POP	DPH
	RET
; Write character[A] to serial port preserving DPTR-1
WRCH1:	POP	A		; Restore A
	PUSH	DPH1		; Save DPH
	PUSH	DPL1		; Save DPL
	MOV	DPTR,#TXBUF	; Point to TX buffer
	MOVX	[DPTR],A	; Write it
	MOV	DPTR,#$B0EB	; Serial 1 transmit length
	CLR	A		; Zero high
	MOVX	[DPTR],A	; Write high
	INC	DPTR		; Skip to low
	INC	A		; Packet length = 1
	MOVX	[DPTR],A	; Write low
; Start transfer
	MOV	DPTR,#$B0E8	; Serial DMA control
	MOV	A,#$01		; Start transfer
	MOVX	[DPTR],A	; Write it
; Wait for transfer to finish
	MOV	DPTR,#$B015	; Interrupt source register
wrch1a:	MOVX	A,[DPTR]	; Read the data
	ANL	A,#%00100000	; Wait for port-1 complete
	JZ	wrch1a		; Wait for it
	MOVX	[DPTR],A	; Reset interrupt
	MOV	DPTR,#TXBUF	; Point to data
	MOVX	A,[DPTR]	; Restore character
	POP	DPL1
	POP	DPH1
	RET
;
; Test for character from serial port, on exit:
;   C=0, A=00 - No data available
;   C=1, A=xx - Data xx received
;
_TSTCHR: MOV	A,DPS		; Get DPS
	JB	A.0,tstc1	; Use DPTR-1
; Test for character from serial port using DPTR-0
TSTCHR:	PUSH	DPH		; Save DPH
	PUSH	DPL		; Save DPL
	MOV	DPTR,#$B013	; Serial interrupt source
	MOVX	A,[DPTR]	; Read register
	CLR	C		; No carry = No data
	ANL	A,#%00100000	; Packet available?
	JNZ	gc0b		; Yes, handle it
	SJMP	gc0d		; Return
;
; Get charcter from serial port, preserving active DPTR
;
_GETCH:	MOV	A,DPS		; Get current mode
	JB	A.0,getc1	; Use DPTR-1
; Get character from serial port preserving DPTR-0
GETCH:	PUSH	DPH		; Save DPH
	PUSH	DPL		; Save DPL
	MOV	DPTR,#$B013	; Serial interrupt source
gc0a:	MOVX	A,[DPTR]	; Read register
	CLR	C		; No carry = No data
	ANL	A,#%00100000	; Packet available?
	JZ	gc0a		; No, report no data
gc0b:	MOV	DPTR,#$B0C8	; Serial DMA 1 control
	MOV	A,#$01		; Start transfer
	MOVX	[DPTR],A	; Write it
	MOV	DPTR,#$B014	; Serial interrupt source
gc0c:	MOVX	A,[DPTR]	; Read data
	ANL	A,#%00100000	; Wait for port-1 complete
	JZ	gc0c		; Wait for it
	MOV	DPTR,#$B013	; Serial interrupt clear
	MOVX	[DPTR],A	; Clear FIFO data
	INC	DPTR		; Next
	MOVX	[DPTR],A	; Clear Packet ready
	MOV	DPTR,#RXBUF	; Point to RX buffer
	MOVX	A,[DPTR]	; Read the data
	SETB	C		; Indicate data ready
gc0d:	POP	DPL		; Restore DPL
	POP	DPH		; Restore DPH
	RET
; Test for character from serial port using DPTR-0
tstc1:	PUSH	DPH1		; Save DPH
	PUSH	DPL1		; Save DPL
	MOV	DPTR,#$B013	; Serial interrupt source
	MOVX	A,[DPTR]	; Read register
	CLR	C		; No carry = No data
	ANL	A,#%00100000	; Packet available?
	JNZ	gc1b		; Yes, handle it
	SJMP	gc1d		; Return
; Get character from serial port preserving DPTR-1
GETC1:	PUSH	DPH1		; Save DPH
	PUSH	DPL1		; Save DPL
	MOV	DPTR,#$B013	; Serial interrupt source
gc1a:	MOVX	A,[DPTR]	; Read register
	CLR	C		; No carry = No data
	ANL	A,#%00100000	; Packet available?
	JZ	gc1a		; No, report no data
gc1b:	MOV	DPTR,#$B0C8	; Serial DMA 1 control
	MOV	A,#$01		; Start transfer
	MOVX	[DPTR],A	; Write it
	MOV	DPTR,#$B014	; Serial interrupt source
gc1c:	MOVX	A,[DPTR]	; Read data
	ANL	A,#%00100000	; Wait for port-1 complete
	JZ	gc1c		; Wait for it
	MOV	DPTR,#$B013	; Serial interrupt clear
	MOVX	[DPTR],A	; Clear FIFO data
	INC	DPTR		; Next
	MOVX	[DPTR],A	; Clear Packet ready
	MOV	DPTR,#RXBUF	; Point to RX buffer
	MOVX	A,[DPTR]	; Read the data
	SETB	C		; Indicate data ready
gc1d:	POP	DPL1		; Restore DPL
	POP	DPH1		; Restore DPH
	RET
;
; Install RAM subroutine to WRITE
;
CPWR:	MOV	DPS,#0		; Select DPTR-0
	MOV	DPTR,#WXF	; Point to function
; Install RAM subroutine
CPSUB:	MOV	R0,#WXFS	; Set size
	MOV	DPH1,#=RAMTFR	; Set dest high
	MOV	DPL1,#RAMTFR	; Set dest low
cpsub1:	CLR	A		; Get zero
	MOVC	A,[A+DPTR]	; get data from flash
	INC	DPTR		; Advance
	INC	DPS		; DPTR1
	MOVX	[DPTR],A	; Write to external
	INC	DPTR		; Advance
	INC	DPS		; DPTR0
	DJNZ	R0,cpsub1	; Copy them all
	RET
;
; RAM subroutine to write XFLASH - Value passed in R0
; NOTE: Must be 6 bytes in size
;
WXF:	NOP			; Common code for prefetch
	MOV	A,R0		; Read result
	MOVX	[DPTR],A	; Write to xflash
	MOV	A,R2		; Get return value
	MOVX	[R1],A		; Select mode
	RET
WXFS:	EQU	*-WXF		; Size of function
;
; Prepare to write prgoram memory
;  - Get DPTR value
;  - Set MPAGE, R1 and R2
;  - Adjust DPH to C000-FFFF & set R3 to bank
;
PREPWP:	MOV	MPAGE,#$B0	; Select MPAGE
	MOV	R1,#$09		; Mode select register
	MOV	R2,#$01		; Return mode
	MOV	A,DPH		; Get DPH value
	ANL	A,#%11000000	; Get block
	RR	A		; Block into bits 6:5
	ORL	A,#%00000011	; PGM/External
	MOV	R3,A		; Save for later
	MOV	A,DPH		; Get high again
	ANL	A,#%00111111	; Save 16k block
	ADD	A,#$C0		; Offset to flash access block
	MOV	DPH,A		; Resave
	RET
;
; Write to external memory
; - Advance to next location
; - And adjust DPTR/R3 for value
;
WXP:	MOV	R0,A		; Save char for later
	MOV	A,R3		; Get mode
	ACALL	ENTER		; Perform write
	INC	DPTR		; Advance to next
	MOV	A,DPH		; Get high value
	JNZ	wxp1		; Not wrapped
	MOV	DPH,#$C0	; Wrap
	MOV	A,R3		; Get mode
	ADD	A,#%00100000	; Increment block
	MOV	R3,A		; Resave
wxp1:	RET
;
; Light debug lamp
;
xled:	PUSH	A		; Save A
	MOV	DPTR,#$B083	; Direction register
	MOV	A,#$0F		; Make  outputs
	MOVX	[DPTR],A	; Set outputs
	MOV	DPTR,#$B087	; Data register
	POP	A		; Get value
	MOVX	[DPTR],A
	RET
